<?php
/**
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2020 vi-solutions
 */
// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Path;
use Joomla\Event\DispatcherInterface;

require_once JPATH_ADMINISTRATOR . '/components/com_visforms/helpers/visforms.php';
require_once JPATH_ROOT . '/libraries/visolutions/tcpdf/tcpdf.php';
require_once JPATH_ADMINISTRATOR . '/components/com_visforms/helpers/pdf/visTCPDF.php';

class visFormsPluginHelper extends CMSPlugin {

	protected $filesToDelete = array();
    // load plugin language files
    protected $autoloadLanguage = true;
    // set to true, if plugin has own datatable
    // if set to false, do not process the respective handler if set to false
    protected $hasOwnTableInDatabase = false;

    public function __construct(DispatcherInterface $subject, array $config) {
        parent::__construct($subject, $config);
        // load com_visforms language files (active language and default en-GB)
        $this->loadLanguage('com_visforms', JPATH_SITE);
        if ($this->hasOwnTableInDatabase && !empty($this->params['wasDisabled'])) {
            // plugin is newly enabled
            // clean it's data table from form records of forms, which were deleted, while the plugin was disables
            $this->removeDeletedFormRecords();
            $this->removeDisabledFlag();
        }
    }

	// Backend actions manage plugin configuration settings in database

	// add plugin specific fields to form configuration in administration
	public function onVisformsPrepareJForm($form) {
        if (!$this->hasOwnTableInDatabase) {
            return true;
        }
		$app = Factory::getApplication();
		if ($app->isClient('administrator')) {
			JForm::addFormPath($this->formDir);
			$form->loadFile($this->name, false);
			$form = $this->removeFields($form);
			$data = $this->loadFormData($form);
			$form->bind($data);
		}
		return true;
	}

	// save values of plugin specific fields from form configuartion
	public function onVisformsSaveJFormExtraData($data, $fid, $isNew) {
        if (!$this->hasOwnTableInDatabase) {
            return true;
        }
		$app = Factory::getApplication();
		if ($app->isClient('administrator')) {
			$data['fid'] = $fid;
			if (isset($data[$this->name . '_params']) && is_array($data[$this->name . '_params'])) {
				$data[$this->name . '_params'] = VisformsHelper::registryStringFromArray($data[$this->name . '_params']);
			}
			$this->saveExtraData($data, $isNew);
		}
		return true;
	}

	// delete values of plugin specific fields from form configuration in administration
	public function onVisformsAfterJFormDelete($context, $table) {
        if (!$this->hasOwnTableInDatabase) {
            return true;
        }
		// Skip plugin if context is wrong
		if ($context != 'com_visforms.visform') {
			return true;
		}
		$app = Factory::getApplication();
		if ($app->isClient('administrator')) {
			$db = Factory::getDbo();
			$query = $db->getQuery(true);
			$query->delete($db->quoteName('#__' . $this->name))
				->where($db->quoteName('fid') . ' = ' . $table->id);
			$db->setQuery($query);
			$result = $db->execute();
		}
		return true;
	}

	// handle plugin specific options on batch copy of form
	public function onVisformsAfterBatchCopyForm($oldFormId, $newFormId) {
        if (!$this->hasOwnTableInDatabase) {
            return true;
        }
		$app = Factory::getApplication();
		if ($app->isClient('administrator')) {
			$data = $this->getItem($oldFormId, false);
			if (empty($data)) {
				return true;
			}
			$data['fid'] = $newFormId;
			return $this->saveExtraData($data, true);
		}
		return true;
	}

	protected function loadFormData($form) {
        if (!$this->hasOwnTableInDatabase) {
            return true;
        }
		$data = $this->getItem();
		if (empty($data)) {
			$data = array();
		}
		return $data;
	}

	protected function getItem($fid = null, $extract = true) {
        if (!$this->hasOwnTableInDatabase) {
            return false;
        }
		if (empty($fid)) {
			$fid = $this->fid;
		}
		$db = Factory::getDbo();
		$query = $db->getQuery(true);
		$query->select('*')
			->from($db->quoteName('#__' . $this->name))
			->where($db->quoteName('fid') . ' = ' . $fid);
		try {
			$db->setQuery($query);
			$data = $db->loadAssoc();
		}
		catch (RuntimeException $e) {
			return false;
		}
		if ($data && $extract && isset($data[$this->name . '_params'])) {
			$data[$this->name . '_params'] = VisformsHelper::registryArrayFromString($data[$this->name . '_params']);
		}
		return $data;
	}


	protected function saveExtraData($data, $isNew) {
        if (!$this->hasOwnTableInDatabase) {
            return true;
        }
		Table::addIncludePath($this->tableDir);
		$table = Table::getInstance($this->plgTableName, 'Table', $config = array());
		$tableKey = $this->tableKey;
		$app = Factory::getApplication();
		if (!($isNew)) {
			if (!isset($data[$tableKey])) {
				return false;
			}
			$data[$tableKey] = (int) $data[$tableKey];
			$table->load($data[$tableKey]);
		} else {
			unset($data[$tableKey]);
		}
		if (!$table->bind($data)) {
			$app->enqueueMessage($table->getError(), 'error');
			return false;
		}

		if (!$table->check()) {
			$app->enqueueMessage($table->getError(), 'error');
			return false;
		}

		if (!$table->store()) {
			$app->enqueueMessage($table->getError(), 'error');
			return false;
		}
		return true;
	}

	protected function createHash() {
		if (function_exists('openssl_random_pseudo_bytes')) {
			$rand = openssl_random_pseudo_bytes(16);
			if ($rand === false) {
				// Broken or old system
				$rand = mt_rand();
			}
		}
		else {
			$rand = mt_rand();
		}
		$hashThis = microtime() . $rand;
		if (function_exists('hash')) {
			$hash = hash('sha256', $hashThis);
		}
		else if (function_exists('sha1')) {
			$hash = sha1($hashThis);
		}
		else {
			$hash = md5($hashThis);
		}
		return $hash;
	}

	protected function storeFile($name, $content) {
		$ret = File::write($this->outputPath . $name, $content);

		if ($ret) {
			// return the name of the file
			return $name;
		}
		else {
			return false;
		}
	}

	protected function deleteFiles() {
		// Delete the PDF data to disk using File::delete();
		$filesToDelete = $this->filesToDelete;
		foreach ($filesToDelete as $fileToDelete) {
			$path = $this->outputPath;
			$file = Path::clean($path . $fileToDelete);
			if (File::exists($file)) {
				File::delete($file);
			}
		}
	}

	protected function createPDF($html, $pdfHideHeader = 1, $pdfHideFooter = 1) {
		// Create the PDF
		$pdf = $this->initializeTCPDF();
		if (!(empty($pdfHideHeader))) {
			$pdf->SetPrintHeader(false);
		}
		if (!(empty($pdfHideFooter))) {
			$pdf->SetPrintFooter(false);
		}
		$pdf->AddPage();
		$pdf->writeHTML($html, true, false, true, false, '');
		$pdf->lastPage();
		$pdfData = $pdf->Output('', 'S');

		unset($pdf);
		// create file name
		$hash = $this->getFileName();
		$name = $hash . '.pdf';
		// write file
		return $this->storeFile($name, $pdfData);
	}
	
	protected function getFileName () {
		return $this->createHash();
	}

	protected function initializeTCPDF() {
		require_once JPATH_ADMINISTRATOR . '/components/com_visforms/helpers/pdf/visTCPDF.php';
		$pdf = new visTCPDF('P', VISFORMS_PDF_UNIT, 'A4', true, 'UTF-8', false);

		// set document information
		$pdf->SetCreator(VISFORMS_PDF_CREATOR);
		$pdf->SetAuthor(VISFORMS_PDF_AUTHOR);
		$pdf->SetTitle('Visforms');
		$pdf->SetSubject('Visforms');
		$pdf->setShowHeader(0);
		$pdf->setShowFooter(0);

		// check and set font areas
		$pdf->setHeaderFont(array('dejavusans', '', 8, '', false));
		$pdf->setFooterFont(array('dejavusans', '', 8, '', false));
		$pdf->SetFont('dejavusans', '', 10, '', false);

		// set default monospaced font
		$pdf->SetDefaultMonospacedFont(VISFORMS_PDF_FONT_MONOSPACED);

		// set margins
		$pdf->SetMargins(15, 27,15);
		$pdf->SetHeaderMargin(5);
		$pdf->SetFooterMargin(10);

		// set image scale factor
		$pdf->setImageScale(VISFORMS_PDF_IMAGE_SCALE_RATIO);

		// set some language-dependent strings (optional)
		if (@file_exists(dirname(__FILE__).'/lang/eng.php')) {
			global $l;
			require_once(dirname(__FILE__).'/lang/eng.php');
			$pdf->setLanguageArray($l);
		}

		// set default font sub setting mode
		$pdf->setFontSubsetting(true);

		return $pdf;
	}
	
	protected function removeFields($form) {
		return $form;
	}

	// make sure data stored in data table of the plugin match existing forms
	protected function removeDeletedFormRecords() {
        // get a list of all existing forms
        $db = Factory::getDbo();
        $query = $db->getQuery(true);
        $query->select($db->qn('id'))
            ->from($db->qn('#__visforms'));
        try {
            $db->setQuery($query);
            $formIds = $db->loadColumn();
        }
        catch (\RuntimeException $e) {
            return false;
        }
        // get list of form records stored in plugins own data table
        $query = $db->getQuery(true);
        $query->select($db->qn('fid'))
            ->from($db->qn('#__'. $this->name));
        try {
            $db->setQuery($query);
            $pluginFormIds = $db->loadColumn();
        }
        catch (\RuntimeException $e) {
            return false;
        }
        if (empty($pluginFormIds)) {
            return true;
        }
        if (empty($formIds)) {
            $formIds = array();
        }
        // remove recordsets of forms which no longer exist
        foreach ($pluginFormIds as $pluginFormId) {
            if (!in_array($pluginFormId, $formIds)) {
                $query = $db->getQuery(true);
                $query->delete($db->quoteName('#__' . $this->name))
                    ->where($db->quoteName('fid') . " = " . $pluginFormId);
                try {
                    $db->setQuery($query);
                    $db->execute();
                }
                catch (RuntimeException $e) {
                    $error = $e->getMessage();
                }
            }
        }
        return true;
    }

    // remove the "wasDisabled" flag from plugin params
    private function removeDisabledFlag () {
        $fieldName = 'params';
        $db = Factory::getDbo();
        $query = $db->getQuery(true);
        $query
            ->select($db->quoteName($fieldName))
            ->from($db->quoteName('#__extensions'))
            ->where($db->qn('element') . ' = ' . $db->q($this->_name) . ' AND ' . $db->qn('type') . ' = ' . $db->q('plugin') . ' AND ' . $db->qn('folder') . ' = ' . $db->q($this->_type));
        try {
            $db->setQuery($query);
            $result = $db->loadResult();
            if (empty($result)) {
                $params = array();
            }
            else {
                $params = json_decode($result, true);
            }
            $params['wasDisabled'] = false;
        }
        catch (RuntimeException $e) {
            return false;
        }
        $paramsString = json_encode($params);
        try {
            $db->setQuery('UPDATE #__extensions SET ' . $fieldName . ' = ' .  $db->quote($paramsString) . ' WHERE element=' . $db->q($this->_name) . ' and type = ' . $db->q('plugin') .' and folder = ' . $db->q($this->_type));
            $db->execute();

        }
        catch (RuntimeException $e) {
            return false;
        }
        return true;
    }
}