<?php
/**
 * @author       Aicha Vack
 * @package      Joomla.Administrator
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2021 vi-solutions
 * @since        Joomla 1.6
 */
namespace Visolutions\Plugin\Visforms\Visforms\Extension;

// No direct access
defined('_JEXEC') or die;

use Joomla\CMS\MVC\Factory\MVCFactoryAwareTrait;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Factory;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Exception\FilesystemException;
use Joomla\Filesystem\Path;
use Joomla\Component\Menus\Administrator\Helper\MenusHelper;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseInterface;
use Joomla\Event\SubscriberInterface;
use Visolutions\Component\Visforms\Administrator\Model\VisformModel;
use Joomla\CMS\Event\Model;
use Joomla\CMS\Event\Finder as FinderEvent;
use Joomla\CMS\Event\Menu\PreprocessMenuItemsEvent;
use Visolutions\Component\Visforms\Site\Event\Visforms\VisformsAfterEditFormSaveErrorEvent;
use Visolutions\Component\Visforms\Site\Event\Visforms\VisformsAfterFormSaveErrorEvent;
use Visolutions\Component\Visforms\Site\Event\Visforms\VisformsAfterFormSaveEvent;
use Visolutions\Component\Visforms\Site\Event\Visforms\VisformsAfterEditFormSaveEvent;
use Visolutions\Component\Visforms\Administrator\Table\VisfieldTable;
use Visolutions\Component\Visforms\Administrator\Helper\AefHelper;
use Visolutions\Component\Visforms\Administrator\Helper\VisformsHelper;
use Visolutions\Component\Visforms\Administrator\Helper\ConditionsHelper;

class Visforms extends CMSPlugin implements SubscriberInterface
{
    use MVCFactoryAwareTrait;
    use DatabaseAwareTrait;
    protected $autoloadLanguage = true;

    public static function getSubscribedEvents(): array {
        return [
            'onPreprocessMenuItems' => 'onPreprocessMenuItems',
            'onVisformsdataAfterJFormSave' => 'onVisformsdataAfterJFormSave',
            'onVisformsAfterJFormSave' => 'onVisformsAfterJFormSave',
            'onVisformsdataJFormChangeState' => 'onVisformsdataJFormChangeState',
            'onVisformsJFormChangeState' => 'onVisformsJFormChangeState',
            'onVisformsdataAfterJFormDelete' => 'onVisformsdataAfterJFormDelete',
            'onVisformsAfterJFormDelete' => 'onVisformsAfterJFormDelete',
            'onVisformsBeforeJFormDelete' => 'onVisformsBeforeJFormDelete',
            'onVisformsAfterFormSaveError' => 'onVisformsAfterFormSaveError',
            'onVisformsAfterFormSave' => 'onVisformsAfterFormSave',
            'onVisformsAfterEditFormSave' => 'onVisformsAfterEditFormSave',
            'onVisformsAfterEditFormSaveError' => 'onVisformsAfterEditFormSaveError',
        ];
    }

    public function onPreprocessMenuItems(PreprocessMenuItemsEvent $event): void {
        $context = $event->getContext();
        $children = $event->getItems();
        // Visforms Submenu in Administration
        // set link to '' if a menu item must be removed dynamically or set menu parameter 'menu_show' to 0
        $app = $this->getApplication();
        if ($app->isClient('site')) {
            return;
        }
        if ($context != 'com_menus.administrator.module') {
            return;
        }

        $input = $app->getInput();
        // should actually always be the case, because plugin is only loaded if we are  in Visforms context
        $isVisformsMenu = ($input->get('option', '', 'cmd') == 'com_visforms');
        $fid = $input->get('fid', 0, 'int');
        if ($isVisformsMenu) {
            // get possible Submenu entries form administrator/components/com_visforms/presets/visforms.xml
            $submenues = MenusHelper::loadPreset('visforms');
            foreach ($children as $i => $child) {
                if ($child->title == 'com_visforms') {
                    // only add PDF Templates link, if we have subscriptions
                    $hasSub = AefHelper::checkAEF();
                    foreach ($submenues->getChildren() as $c) {
                        if (empty($hasSub) && $c->title == 'COM_VISFORMS_PDF_TEMPLATES') {
                            continue;
                        }
                        // if we have a selected form id, add this id to the submenu link
                        if (0 < $fid) {
                            // unless it is the data view menu item and saveresult is not enabled for the form
                            if ($c->title == 'COM_VISFORMS_VISFORMSDATA_VIEW_DEFAULT_TITLE') {
                                $formModel = new VisformModel();
                                $form = $formModel->getItem($fid);
                                if (!empty($form->saveresult)) {
                                    $c->link .= $fid;
                                }
                            }
                            else {
                                $c->link .= $fid;
                            }
                        }
                        // Subemnu to visforms menu
                        $children[$i]->addChild($c);
                    }
                }
            }
        }
    }

    // Re-indexing of Finder Index
    public function onVisformsAfterFormSave(VisformsAfterFormSaveEvent $event): void {
        $context = $event->getContext();
        $form = $event->getForm();
        $fields = $event->getFields();
        $hasDataRecord = (!empty($form->saveresult) && isset($form->dataRecordId));
        // delete temporary uploaded files
        // delete value in databaes
        if (!empty($form->tmp_upload)) {
            $this->removeUploadedFiles($context, $form, $fields, $hasDataRecord);
        }
        // Re-indexing of Finder Index
        // records are not stored in the database
        if (!$hasDataRecord) {
            return;
        }
        if ($context === 'com_visforms.form') {
            $row = new \stdClass();
            $row->id = $form->id . ':' . $form->dataRecordId;
            PluginHelper::importPlugin('finder');
            $this->getDispatcher()->dispatch('onFinderAfterSave', new FinderEvent\AfterSaveEvent('onFinderAfterSave', [
                'context' => $context,
                'subject' => $row,
                'isNew'   => true,
            ]));
        }
    }

    public function onVisformsAfterEditFormSave(VisformsAfterEditFormSaveEvent $event): void {
        $context = $event->getContext();
        $form = $event->getForm();
        $fields = $event->getFields();
        $hasDataRecord = (!empty($form->saveresult) && isset($form->dataRecordId));
        // delete temporary uploaded files
        // delete value in databaes
        if (!empty($form->tmp_upload)) {
            $this->removeUploadedFiles($context, $form, $fields, $hasDataRecord);
        }
        // Re-indexing of Finder Index
        // records are not stored in the database
        if (!$hasDataRecord) {
            return;
        }
        if ($context === 'com_visforms.form') {
            $row = new \stdClass();
            $row->id = $form->id . ':' . $form->dataRecordId;
            PluginHelper::importPlugin('finder');
            $this->getDispatcher()->dispatch('onFinderAfterSave', new FinderEvent\AfterSaveEvent('onFinderAfterSave', [
                'context' => $context,
                'subject' => $row,
                'isNew'   => false,
            ]));
        }
    }

    public function onVisformsdataAfterJFormSave(Model\AfterSaveEvent $event): void {
        $context = $event->getContext();
        $table = $event->getItem();
        $isNew = $event->getIsNew();
        $app = $this->getApplication();
        // event is triggered from administration and from site
        // we only want in from administration, because the site is handled by onVisformsAfterEditFormSave
        if ($app->isClient('site')) {
            return;
        }
        if ($context === 'com_visforms.visdata') {

            $fid = $app->getInput()->get('fid', 0, 'int');
            $row = new \stdClass();
            $row->id = $fid . ':' . $table->id;
            PluginHelper::importPlugin('finder');
            $this->getDispatcher()->dispatch('onFinderAfterSave', new FinderEvent\AfterSaveEvent('onFinderAfterSave', [
                'context' => $context,
                'subject' => $row,
                'isNew'   => $isNew,
            ]));
        }
    }

    public function onVisformsAfterJFormSave(Model\AfterSaveEvent $event): void {
        $context = $event->getContext();
        $table = $event->getItem();
        $isNew = $event->getIsNew();
        if (($context === 'com_visforms.visfield') || ($context === 'com_visforms.visform')) {
            PluginHelper::importPlugin('finder');
            $this->getDispatcher()->dispatch('onFinderAfterSave', new FinderEvent\AfterSaveEvent('onFinderAfterSave', [
                'context' => $context,
                'subject' => $table,
                'isNew'   => $isNew,
            ]));
        }
    }

    public function onVisformsdataJFormChangeState(Model\AfterChangeStateEvent $event): void {
        $context = $event->getContext();
        $pks = $event->getPks();
        $value = $event->getValue();
        if ($context === 'com_visforms.visdata') {

            $fid = $this->getApplication()->getInput()->get('fid', 0, 'int');
            foreach ($pks as $i => $pk) {
                $pks[$i] = $fid . ':' . $pk;
            }
            PluginHelper::importPlugin('finder');
            $this->getDispatcher()->dispatch('onFinderChangeState', new FinderEvent\AfterChangeStateEvent('onFinderChangeState', [
                'context' => $context,
                'subject' => $pks,
                'value'   => $value,
            ]));
        }
    }

    public function onVisformsJFormChangeState(Model\AfterChangeStateEvent $event): void {
        $context = $event->getContext();
        $value = $event->getValue();
        if ($context === 'com_visforms.visform') {
            $pks = $event->getPks();
            // $pks is the selected forms ids
            PluginHelper::importPlugin('finder');
            $this->getDispatcher()->dispatch('onFinderChangeState', new FinderEvent\AfterChangeStateEvent('onFinderChangeState', [
                'context' => $context,
                'subject' => $pks,
                'value'   => $value,
            ]));
        }
        else if ($context === 'com_visforms.visfield') {
            // $pks is the selected fields ids, we need to reindex on the basis of the form
            $fid = $this->getApplication()->getInput()->get('fid', 0, 'int');
            $pks = array($fid);
            PluginHelper::importPlugin('finder');
            $this->getDispatcher()->dispatch('onFinderChangeState', new FinderEvent\AfterChangeStateEvent('onFinderChangeState', [
                'context' => $context,
                'subject' => $pks,
                'value'   => $value,
            ]));
        }
    }

    public function onVisformsdataAfterJFormDelete(Model\AfterDeleteEvent $event): void {
        $context = $event->getContext();
        $table = $event->getItem();
        if ($context === 'com_visforms.visdata') {
            PluginHelper::importPlugin('finder');
            $fid = $this->getApplication()->getInput()->get('fid', 0, 'int');
            // we need another value in table key field for finder plugin
            $tmp = $fid . ':' . $table->id;
            // clone table object for use in finder plugin
            $finderPreparedTable = clone $table;
            $finderPreparedTable->id = $tmp;
            $this->getDispatcher()->dispatch('onFinderAfterDelete', new FinderEvent\AfterDeleteEvent('onFinderAfterDelete', [
                'context' => $context,
                'subject' => $finderPreparedTable,
            ]));
        }
    }

    public function onVisformsAfterJFormDelete(Model\AfterDeleteEvent $event): void {
        $context = $event->getContext();
        $table = $event->getItem();
        if (($context === 'com_visforms.visform') || ($context === 'com_visforms.visfield')) {
            PluginHelper::importPlugin('finder');
            $this->getDispatcher()->dispatch('onFinderAfterDelete', new FinderEvent\AfterDeleteEvent('onFinderAfterDelete', [
                'context' => $context,
                'subject' => $table,
            ]));
        }
    }

    // End re-indixing finder index

    public function onVisformsBeforeJFormDelete(Model\BeforeDeleteEvent $event): void {
        $context = $event->getContext();
        $data = $event->getItem();
        // Skip plugin if we are deleting something other than a visforms form or field
        if (($context != 'com_visforms.visfield') && ($context != 'com_visforms.visform')) {
            return;
        }
        if ($context == 'com_visforms.visfield') {
            $success = true;
            $fid = $data->fid;
            $id = $data->id;

            // Convert the defaultvalues field to an array.
            $defaultvalues = VisformsHelper::registryArrayFromString($data->defaultvalue);

            // Remove restrtictions
            // getRestricts
            if ($restricts = ConditionsHelper::setRestricts($data->id, $defaultvalues, $data->name, $fid)) {
                //remove Restrictions
                try {
                    ConditionsHelper::removeRestriction($restricts);
                }
                catch (\RuntimeException $e) {
                    Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
                    $event->addResult(false);
                    return;
                }
            }

            $db = $this->getDatabase();
            $tablesAllowed = $db->getTableList();
            if (!empty($tablesAllowed)) {
                $tablesAllowed = array_map('strtolower', $tablesAllowed);
            }
            $tablesToDeleteFrom = array("visforms_" . $fid, "visforms_" . $fid . "_save");
            foreach ($tablesToDeleteFrom as $tn) {
                $tnfull = strtolower($db->getPrefix() . $tn);

                //Delete field in data table when deleting a field
                if (in_array($tnfull, $tablesAllowed)) {

                    $tableFields = $db->getTableColumns('#__' . $tn, false);
                    $fieldname = "F" . $id;

                    if (isset($tableFields[$fieldname])) {

                        $query = "ALTER TABLE #__" . $tn . " DROP " . $fieldname;
                        try {
                            $db->setQuery($query);
                            $db->execute();
                        }
                        catch (\RuntimeException $e) {
                            Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
                            $success = false;
                            continue;
                        }
                    }
                }
            }

            if (!$success) {
                // set already deleted restrictions again
                ConditionsHelper::setRestriction($restricts);
                $event->addResult($success);
            }
            return;
        }

        // Delete fields in visfields table when deleting a form, delete datatable if table exists and delete pdfs in vispdf table
        // Trigger Visforms plugins which may store field specific extra configuration data, so that these data can be deleted properly
        if ($context == 'com_visforms.visform') {
            $success = true;
            $fid = $data->id;
            // get a list of all field id's for the form
            $db = $this->getDatabase();
            $query = $db->createQuery();
            $query->select($db->qn('id'))
                ->from($db->qn('#__visfields'))
                ->where($db->quoteName('fid') . " = " . $fid);
            try {
                $db->setQuery($query);
                $fieldIds = $db->loadColumn();

            }
            catch (\RuntimeException $e) {
                $success = false;
            }
            // Delete field using Joomla Table Class and Joomla Table Events
            // we need the DatabaseInterface class ($dbo), not the database ($db)
            $dbo =  Factory::getContainer()->get(DatabaseInterface::class);
            $table = new VisfieldTable($dbo);
            if (!empty($fieldIds)) {
                foreach ($fieldIds as $fieldId) {
                    try {
                        $table->delete($fieldId);
                    }
                    catch (\RuntimeException $e) {
                        Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
                        $success = false;
                    }
                }
            }
            $query = $db->createQuery();
            $query->delete($db->quoteName('#__vispdf'))
                ->where($db->quoteName('fid') . " = " . $fid);
            try {
                $db->setQuery($query);
                $db->execute();
            }
            catch (\RuntimeException $e) {
                Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
                $success = false;
            }
            $tablesToDelete = array("visforms_" . $fid, "visforms_" . $fid . "_save");
            foreach ($tablesToDelete as $tn) {
                try {
                    $db->setQuery("drop table if exists #__" . $tn);
                    $db->execute();
                }
                catch (\RuntimeException $e) {
                    Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
                    $success = false;
                }
            }
        }
        $event->addResult($success);
    }

    public function onVisformsAfterFormSaveError(VisformsAfterFormSaveErrorEvent $event): void {
        $context = $event->getContext();
        $form = $event->getForm();
        $fields = $event->getFields();
        $this->removeUploadedFiles($context, $form, $fields);
    }

    public function onVisformsAfterEditFormSaveError(VisformsAfterEditFormSaveErrorEvent $event): void {
        $context = $event->getContext();
        $form = $event->getForm();
        $fields = $event->getFields();
        $this->removeUploadedFiles($context, $form, $fields);
    }

    protected function removeUploadedFiles($context, $form, $fields, $deleteDbValue = false): bool {
        if ($context != 'com_visforms.form') {
            return true;
        }
        if (Factory::getApplication()->isClient('administrator')) {
            return true;
        }
        foreach ($fields as $field) {
            if ($field->typefield !== "file") {
                continue;
            }
            if (empty ($field->file) || !is_array($field->file)) {
                continue;
            }
            if (!empty($field->file['filepath'])) {
                $file = Path::clean($field->file['filepath']);
                if (file_exists($file)) {
                    try {
                        File::delete($file);
                    }
                    catch (FilesystemException $e) {

                    }
                }
                if (!empty($deleteDbValue)) {
                    // update record in database
                    $this->deleteFileInfo($form, $field);
                }
            }
        }
        return true;
    }

    private function deleteFileInfo($form, $field) {
        $db = $this->getDatabase();
        try {
            $query = $db->createQuery();
            $query->update($db->quoteName('#__visforms_' . $form->id))
                ->set($db->quoteName('F'. $field->id) . ' = ' . $db->quote(""))
                ->where(array($db->quoteName('id') . ' = ' . $form->dataRecordId));
            $db->setQuery($query);
            $db->execute();
        }
        catch (\RuntimeException $e) {
        }
    }
}