<?php
/**
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2020 vi-solutions
 */
namespace Visolutions\Component\Visforms\Administrator\Helper;
// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Filesystem\Exception\FilesystemException;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Path;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Form\Form;
use Joomla\Event\SubscriberInterface;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisformsPrepareJFormEvent;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisformsSaveJFormExtraDataEvent;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisformsAfterBatchCopyFormEvent;
use Joomla\CMS\Event\Model;
use Joomla\Database\DatabaseInterface;

class FormPluginHelper extends CMSPlugin implements SubscriberInterface {

    // This helper can be used to create a custom plugin that adds a custom tab with custom configuration options to form configuration
    // In addition this helper provides code that allows to create temporary pdf and csv files, which can be used in a custom plugin

    // must be set in subclass, if custom plugin adds tab to form configuration and stores configuration settings in own data table
    // directory of xml definition file for the additional fields (configuration option) for the form configuration
    protected $formDir;
    // directory of table class file for the additional fields (configuration option) for the form configuration
    // @deprecated 5.4.0: will be removed in Visforms 6.0
    protected $tableDir;
    // table class name for the additional fields (configuration option) for the form configuration
    // @deprecated 5.4.0: will be removed in Visforms 6.0
    protected $plgTableName;
    // name of index field in table
    protected $tableKey;

    // set $hasOwnTableInDatabase to true, if custom plugin which extends this helper, has own datatable in which the custom configuration settings are stored
    // if set to true: make sure, that the custom plugin sets the following class members $formDir, $tableDir, $plgTableName, $tableKey, then!
    // the name of the plugin data table must be '#__'. $this->name
    // plugin data table/form xml definition must have the following fields: fid, $tableKey, $this->name.'_params'.
    // fid is reference to the id of the form which is extended.
    // make sure fid is set properly for the administration parts and the frontend parts"
    // $this->name.'_params' is a fields node in form xml definition
    // examples are all visforms plugins which create own tabs in form configuration (like vfmaxsubmissions)

    // if set to false or required members are not set properly in subclasses, default implementation does not process the respective handler (like onVisformsPrepareJForm)
    protected $hasOwnTableInDatabase = false;

    // Additional plugin functions: create (temporary) pdf and csv files
    protected $outputPath = JPATH_ADMINISTRATOR . '/components/com_visforms/pdfs/';
    protected $filesToDelete = array();
    // load plugin language files
    protected $autoloadLanguage = true;
    // id of form
    protected $fid;
	// logger name to get set by derived class
	protected $loggerName;
	// plugin table class object
	protected $plgTable;

    public function __construct(DispatcherInterface $subject, array $config) {
        parent::__construct($subject, $config);
        $this->setApplication(Factory::getApplication());
        // load com_visforms language files (active language and default en-GB)
        $this->loadLanguage('com_visforms', JPATH_SITE);
        $this->setPlgTable();
        if ($this->hasOwnTableInDatabase && !empty($this->params['wasDisabled'])) {
            // plugin is newly enabled
            // clean it's data table from form records of forms, which were deleted, while the plugin was disables
            $this->removeDeletedFormRecords();
            $this->removeDisabledFlag();
        }
        $this->fid = $this->getApplication()->getInput()->getInt('id', 0);
    }

    public static function getSubscribedEvents(): array {
        return [
            'onVisformsPrepareJForm' => 'onVisformsPrepareJForm',
            'onVisformsSaveJFormExtraData' => 'onVisformsSaveJFormExtraData',
            'onVisformsAfterJFormDelete' => 'onVisformsAfterJFormDelete',
            'onVisformsAfterBatchCopyForm' => 'onVisformsAfterBatchCopyForm',
        ];
    }

	// Backend actions manage plugin configuration settings in database

	// add plugin specific fields to form configuration in administration
	public function onVisformsPrepareJForm(VisformsPrepareJFormEvent $event): void {
        $form = $event->getForm();
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
		if ($this->getApplication()->isClient('administrator')) {
			Form::addFormPath($this->formDir);
			$form->loadFile($this->name, false);
			$form = $this->removeFields($form);
			$data = $this->loadFormData($form);
			$form->bind($data);
		}
	}

	// save values of plugin specific fields from form configuartion
	public function onVisformsSaveJFormExtraData(VisformsSaveJFormExtraDataEvent $event) : void {
        $data = $event->getData();
        $fid = $event->getFid();
        $isNew = $event->getIsNew();
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
		if ($this->getApplication()->isClient('administrator')) {
			$data['fid'] = $fid;
			if (isset($data[$this->name . '_params']) && is_array($data[$this->name . '_params'])) {
				$data[$this->name . '_params'] = VisformsHelper::registryStringFromArray($data[$this->name . '_params']);
			}
			$this->saveExtraData($data, $isNew);
		}
	}

	// delete values of plugin specific fields from form configuration in administration
    public function onVisformsAfterJFormDelete(Model\AfterDeleteEvent $event) : void {
        $context = $event->getContext();
        $table = $event->getItem();
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
		// Skip plugin if context is wrong
		if ($context != 'com_visforms.visform') {
			return;
		}
		if ($this->getApplication()->isClient('administrator')) {
			$db = Factory::getContainer()->get(DatabaseInterface::class);
			$query = $db->createQuery();
			$query->delete($db->quoteName('#__' . $this->name))
				->where($db->quoteName('fid') . ' = ' . $table->id);
			try {
                $db->setQuery($query);
                $db->execute();
            }
            catch (\RuntimeException $e) {}
		}
	}

	// handle plugin specific options on batch copy of form
	public function onVisformsAfterBatchCopyForm(VisformsAfterBatchCopyFormEvent $event) : void {
        $oldFormId = $event->getPk();
        $newFormId = $event->getNewId();
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
		if ($this->getApplication()->isClient('administrator')) {
			$data = $this->getItem($oldFormId, false);
			if (empty($data)) {
				return;
			}
			$data['fid'] = $newFormId;
            $this->saveExtraData($data, true);
		}
	}

	protected function loadFormData($form) {
        if (!$this->hasOwnTableInDatabase) {
            return true;
        }
		$data = $this->getItem();
		if (empty($data)) {
			$data = array();
		}
		return $data;
	}

	protected function getItem($fid = null, $extract = true) {
        if (!$this->hasOwnTableInDatabase) {
            return false;
        }
		if (empty($fid)) {
			$fid = $this->fid;
		}
		$db = Factory::getContainer()->get(DatabaseInterface::class);
		$query = $db->createQuery();
		$query->select('*')
			->from($db->quoteName('#__' . $this->name))
			->where($db->quoteName('fid') . ' = ' . $fid);
		try {
			$db->setQuery($query);
			$data = $db->loadAssoc();
		}
		catch (\RuntimeException $e) {
			return false;
		}
		if ($data && $extract && isset($data[$this->name . '_params'])) {
			$data[$this->name . '_params'] = VisformsHelper::registryArrayFromString($data[$this->name . '_params']);
		}
		return $data;
	}


	protected function saveExtraData($data, $isNew) : void {
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
        $table = $this->plgTable;
		$tableKey = $this->tableKey;
		$app = $this->getApplication();
        try {
            if (!($isNew)) {
                if (!isset($data[$tableKey])) {
                    return;
                }
                $data[$tableKey] = (int) $data[$tableKey];
                $table->load($data[$tableKey]);
            }
            else {
                unset($data[$tableKey]);
            }
            $table->bind($data);
            if (!$table->check()) {
                $app->enqueueMessage($table->getError(), 'error');
                return;
            }
            if (!$table->store()) {
                $app->enqueueMessage($table->getError(), 'error');
                return;
            }
        }
        catch (\InvalidArgumentException | \RuntimeException $e) {
            $app->enqueueMessage($e->getMessage(), 'error');
            return;
        }
	}

	protected function createHash() {
		if (function_exists('openssl_random_pseudo_bytes')) {
			$rand = openssl_random_pseudo_bytes(16);
			if ($rand === false) {
				// Broken or old system
				$rand = mt_rand();
			}
		}
		else {
			$rand = mt_rand();
		}
		$hashThis = microtime() . $rand;
		if (function_exists('hash')) {
			$hash = hash('sha256', $hashThis);
		}
		else if (function_exists('sha1')) {
			$hash = sha1($hashThis);
		}
		else {
			$hash = md5($hashThis);
		}
		return $hash;
	}

	protected function storeFile($name, $content) {
		$ret = File::write($this->outputPath . $name, $content);

		if ($ret) {
			// return the name of the file
			return $name;
		}
		else {
			return false;
		}
	}

	protected function deleteFiles() {
		// Delete the PDF data to disk using File::delete();
		$filesToDelete = $this->filesToDelete;
		foreach ($filesToDelete as $fileToDelete) {
			$path = $this->outputPath;
			$file = Path::clean($path . $fileToDelete);
			if (file_exists($file)) {
			    try {
                    File::delete($file);
                }
                catch (FilesystemException) {

                }

			}
		}
	}

	protected function createPDF($html, $pdfHideHeader = 1, $pdfHideFooter = 1) {
        require_once JPATH_LIBRARIES . '/visolutions/simple_html_dom/simple_html_dom.php';
        if (!empty($html)) {
            $html = str_get_html($html);
            // test, that we have a node with [class="AddPage"]
            $nodes = $html->find('[class="AddPage"]');
            foreach ($nodes as $node) {
                $pageBreak = '<tcpdf method="AddPage"> </tcpdf>';
                $node->innertext = $pageBreak . $node->innertext;
                $html = $html->save();
            }
        }
		// Create the PDF
		$pdf = $this->initializeTCPDF();
		if (!(empty($pdfHideHeader))) {
			$pdf->SetPrintHeader(false);
		}
		if (!(empty($pdfHideFooter))) {
			$pdf->SetPrintFooter(false);
		}
		$pdf->AddPage();
		$pdf->writeHTML($html, true, false, true, false, '');
		$pdf->lastPage();
		$pdfData = $pdf->Output('', 'S');

		unset($pdf);
		// create file name
		$hash = $this->getFileName();
		$name = $hash . '.pdf';
		// write file
		return $this->storeFile($name, $pdfData);
	}
	
	protected function getFileName () {
		return $this->createHash();
	}

	protected function initializeTCPDF() {
				$pdf = new \Visolutions\Component\Visforms\Administrator\Helper\Pdf\VisTcpdfHelper('P', VISFORMS_PDF_UNIT, 'A4', true, 'UTF-8', false);

		// set document information
		$pdf->SetCreator(VISFORMS_PDF_CREATOR);
		$pdf->SetAuthor(VISFORMS_PDF_AUTHOR);
		$pdf->SetTitle('Visforms');
		$pdf->SetSubject('Visforms');
		$pdf->setShowHeader(0);
		$pdf->setShowFooter(0);

		// check and set font areas
		$pdf->setHeaderFont(array('dejavusans', '', 8, '', false));
		$pdf->setFooterFont(array('dejavusans', '', 8, '', false));
		$pdf->SetFont('dejavusans', '', 10, '', false);

		// set default monospaced font
		$pdf->SetDefaultMonospacedFont(VISFORMS_PDF_FONT_MONOSPACED);

		// set margins
		$pdf->SetMargins(15, 27,15);
		$pdf->SetHeaderMargin(5);
		$pdf->SetFooterMargin(10);

		// set image scale factor
		$pdf->setImageScale(VISFORMS_PDF_IMAGE_SCALE_RATIO);

		// set some language-dependent strings (optional)
		if (@file_exists(dirname(__FILE__).'/lang/eng.php')) {
			global $l;
			require_once(dirname(__FILE__).'/lang/eng.php');
			$pdf->setLanguageArray($l);
		}

		// set default font sub setting mode
		$pdf->setFontSubsetting(true);

		return $pdf;
	}
	
	protected function removeFields($form) {
		return $form;
	}

	// make sure data stored in data table of the plugin match existing forms
	protected function removeDeletedFormRecords() {
        // get a list of all existing forms
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->createQuery();
        $query->select($db->qn('id'))
            ->from($db->qn('#__visforms'));
        try {
            $db->setQuery($query);
            $formIds = $db->loadColumn();
        }
        catch (\RuntimeException $e) {
            return false;
        }
        // get list of form records stored in plugins own data table
        $query = $db->createQuery();
        $query->select($db->qn('fid'))
            ->from($db->qn('#__'. $this->name));
        try {
            $db->setQuery($query);
            $pluginFormIds = $db->loadColumn();
        }
        catch (\RuntimeException $e) {
            return false;
        }
        if (empty($pluginFormIds)) {
            return true;
        }
        if (empty($formIds)) {
            $formIds = array();
        }
        // remove recordsets of forms which no longer exist
        foreach ($pluginFormIds as $pluginFormId) {
            if (!in_array($pluginFormId, $formIds)) {
                $query = $db->createQuery();
                $query->delete($db->quoteName('#__' . $this->name))
                    ->where($db->quoteName('fid') . " = " . $pluginFormId);
                try {
                    $db->setQuery($query);
                    $db->execute();
                }
                catch (\RuntimeException $e) {
                    $error = $e->getMessage();
                }
            }
        }
        return true;
    }

    // remove the "wasDisabled" flag from plugin params
    private function removeDisabledFlag () {
        $fieldName = 'params';
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->createQuery();
        $query
            ->select($db->quoteName($fieldName))
            ->from($db->quoteName('#__extensions'))
            ->where($db->qn('element') . ' = ' . $db->q($this->_name) . ' AND ' . $db->qn('type') . ' = ' . $db->q('plugin') . ' AND ' . $db->qn('folder') . ' = ' . $db->q($this->_type));
        try {
            $db->setQuery($query);
            $result = $db->loadResult();
            if (empty($result)) {
                $params = array();
            }
            else {
                $params = json_decode($result, true);
            }
            $params['wasDisabled'] = false;
        }
        catch (\RuntimeException $e) {
            return false;
        }
        $paramsString = json_encode($params);
        try {
            $db->setQuery('UPDATE #__extensions SET ' . $fieldName . ' = ' .  $db->quote($paramsString) . ' WHERE element=' . $db->q($this->_name) . ' and type = ' . $db->q('plugin') .' and folder = ' . $db->q($this->_type));
            $db->execute();

        }
        catch (\RuntimeException $e) {
            return false;
        }
        return true;
    }

    // custom implementation in child classes must set the table if plugins has own table
    protected function setPlgTable() {
        $this->setLegacyTable();
        // ToDo: Visforms 6
        // $this->hasOwnTableInDatabase = false;
        // $this->plgTable = null;
    }

    /* @deprecated 5.4 will be removed in Visforms 6.0
     * Table class must be moved to src/Table directory, refactored and auto loaded
     * Implement setPlgTable() in Plugin
     */
    protected function setLegacyTable() {
        // Plugin does not provide proper values for members, which are used in the functions that modify the admin forms and the database
        if (empty($this->formDir) || empty($this->tableDir) || empty($this->plgTableName) || empty($this->tableKey)) {
            $this->hasOwnTableInDatabase = false;
        }
        else {
            $tableClass = 'Table' . $this->plgTableName;
            $path = $this->tableDir . '/' . (lcfirst($this->plgTableName)) . '.php';
            if (!class_exists($tableClass) && file_exists($path)) {
                require_once($path);
                try {
                    $db = Factory::getContainer()->get(DatabaseInterface::class);
                    $this->plgTable = new $tableClass($db);
                }
                catch (\RuntimeException $e) {
                    $this->hasOwnTableInDatabase = false;
                }
            }
            else {
                $this->hasOwnTableInDatabase = false;
            }
        }
    }
}