<?php
/**
 * Visforms field textarea business class
 *
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2012 vi-solutions
 * @since        Joomla 1.6
 */
namespace  Visolutions\Component\Visforms\Site\Lib\Business;

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\String\StringHelper;
use Visolutions\Component\Visforms\Site\Lib\Message\MinLenMessage;
use Visolutions\Component\Visforms\Site\Lib\Message\MaxLenMessage;
use Visolutions\Component\Visforms\Site\Lib\Message\RequiredMessage;
use Visolutions\Component\Visforms\Site\Lib\Validation\MinNumberValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\MaxNumberValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\NotEmptyValidation;

class TextareaFieldBusiness extends Business
{
	public function getFields() {
		$this->setField();
		return $this->fields;
	}

	protected function setField() {
		$this->setIsDisabled();
		if (isset($this->field->dataSource) && $this->field->dataSource == 'post') {
			$this->validatePostValue();
		}
		$this->addShowWhenForForm();
	}

	protected function validatePostValue(): void {
		// rules for textarea are: minlength, maxlength, unique value
		// update $this->field with value from $this->fields
		$this->updateField();
		$valid = true;
		// only to perform when the value is not empty
		if ($this->field->initvalue != "") {
            // do not validate the value in initvalue for disabled fields, if initvalue and configurationDefault are exactly identical
            if ((!empty($this->field->isDisabled) && ($this->field->initvalue === $this->field->configurationDefault))) {
                return;
            }
			// check for right minlength
			if ((isset($this->field->validate_minlength)) && ($this->field->validate_minlength != '')) {
				$mincount = $this->field->validate_minlength;
				$count = StringHelper::strlen($this->field->initvalue);
				$validation = new MinNumberValidation(array('count' => $count, 'mincount' => $mincount));
				if (!$validation->validate()) {
					// invalid value
					$valid = false;
                    $message = new MinLenMessage($this->field->label, $this->field->custom_php_error, array('mincount' => $mincount));
                    $error = $message->getMessage();
					// attach error to form
					$this->setErrors($error);
				}
			}

			// check for right maxlength
			if ((isset($this->field->attribute_maxlength)) && ($this->field->attribute_maxlength != '')) {
				$maxcount = $this->field->attribute_maxlength;
				$count = StringHelper::strlen($this->field->initvalue);
				$validation = new MaxNumberValidation(array('count' => $count, 'maxcount' => $maxcount));
				if (!$validation->validate()) {
					// invalid value
					$valid = false;
					$message = new MaxLenMessage($this->field->label, $this->field->custom_php_error, array('maxcount' => $maxcount));
                    $error = $message->getMessage();
					// attach error to form
					$this->setErrors($error);
				}
			}
			// validate unique field value in database
			$this->validateUniqueValue();
		}

		// at least one validation failed
		if (!$valid) {
			$this->field->isValid = false;
		}
	}

	public function validateRequired() {
		if (isset($this->field->dataSource) && $this->field->dataSource == 'post') {
			// check that a value is set if field is required
			if (isset($this->field->attribute_required)) {
				// only for fields that are not disabled
				if (empty($this->field->isDisabled)) {
				    $validation = new NotEmptyValidation(array('value' => $this->field->initvalue));
                    if (!$validation->validate()) {
						$this->field->isValid = false;
                        $message = new RequiredMessage($this->field->label, $this->field->custom_php_error);
                        $error = $message->getMessage();
						// attach error to form
						$this->setErrors($error);
					}
				}
			}
		}
		return $this->field;
	}

	/**
	 * we always use the configuration defaults as field "value" (attribute value, attribute selected, attribute checked or text in textarea)
	 * only then, we can reset the field properly
	 * we use javascript to set field "value state" (val(), prop selected, checked...) to the proper value (user input, configuration default...)
	 */
	public function setFieldValueProperties() {
		// stored (validated) "userinput" in new parameter
		$this->field->userInput = $this->getUserInputForJs();
		// set value, which is first displayed to the configuration defaults
		$this->field->initvalue = $this->field->configurationDefault;
		return $this->field;
	}

	private function getUserInputForJs() {
		$value = $this->field->initvalue;
		if ($this->isEditTask) {
			// use configuartion default as default, if field is disabled
			if (!empty($this->field->isDisabled)) {
				$value = $this->field->configurationDefault;
			}
		}
		if ($this->isRedisplayEditTask) {
			if (!empty($this->field->isDisabled)) {
				// if field was originally not disabled use dbValue
				$fieldsDisabledState = Factory::getApplication()->getUserState('com_visforms.fieldsDisabledState.' . $this->form->context, null);
				if (!empty($fieldsDisabledState) && (is_array($fieldsDisabledState)) && (empty($fieldsDisabledState[$this->field->name])) && isset($this->field->editValue)) {
					$value = $this->field->editValue;
				}
				else {
					$value = $this->field->configurationDefault;
				}
			}
		}
		// Do not use single quotes with \n\r!
        if (!empty($value)) {
            return str_replace(array('@'), array('&#64'), $value);
        }
        return $value;
	}
}