<?php
/**
 * Visforms model for Visforms
 *
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2012 vi-solutions
 * @since        Joomla 1.6
 */

namespace Visolutions\Component\Visforms\Site\Model;

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\Registry\Registry;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\HTML\HTMLHelper;
use Visolutions\Component\Visforms\Administrator\Helper\AefHelper;
use Visolutions\Component\Visforms\Administrator\Helper\MediaHelper;
use Visolutions\Component\Visforms\Administrator\Service\HTML\Select as Visformsselect;
use Visolutions\Component\Visforms\Site\Event\Visforms\VisformsBeforeFormSaveAfterUploadEvent;
use Visolutions\Component\Visforms\Site\Lib\Layout\Helper\FormLayoutHelper;
use Visolutions\Component\Visforms\Site\Log\VisformsLogTrait;
use Visolutions\Component\Visforms\Site\Model\Helper\Mail\BaseMailHelper;
use Visolutions\Component\Visforms\Site\Model\Helper\Mail\FormMailAdapter;
use Visolutions\Component\Visforms\Administrator\Helper\VisformsHelper;
use Visolutions\Component\Visforms\Administrator\Visfield\VisfieldPlaceholder;
use Visolutions\Component\Visforms\Administrator\Visfield\VisfieldPlaceholderEntry;
use Visolutions\Component\Visforms\Site\Lib\Business\FieldBusinessAwareTrait;
use Visolutions\Component\Visforms\Site\Lib\Field\FieldAwareTrait;
use Visolutions\Component\Visforms\Site\Lib\Html\Field\HtmlFieldAwareTrait;
use Visolutions\Component\Visforms\Site\Lib\Html\Layout\HtmlLayoutAwareTrait;

/**
 * Visforms model
 *
 * @package        Joomla.Site
 * @subpackage     com_visforms
 * @since          1.6
 */
class VisformsModel extends BaseDatabaseModel
{
    use VisformsLogTrait;
    use FieldBusinessAwareTrait;
    use FieldAwareTrait;
    use HtmlFieldAwareTrait;
    use HtmlLayoutAwareTrait;
	protected $_id;
	protected $input;
	protected $fields;
	// Member is private
    // getForm() is implemented in VisformsModel
    // EditModel extends VisformsModel
    // When getting form using getForm() from EditModel, this->form is null in EditModel
    // But getForm() always loads form from user state and sets it back into user state
    // Edit Model accesses form through getForm() and/or user state
    // What is the difference in making $form protected, so that $this->form would be set in EditModel?
	protected $form;
	public static $newSubmission = 0;
	public static $editSubmission = 1;
	public static $displayStateIsNew = 1;
	public static $displayStateIsRedisplay = 2;
	public static $displayStateIsNewEditData = 3;
	public static $displayStateIsRedisplayEditData = 4;
    protected $hasSub;
	protected $supportedFieldTypes;
	protected $context;
	protected $caller;
	protected $dataEditMenuExists = false;
	protected $breakPoints = array('Sm', 'Md', 'Lg', 'Xl', 'Xxl');
	protected $mailHelper;

	public function __construct($config = array(), ?MVCFactoryInterface $factory = null) {
		$config['id'] = (!empty($config['id'])) ? $config['id'] : null;
		$config['context'] = (!empty($config['context'])) ? $config['context'] : '';
		$this->input = Factory::getApplication()->getInput();
        $this->hasSub = AefHelper::checkAEF();
		$this->setId($config['id']);
		$this->setCaller($config['context']);
		$this->setContext($config['context']);
		$this->setSupportedFieldTypes();
		parent::__construct($config, $factory);
		// never use functions with HTMLHelper before parent::__construct
        // Or include Service class in use statement (use  Visolutions\Component\Visforms\Administrator\Service\HTML\Visforms as VisformsHtmlHelper;)
        // and use class directly then VisformsHtmlHelper::checkDataViewMenuItemExists($this->_id);
        $this->setDataEditMenuExists();
        $this->initializeLogger('visformssubmit');
	}

	// store the form id in _id
	public function setId($id = null) {
		if (is_null($id)) {
			$id = $this->input->getInt('id', 0);
		}
		$this->_id = $id;
	}

	protected function setCaller($context = '') {
		$this->caller = '';
		if ((!empty($context)) && (str_contains($context, 'plgvfformview_'))) {
			$this->caller = 'plgvfformview';
		}
		if ((!empty($context)) && (str_contains($context, 'vfedit'))) {
			$this->caller = 'vfedit';
		}
		if ((!empty($context)) && (str_contains($context, 'modvisform'))) {
			$this->caller = 'modvisform';
		}
	}

	protected function setContext($context = '') {
		// we may come from the controller, context is transported as post value then
		if (empty($context)) {
			$context = $this->input->getCmd('context', '');
		}
		// we deal with form displayed by a menu item and have no specific context
		// we fall back to our old standard 'context'
		if (empty($context)) {
			$context = 'form' . $this->_id;
		}
		$this->context = $context;
	}

	public function getContext() {
		return $this->context;
	}

	protected function setSupportedFieldTypes() {
		$this->supportedFieldTypes = array(
			'text', 'password', 'email', 'date', 'number', 'url', 'tel', 'hidden', 'checkbox', 'multicheckbox', 'radio', 'select', 'file', 'image', 'submit', 'reset', 'fieldsep', 'textarea'
		);
		if (!empty($this->hasSub)) {
			$this->supportedFieldTypes[] = 'pagebreak';
			$this->supportedFieldTypes[] = 'calculation';
			$this->supportedFieldTypes[] = 'location';
			$this->supportedFieldTypes[] = 'signature';
			$this->supportedFieldTypes[] = 'radiosql';
			$this->supportedFieldTypes[] = 'selectsql';
			$this->supportedFieldTypes[] = 'multicheckboxsql';
		}
	}

	// get the form definition from the database or the stored form definition from the session
	public function getForm() {
		$app = Factory::getApplication();
		$form = $app->getUserState('com_visforms.' . $this->context);
		$storedFormIsValid = $this->validateCachedFormSettings($form);
		// only use stored form if it's settings are valid
		if (empty($storedFormIsValid)) {
			$this->cleanUserState();
			$db = $this->getDatabase();
			$query = $db->createQuery();
			$query->select('*')
				->from($db->qn('#__visforms'))
				->where($db->qn('id') . ' = ' . $this->_id);
			try {
				$db->setQuery($query);
				$form = $db->loadObject();
			}
			catch (\RuntimeException $e) {
				$form = false;
			}
			$this->form = $form;
			if (empty($form)) {
				return $form;
			}
			$mailadapter = $this->getMailAdapter();
			if (is_object($mailadapter)) {
				$this->form->emailreceiptsettings = $mailadapter->receipt();
				$this->makePropertyNameShort('emailreceiptsettings');
				$this->form->emailresultsettings = $mailadapter->result();
				$this->makePropertyNameShort('emailresultsettings');
			}
			$this->form->savesettings = VisformsHelper::registryArrayFromString($this->form->savesettings);
			$this->makePropertyNameShort('savesettings', true);
			$this->form->subredirectsettings = VisformsHelper::registryArrayFromString($this->form->subredirectsettings);
            // fix invalid types
            $this->form->subredirectsettings['pdf_download_link_template'] = (!empty($this->form->subredirectsettings['pdf_download_link_template'])) ? VisformsHelper::fixInvalidMultiSelectOption($this->form->subredirectsettings['pdf_download_link_template']) : array();
			$this->makePropertyNameShort('subredirectsettings', true);
			$this->form->frontendsettings = VisformsHelper::registryArrayFromString($this->form->frontendsettings);
			// fix invalid types
            $this->form->frontendsettings['singleRecordPdfTemplate'] = (!empty($this->form->frontendsettings['singleRecordPdfTemplate'])) ? VisformsHelper::fixInvalidMultiSelectOption($this->form->frontendsettings['singleRecordPdfTemplate']) : array();
            $this->form->frontendsettings['listPdfTemplate'] = (!empty($this->form->frontendsettings['listPdfTemplate'])) ? VisformsHelper::fixInvalidMultiSelectOption($this->form->frontendsettings['listPdfTemplate']) : array();
			$this->makePropertyNameShort('frontendsettings');
			$this->form->layoutsettings = VisformsHelper::registryArrayFromString($this->form->layoutsettings);
            // fix old layout framework values, reset to matching modern framework
            if (isset($this->form->layoutsettings['formlayout'])) {
                if (in_array($this->form->layoutsettings['formlayout'], array('btdefault', 'bt3default', 'bt4mcindividual'))) {
                    $this->form->layoutsettings['formlayout'] = 'bt5';
                }
                if ($this->form->layoutsettings['formlayout'] === 'uikit2') {
                    $this->form->layoutsettings['formlayout'] = 'uikit3';
                }
            }
			$this->makePropertyNameShort('layoutsettings');
			$this->form->captchaoptions = VisformsHelper::registryArrayFromString($this->form->captchaoptions);
			$this->makePropertyNameShort('captchaoptions');
			$this->form->viscaptchaoptions = VisformsHelper::registryArrayFromString($this->form->viscaptchaoptions);
			if (empty($this->hasSub)) {
				$this->form->preventsubmitonenter = 0;
				$this->form->defaultresponsive = 0;
			}
			$formLayout = new FormLayoutHelper($this->form);
			$this->form = $formLayout->fixInvalidLayoutSelection();
            if (!isset($this->form->displaysublayout)) {
                // set to default
                $this->form->displaysublayout = 'horizontal';
            }
			$this->setDisplayState();
			// initialize helper properties
			$this->form->errors = array();
			$this->form->steps = 1;
			$this->form->accordioncounter = (int) 0;
			$this->form->mapCounter = (int) 0;
			$this->form->canHideSummaryOnMultiPageForms = (empty($this->hasSub)) ? false : true;
			$this->form->mpdisplaytype = (empty($this->form->mpdisplaytype) || empty($this->hasSub)) ? 0 : $form->mpdisplaytype;
			$this->form->firstpanelcollapsed = (empty($this->form->mpdisplaytype) || empty($this->form->firstpanelcollapsed)) ? 0 : $this->form->firstpanelcollapsed;
            $this->form->mpfbuttonposition = (!empty($this->form->mpdisplaytype)) ? 'center' : (empty($this->form->mpfbuttonposition) ? '' : $this->form->mpfbuttonposition);
            // focus for multipage form according to configuration. Accordion always focus top
            $this->form->mpforcusclass = ((!empty($this->form->mpfocus)) ? (($this->form->mpfocus == '1') ? ' mpfocustop' : ' mpfocusbottom') : (($this->form->mpdisplaytype == '1') ? ' mpfocustop' : ''));
			$this->form->context = $this->context;
			if (empty($this->hasSub)) {
				$this->form->redirecttoeditview = false;
			}
			$this->form->dataEditMenuExists = $this->dataEditMenuExists;
			if (empty($this->form->formprocessingmessage)) {
				$this->form->formprocessingmessage = Text::_('COM_VISFORMS_FORM_PROCESSING_DEFAULT_MESSAGE');
			}
			// trigger content Events on formprocessingmessage
            $this->form->formprocessingmessage = HTMLHelper::_('content.prepare', $this->form->formprocessingmessage);
            // disable Captcha on form edit
            if ($this->form->displayState == self::$displayStateIsNewEditData || $this->form->displayState == self::$displayStateIsRedisplayEditData) {
                $this->form->captcha = "0";
            }
            // ReCaptcha Support removed in 5.0.0
			// ReCaptchas readded in 5.3.0 have keys 4 and 5
			if (!empty($this->form->captcha) && ($this->form->captcha == "2")) {
				$this->form->captcha = "0";
			}
            $this->form = $formLayout->setLayoutOptions();
            $app->setUserState('com_visforms.' . $this->context, $this->form);
		}
		else {
			$this->form = $form;
		}
		return $this->form;
	}

	protected function cleanUserState() {
		$app = Factory::getApplication();
		$task = $this->input->getCmd('task', '');
		// urlparams are stored in the session, so that we can use them to reset invalid user inputs and/or disabled fields to the proper default value, which may be set as an url param
		// stored url params may be junk, for example if a user has just left the form (clicked to another menu item...)
		// these stored url params are only no junk if the task is 'send', else they must be removed
		if ($task !== 'send') {
			if ($app->getUserState('com_visforms.urlparams.' . $this->context, null)) {
				$app->setUserState('com_visforms.urlparams.' . $this->context, null);
			}
		}
		// we stored the disabled stated that results from the stored user inputs in the session
		// these stored disabled states are only no junk, if the task is saveedit
		if ($task !== 'saveedit') {
			if ($app->getUserState('com_visforms.fieldsdisabledstate.' . $this->context, null)) {
				$app->setUserState('com_visforms.fieldsdisabledstate.' . $this->context, null);
			}
		}
	}

	protected function setDisplayState() {
		// the display state is used in the field.php function setQueryValue in order to decide if url params from a get request should be stored in the session
		// url params (from get) are only stored if $displayStateIsNew and we are not in an edit view task
		// the display state is also used in the plugin mail attachments and visdoubleoptin in order to decide, if we send a mail after form submit or after form edit
		// the display state is used in business.php function setOrgDisabledStateFromStoredDataInUserState()
		$task = $this->input->getCmd('task', '');
		if ($task === "editdata") {
			$this->form->displayState = self::$displayStateIsNewEditData;
		}
		else if ($task === "saveedit") {
			$this->form->displayState = self::$displayStateIsRedisplayEditData;
		}
		else {
			$this->form->displayState = self::$displayStateIsNew;
		}
	}

	protected function makePropertyNameShort($propertiesName, $subOnly = false) {
		foreach ($this->form->$propertiesName as $name => $value) {
			if ($subOnly && empty($this->hasSub)) {
				$value = false;
			}
			// make names shorter and set all properties as properties of form object
			$this->form->$name = $value;
		}
	}

	public function getItems() {
		// make sure the form is created
		$this->getForm();
		$db = $this->getDatabase();
		$query = $db->createQuery();
		$supportedFieldTypes = implode("','", $this->supportedFieldTypes);
		$query->select('*')
			->from($db->qn('#__visfields'))
			->where($db->qn('fid') . ' = ' . $this->_id)
			->where($db->qn('published') . " = " . 1)
			->where('not ' . $db->qn('editonlyfield') . ' = ' . 1)
			->where($db->qn('typefield') . ' in (\'' . $supportedFieldTypes . '\')')
			->order($db->qn('ordering') . ' asc');
		$items = $this->_getList($query);
		return $items;
	}

	public function getEditOnlyItems() {
		$db = $this->getDatabase();;
		$query = $db->createQuery();
		$query->select('*')
			->from($db->qn('#__visfields'))
			->where($db->qn('fid') . ' = ' . $this->_id)
			->where($db->qn('published') . " = " . 1)
			->where($db->qn('editonlyfield') . ' = ' . 1)
			->order($db->qn('ordering') . ' asc');
		$items = $this->_getList($query);
		return $items;
	}

	public function getValidatedFields($submissionType = 0) {
		$visform = $this->getForm();
		$app = Factory::getApplication();
		$this->fields = $app->getUserState('com_visforms.' . $this->context . '.fields');
		if (!is_array($this->fields)) {
			$fields = $this->getItems();
			$n = count($fields);
			for ($i = 0; $i < $n; $i++) {
				// remove unsupported field types
				if (!in_array($fields[$i]->typefield, $this->supportedFieldTypes)) {
					unset($fields[$i]);
					continue;
				}
			}
			// reset keys
			$fields = array_values($fields);
			// get new count
			$n = count($fields);
			// get basic field definition
			for ($i = 0; $i < $n; $i++) {
			    $this->setField($fields[$i], $visform);
				$ofield = $this->getField();
				if (is_object($ofield)) {
					if ($submissionType == self::$editSubmission) {
						$cid = $this->input->get('cid', 0, 'int');
						if (!empty($cid)) {
							$ofield->setRecordId($cid);
						}
					}
					$fields[$i] = $ofield->getField();
				}
			}
			// get new count
			$n = count($fields);
			// perform business logic
			for ($i = 0; $i < $n; $i++) {
                $this->setBusinessField($fields[$i], $visform, $fields);
                $ofield = $this->getBusinessField();
				if (is_object($ofield)) {
					if (isset($fields[$i]->typefield) && ($fields[$i]->typefield !== "calculation")) {
						// as there may be interactions between the field processed and the rest of the form fields we always return the fields array
						$fields = $ofield->getFields();
					}
				}
			}
			// only after we have performed the business logic on all fields we know which fields are disabled
			// we can do some further field specific business stuff only now
			// reset default values of disabled fields
			// validate the "required" - omit the required validation for disabled fields!
			// we use the business class for this as well
			for ($i = 0; $i < $n; $i++) {
                $this->setBusinessField($fields[$i], $visform, $fields);
                $ofield = $this->getBusinessField();
				if (is_object($ofield)) {
					if (isset($fields[$i]->typefield) && ($fields[$i]->typefield !== "calculation")) {
                        $ofield->setOrgDisabledStateFromStoredDataInUserState();
                        $fields[$i] = $ofield->validateRequired();
                        $fields[$i] = $ofield->setCustomPhpErrorMessage();
						$fields[$i] = $ofield->setFieldValueProperties();
					}
				}
			}
			// fields of type calculation only set the custom javascript that calculates the fields in the form view and calculates the dbValue if we are in a "send" task
			// both functions need all other fields to be completely processed so that the calculation uses the correct values
			// if the form is displayed, values of calculation fields are calculated with javascript according to all field settings
			// if the submitted data are stored, values are calculated completely independently
			// the value only depends on the values of all other fields, but it does not matter if a calculation field is disabled itself or not
			// having a conditional calculation field would only affect, whether a user can see the field or not
			for ($i = 0; $i < $n; $i++) {
                $this->setBusinessField($fields[$i], $visform, $fields);
                $ofield = $this->getBusinessField();
				if (is_object($ofield)) {
					if (isset($fields[$i]->typefield) && ($fields[$i]->typefield === "calculation")) {
						// as there may be interactions between the field processed and the rest of the form fields we always return the fields array
						$fields = $ofield->getFields();
                        $fields[$i] = $ofield->setCustomPhpErrorMessage();
					}
				}
			}
			$this->fields = $fields;
		}
		$app->setUserState('com_visforms.' . $this->context . '.fields', $this->fields);
		return $this->fields;
	}

	public function reloadFields() {
		// this function is called, when user task send is preform and user inputs were valid but an error occurred on storing data in db
		// this is usually the case, when a unique value validation was no longer "valid" because another recordset was stored in the meantime
		// the form is invalid then, and we are on the way to re-display the form,
		// so we assume that the user inputs are still valid expect the value of the field with the failed unique value validation
		// basically this functions makes sure, that all already used options of radios, selects and checkbox groups are disabled and that text inputs with invalid duplicate values are marked as invalid
		// if the invalid field is disabled the default values are restored according to the implementation of the resetDisabledFieldToDefaultvalue() function in the fields business class
		$visform = $this->getForm();
		$visform->steps = (int) 1;
		$visform->accordioncounter = (int) 0;
		$visform->mapCounter = (int) 0;
		$visform->canHideSummaryOnMultiPageForms = (empty($this->hasSub)) ? false : true;
		$visform->mpdisplaytype = (empty($visform->mpdisplaytype) || empty($this->hasSub)) ? 0 : $visform->mpdisplaytype;
		$app = Factory::getApplication();
		$this->fields = $app->getUserState('com_visforms.' . $this->context . '.fields');
		if (!is_array($this->fields)) {
			// should not happen because wie have already stored fields in user state
			// but if, the field should be completely correct
			$fields = $this->getValidatedFields();
		}
		else {
			$fields = $this->getItems();
			$n = count($fields);
			// get basic field definition
			for ($i = 0; $i < $n; $i++) {
                $this->setField($fields[$i], $visform);
                $ofield = $this->getField();
				if (is_object($ofield)) {
					$fields[$i] = $ofield->getField();
				}
			}
			// perform business logic
			for ($i = 0; $i < $n; $i++) {
				// no field type specific process order necessary on reload field
				// if we reload the fields we know, that we will display the form
				// at this point it does not matter which value is calculated for a field of type calculation because it is recalculated by javascript on document.ready
				// as the custom javascript only replaces field names with html id attributes, no special treatment of these fields is necessary for this task either
                $this->setBusinessField($fields[$i], $visform, $fields);
                $ofield = $this->getBusinessField();
				if (is_object($ofield)) {
					// as there may be interactions between the field processed and the rest of the form fields we always return the fields array
					$fields = $ofield->getFields();
				}
			}
			// only after we have performed the business logic on all fields we know which fields are disabled
			// we can validate the "required" only then, because we have to omit the required validation for disabled fields!
			// we use the business class for this as well
			for ($i = 0; $i < $n; $i++) {
                $this->setBusinessField($fields[$i], $visform, $fields);
                $ofield = $this->getBusinessField();
				if (is_object($ofield)) {
					// we don't need to set the orgDisabledStates here because they only needs to be set once and that is already done!
					$fields[$i] = $ofield->validateRequired();
                    $fields[$i] = $ofield->setCustomPhpErrorMessage();
                    $fields[$i] = $ofield->setFieldValueProperties();
                }
			}
		}
		$this->fields = $fields;
		$app->setUserState('com_visforms.' . $this->context . '.fields', $this->fields);
		return $this->fields;
	}

	// called from view
	public function getFields() {
		$visform = $this->getForm();
		$app = Factory::getApplication();
		$this->fields = $app->getUserState('com_visforms.' . $this->context . '.fields');
		if (!is_array($this->fields)) {
			$fields = $this->getValidatedFields();
		}
		else {
			$fields = $this->fields;
		}
		$n = count($fields);
		// prepare HTML
		for ($i = 0; $i < $n; $i++) {
		    $this->setHtmlField($fields[$i]);
			$html = $this->getHtmlField();
			if (is_object($html)) {
			    $this->setHtmlLayout($visform->formlayout, $html, $visform->displaysublayout);
				$oField = $this->getHtmlLayout();
				if (is_object($oField)) {
					$fields[$i] = $oField->prepareHtml();
				}
			}
		}
		$this->fields = $fields;
		return $this->fields;
	}

	function addHits() {
		$db = $this->getDatabase();;
		$visform = $this->getForm();
		if (isset($visform->id)) {
			$query = $db->createQuery();
			$hits = $visform->hits + 1;
			$updatefields = array($db->quoteName('hits') . ' = ' . $hits);
			$conditions = array($db->quoteName('id') . ' = ' . $visform->id);
			$query->update($db->quoteName('#__visforms'))->set($updatefields)->where($conditions);
			$db->setQuery($query);
			$db->execute();
		}
	}

	function saveData() {
		// Form and Field structure and info from db
		$visform = $this->getForm();
        $currentDate = Factory::getDate()->format('Y-m-d H:i:s');
        $isNotPublishedYet = $visform->publish_up > $currentDate;
        $isExpired = !is_null($visform->publish_down) && $visform->publish_down < $currentDate;
        if ($visform->published != '1' || $isNotPublishedYet || $isExpired) {
            // log error to inspector
            $this->addLogEntry(Text::_('COM_VISFORMS_SAVING_DATA_FAILED') . ' ' . Text::_('COM_VISFORMS_FORM_IS_UNPUBLISHED'), Log::ERROR);
            // stop Visforms from saving the form; message must not be empty, in order to insure proper handling in controller
            // this message is pushed into the form errors and displayed to the user
            throw new \RuntimeException(Text::_('COM_VISFORMS_SAVING_DATA_FAILED') . ' ' . Text::_('COM_VISFORMS_FORM_MISSING'));
        }
		$fields = $this->getValidatedFields();
		$visform->fields = $fields;
        try {
            MediaHelper::uploadFiles($visform);
        }
        catch (\RuntimeException $e) {
            // file upload was not successful
            // log error to inspector
            // proceed with form submit
            $this->addLogEntry(Text::sprintf('COM_VISFORMS_PROBLEMS_SENDING_FORM', Text::_('COM_VISFORMS_CANNOT_UPLOAD_FILES')) . $e->getMessage(), Log::WARNING);
        }
		// fire custom event
        $beforeSaveFormAfterUploadEvent = new VisformsBeforeFormSaveAfterUploadEvent('onVisformsBeforeFormSaveAfterUpload', [
            'subject' => $visform,
            'context' => 'com_visforms.form.form',
            'fields' => $fields
        ]);
        $this->getDispatcher()->dispatch('onVisformsBeforeFormSaveAfterUpload', $beforeSaveFormAfterUploadEvent);
		if ($visform->saveresult == 1) {
		    try {
                $this->storeData($visform);
            }
            catch (\RuntimeException $e) {
                $message = $e->getMessage();
                if (!empty($message)) {
                    // log error to inspector
                    $this->addLogEntry(Text::_('COM_VISFORMS_SAVING_DATA_FAILED') . ' ' . $message, Log::ERROR);
                    // stop Visforms from saving the form; message must not be empty, in order to insure proper handling in controller
                    // this message is pushed into the form errors and displayed to the user
                    throw new \RuntimeException(Text::_('COM_VISFORMS_SAVING_DATA_FAILED') . ' ' . $message);
                }
                else {
                    throw New \RuntimeException('');
                }
            }
		}
        if ($visform->emailreceipt == 1 || $visform->emailresult == 1) {
            $this->mailHelper = new BaseMailHelper($this, $visform);
        }
		if ($visform->emailreceipt == 1) {
            $this->mailHelper->sendUserMail();
		}
		if ($visform->emailresult == 1) {
            $this->mailHelper->sendAdminMail();
		}
		return true;
	}

	function getMenuparams() {
		$app = Factory::getApplication();
		$menu_params = $app->getParams();
		$this->setState('menu_params', $menu_params);
		return $menu_params;
	}

	private function storeData($visform) {
		$folder = $visform->uploadpath;
		$db = $this->getDatabase();;
		$lockValidationFields = array();
		$datas = new \stdClass();
		$datas->created = $visform->dataRecord_created;
        $datas->ipaddress = $visform->dataRecord_ipaddress;
		$datas->published = $visform->dataRecord_published;
		$datas->created_by = $visform->dataRecord_created_by;
		$n = count($visform->fields);
		for ($i = 0; $i < $n; $i++) {
			$field = $visform->fields[$i];
			// buttons, pagebreaks, fieldseps are not stored, therefore they have a value of NULL in the db
			if ((empty($field->isButton)) && ($field->typefield != 'pagebreak') && ($field->typefield != 'fieldsep')) {
				// make sure that dbValue of disabled fields is set to ''
				if (!empty($field->isDisabled)) {
					// all disabled fields are stored with empty value in db
					$dbfieldvalue = "";
				}
				else {
					if ($field->typefield == 'file' && isset($_FILES[$field->name]['name']) && $_FILES[$field->name]['name'] != '') {
						// save folder and filename
                        // Todo: actually the field property dbValue was already set during upload in media helper
                        // Todo: check if we really need this conditional statement
						$file = new \stdClass();
						$file->folder = $folder;
						$file->file = $field->file['new_name'];
						$registry = new Registry($file);
						$dbfieldvalue = $registry->toString();
					}
					else {
						if (isset($field->dbValue) && $field->typefield != 'file') {
							$dbfieldvalue = $field->dbValue;
						}
						else {
							// Todo Check, if we should throw an error here and redisplay the form, because this should actually never be the case
							$dbfieldvalue = '';
						}
					}
				}
				$dbfieldname = 'F' . $field->id;
				// Add field to insert object
				$datas->$dbfieldname = $dbfieldvalue;
				if ((!empty($field->uniquevaluesonly)) && (isset($dbfieldvalue)) && (empty($field->isDisabled)) && ($dbfieldvalue !== '')) {
					$validation = new \stdClass();
					$validation->id = $field->id;
					$validation->name = $dbfieldname;
					$validation->value = $dbfieldvalue;
					// parameter uniquepublishedvaluesonly does not yet exist, therefore results always in 0
					$validation->publishedonly = (!empty($field->uniquepublishedvaluesonly)) ? 1 : 0;
					$validation->label = $field->label;
					$validation->typefield = $field->typefield;
					$lockValidationFields[] = $validation;
				}
				unset($file);
				unset($dbfieldvalue);
				unset($dbfieldname);
				unset($validation);
			}
		}
		try {
			// check if value of field with unique value validation is already stored in the database
			foreach ($lockValidationFields as $test) {
				if ((!empty($test)) && (is_object($test))) {
					$query = $db->createQuery();
					$query->select($db->qn('id'))
						->from($db->qn('#__visforms_' . $visform->id));
					if (in_array($test->typefield, array('select', 'multicheckbox', 'selectsql', 'multicheckboxsql'))) {
						$formSelections = HTMLHelper::_('visformsselect.explodeMsDbValue', $test->value);
						$storedSelections = $query->concatenate(array($db->q(Visformsselect::$msdbseparator), $db->quoteName($test->name), $db->q(Visformsselect::$msdbseparator)));
						foreach ($formSelections as $formselection) {
							$formselection = '%' . Visformsselect::$msdbseparator . $formselection . Visformsselect::$msdbseparator . '%';
							$query->where('(' . $storedSelections . ' like ' . $db->q($formselection) . ')');
						}
					}
					else {
						// radio and radio sql fields are handled here
						$query->where($db->qn($test->name) . ' = ' . $db->q($test->value));
					}
					if (!empty($test->publishedonly)) {
						$query->where($db->qn('published') . ' = ' . 1);
					}
					try {
						$db->setQuery($query);
						$valueExistes = $db->loadResult();
					}
					catch (\RuntimeException $e) {
					}
					if (!empty($valueExistes)) {
						throw New \RuntimeException('');
					}
				}
			}
			$result = $db->insertObject('#__visforms_' . $visform->id, $datas);
		}
		catch (\RuntimeException $e) {
			$message = $e->getMessage();
			if (!empty($message)) {
				throw new \RuntimeException(Text::_('COM_VISFORMS_SAVING_DATA_FAILED') . ' ' . $message);
			}
			else {
				throw New \RuntimeException('');
			}
		}
		$visform->dataRecordId = $db->insertid();
		// store default values of edit only fields in db
		$editOnlyFields = $this->getEditOnlyItems();
		if (!empty($editOnlyFields)) {
			$editOnlyFieldData = new \stdClass();
			$editOnlyFieldData->id = $visform->dataRecordId;
			for ($i = 0; $i < count($editOnlyFields); $i++) {
                $this->setField($editOnlyFields[$i], $visform);
                $ofield = $this->getField();
                if (is_object($ofield)) {
					$editOnlyFields[$i] = $ofield->getField();
					// empty string is not a default value but "0" would be
					if (isset($editOnlyFields[$i]->editOnlyFieldDbValue) && $editOnlyFields[$i]->editOnlyFieldDbValue !== "") {
						$dbfieldname = 'F' . $editOnlyFields[$i]->id;
						// Add field to insert object
						$editOnlyFieldData->$dbfieldname = $editOnlyFields[$i]->editOnlyFieldDbValue;
					}
				}
			}
			$db->updateObject('#__visforms_' . $visform->id, $editOnlyFieldData, 'id');
		}

		// check whether someone has submitted and stored a form with the same unique value in the meantime (simultaneously)
		// if there are duplicate values for unique value fields in the db
		// if so, we check, if our recordset has the highest id in the group of record sets with duplicate values
		// if so, we delete the record set and throw an error
		foreach ($lockValidationFields as $test) {
			if ((!empty($test)) && (is_object($test))) {
				$query = $db->createQuery();
				$query->select($db->qn('id'))
					->from($db->qn('#__visforms_' . $visform->id));
				if (in_array($test->typefield, array('select', 'multicheckbox', 'selectsql', 'multicheckboxsql'))) {
					$formSelections = HTMLHelper::_('visformsselect.explodeMsDbValue', $test->value);
					$storedSelections = $query->concatenate(array($db->q(Visformsselect::$msdbseparator), $db->quoteName($test->name), $db->q(Visformsselect::$msdbseparator)));
					foreach ($formSelections as $formselection) {
						$formselection = '%' . Visformsselect::$msdbseparator . $formselection . Visformsselect::$msdbseparator . '%';
						$query->where('(' . $storedSelections . ' like ' . $db->q($formselection) . ')');
					}
				}
				else {
					// radio and radio sql fields are handled here
					$query->where($db->qn($test->name) . ' = ' . $db->q($test->value));
				}
				if (!empty($test->publishedonly)) {
					$query->where($db->qn('published') . ' = ' . 1);
				}
				$query->order($db->qn('id'));
				try {
					$db->setQuery($query);
					$checkValueExistes = $db->loadColumn();
				}
				catch (\RuntimeException $e) {
				}
				if ((!empty($checkValueExistes)) && (count($checkValueExistes) > 1) && (is_array($checkValueExistes))) {
					// remove the first element
					array_shift($checkValueExistes);
					// we are not the first recordset stored and have to delete ourselves and throw an error
					if (in_array($visform->dataRecordId, $checkValueExistes)) {
						$query = $db->createQuery();
						$query->delete($db->qn('#__visforms_' . $visform->id));
						$query->where($db->qn('id') . ' = ' . $visform->dataRecordId . ' LIMIT 1');
						try {
							$db->setQuery($query);
							$db->execute();
						}
						catch (\RuntimeException $e) {
						}
						// Empty exception message signals the controller,
                        // that this is a unique value validation which failed during storing the data in the database
						throw New \RuntimeException('');
					}
				}
			}
		}
		return true;
	}

    public function replacePlaceholder($form, $text = '', $dataTypes = array(), $textIsPlaceholderOnly = false) {
        // this function replaces any placeholder even those who have no matching replace value
        if (empty($text)) {
            return $text;
        }
        $placeholders = new VisfieldPlaceholder($text);
        while ($placeholders->hasNext()) {
            $placeholders->getNext();
            $replace = '';
            $pName = $placeholders->getPlaceholderPart('name');
            if (empty($pName)) {
                // should never happen: just remove the placeholder's
            }
            else if ($placeholders->isNonFieldPlaceholder() && (empty($dataTypes) || in_array($pName, $dataTypes))) {
                $pParams = $placeholders->getPlaceholderPart('params');
                // overhead placeholder
                switch ($pName) {
                    // form information
                    case 'formid' :
                        $replace = (!empty($form->dataRecordId)) ? $form->id : '';
                        break;
                    case 'formtitle' :
                        $replace = (!empty($form->title)) ? $form->title : '';
                        break;
                    // data record information
                    case 'id' :
                        $replace = (!empty($form->dataRecordId)) ? $form->dataRecordId : '';
                        break;
                    case 'created':
                    case 'created_by':
                    case 'modified':
                    case 'modified_by':
                    case 'ipaddress':
                    case 'ismfd':
                    case 'published':
                        if (isset($form->{'dataRecord_' . $pName})) {
                            $placeholder = VisfieldPlaceholderEntry::getInstance($pParams, $form->{'dataRecord_' . $pName}, $pName);
                            $replace = $placeholder->getReplaceValue();
                        }
                        else {
                            $placeholder = VisfieldPlaceholderEntry::getInstance($pParams, null, $pName);
                            $replace = $placeholder->getReplaceValue();
                        }
                        break;
                    default :
                        $placeholder = VisfieldPlaceholderEntry::getInstance($pParams, null, $pName);
                        $replace = $placeholder->getReplaceValue();
                        break;
                }
            }
            else if (is_array($form->fields)) {
                foreach ($form->fields as $field) {
                    $fieldName = (!empty($form->context)) ? str_replace($form->context, '', $field->name) : $field->name;
                    if ($pName === $fieldName && (empty($dataTypes) || in_array($field->typefield, $dataTypes))) {
                        $pParams = $placeholders->getPlaceholderPart('params');
                        $placeholder = VisfieldPlaceholderEntry::getInstance($pParams, ((!empty($field->isDisabled)) ? '' : $field->dbValue), $field->typefield, $field);
                        $replace = $placeholder->getReplaceValue();
                        break;
                    }
                }
            }
            // we expect the text to contain exactly one placeholder and nothing more, but that is not true. Remove placeholder
            if ($textIsPlaceholderOnly) {
                $fullPlaceholder = $placeholders->getPlaceholderPart('placeholder');
                if ($fullPlaceholder !== $text) {
                    $replace = '';
                }
            }
            // replace the match
            $placeholders->replace($replace);
            unset($replace);
        }
        return $placeholders->getText();
    }

	protected function validateCachedFormSettings($form) {
		if (empty($form)) {
			return false;
		}
		if (!is_object($form)) {
			return false;
		}
		if (empty($form->formlayout)) {
			return false;
		}
		return true;
	}

	public function getRedirectParams($fields, $query = array(), $formcontext = '') {
		if (empty($fields)) {
			return $query;
		}
		foreach ($fields as $field) {
			// setting this param is handled by the field
			// only set, if field option addtoredirecturl is enabled
			if (isset($field->redirectParam)) {
				$contextFreeFieldName = (empty($formcontext)) ? ($field->name) : substr($field->name, strlen($formcontext));
				$rdtParamName = (!empty($field->rdtparamname)) ? $field->rdtparamname : $contextFreeFieldName;
                $encode = (!empty($field->encoderedirectvalue)) ? $field->encoderedirectvalue : 0;
				switch ($field->typefield) {
					// just make sure that values of this field types are not added accidentally
					case 'file' :
					case 'image' :
					case 'submit' :
					case 'reset' :
						break;
					case 'select' :
					case 'multicheckbox' :
						$query[$rdtParamName] = array();
						foreach ($field->redirectParam as $value) {
							$query[$rdtParamName][] = (!empty($encode)) ? HTMLHelper::_('visforms.base64_url_encode', $value) : $value;
						}
						break;
					default :
						$query[$rdtParamName] = (!empty($encode)) ? HTMLHelper::_('visforms.base64_url_encode', $field->redirectParam) : $field->redirectParam;
						break;
				}
			}
		}
		return $query;
	}

	protected function setDataEditMenuExists() {
		// default value of $this->dataEditMenuExists is false
		if (empty($this->hasSub)) {
			return;
		}
		$dataViewMenuItemExists = HTMLHelper::_('visforms.checkDataViewMenuItemExists', $this->_id);
		$mysubmenuexists = HTMLHelper::_('visforms.checkMySubmissionsMenuItemExists');
		$this->dataEditMenuExists = $dataViewMenuItemExists ? $dataViewMenuItemExists : $mysubmenuexists;
	}

	public function getRecords() {
		$user = Factory::getApplication()->getIdentity();
		$userId = $user->id;
		$db = $this->getDatabase();;
		$query = $db->createQuery();
		$query->select('*')
			->from($db->quoteName('#__visforms_' . $this->_id))
			->where($db->quoteName('created_by') . " = " . $userId);
		try {
			$db->setQuery($query);
			$details = $db->loadObjectList();
			return $details;
		}
		catch (\Exception $ex) {
			return false;
		}
	}

	protected function removeForbiddenLayouts() {
	}

	// used in VisformsController
	public function checkFormViewMenuItemExists($id = null) {
		$app = Factory::getApplication();
		$id = (is_null($id)) ? $this->id : (int) $id;
		$menuitems = $app->getMenu()->getItems('link', 'index.php?option=com_visforms&view=visforms&id=' . $id);
		if ((!(empty($menuitems))) && (is_array($menuitems)) && (!empty($menuitems[0]->id))) {
			return $menuitems[0]->id;
		}
		return false;
	}

	public function setDataRecordOverheadValuesInForm() {
        // make data record overhead information available in form
        $user = Factory::getApplication()->getIdentity();
        $this->form->dataRecord_created = Factory::getDate()->toSql();
        $this->form->dataRecord_ipaddress = (!empty($this->form->save_exclude_ip) && !empty($this->hasSub)) ? '' : $_SERVER['REMOTE_ADDR'];
        $this->form->dataRecord_published = ($this->form->autopublish == 1) ? 1 : 0;
        $this->form->dataRecord_created_by = (isset($user->id)) ? $user->id : 0;
        $this->form->dataRecord_modified = null;
        $this->form->dataRecord_modified_by = 0;
        $this->form->dataRecord_ismfd = 0;
    }

    protected function getMailAdapter() {
	    return new FormMailAdapter($this->form, $this->caller);
    }
}