<?php
/**
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2018 vi-solutions
 */

namespace Visolutions\Component\Visforms\Administrator\Helper;

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\Database\DatabaseInterface;

class ConditionsHelper
{

	protected static array $restrictRestrictionMap = array(
		'_validate_equalTo' => 'UsedAsEqualTo',
		'_minvalidation_type' => 'UsedAsMinDate',
		'_maxvalidation_type' => 'UsedAsMaxdate',
		'_showWhen' => 'UsedAsShowWhen',
		'_equation' => 'UsedInCal',
		'_reload' => 'UsedAsReloadTrigger');

	protected static array $restrictionTypeNameMap = array(
		'usedAsEqualTo' => 'COM_VISFORMS_EQUAL_TO',
		'usedAsMinDate' => 'COM_VISFORMS_MIN_DATE_VALIDATION_TYPE',
		'usedAsMaxdate' => 'COM_VISFORMS_MAX_DATE_VALIDATION_TYPE',
		'usedAsShowWhen' => 'COM_VISFORMS_SHOW_WHEN',
		'usedInCal' => 'COM_VISFORMS_CALCULATION_EQUATION',
		'usedAsReloadTrigger' => 'COM_VISFORMS_USED_AS_RELOAD_TRIGGER');

	public static function getDefaultValueFromDb($id): string {
		$db = Factory::getContainer()->get(DatabaseInterface::class);
		$query = $db->createQuery();
		$query->select($db->quoteName(array('defaultvalue')))
			->from('#__visfields')
            ->where($db->qn('id') . ' = ' . $id);
		try {
			$db->setQuery($query);
			$result = $db->loadResult();
			return (!empty($result) ? $result : '');
		}
		catch (\RuntimeException $e) {
            return '';
        }
	}

	// note: also used in importExportHelper
	public static function saveDefaultValue($id, $value) {
		$db = Factory::getContainer()->get(DatabaseInterface::class);
		$query = $db->createQuery();
		$query->update($db->quoteName('#__visfields'))
			->set($db->quoteName('defaultvalue') . " = " . $db->quote($value))
			->where($db->quoteName('id') . " = " . $id);
		try {
            $db->setQuery($query);
            $db->execute();
        }
        catch (\RuntimeException $e) {}
	}

	public static function getOldFieldNameFromDb($id): string {
		$db = Factory::getContainer()->get(DatabaseInterface::class);
		$query = $db->createQuery();
		$query->select($db->quoteName(array('name')))
			->from('#__visfields')
            ->where($db->qn('id') . ' = ' . $id);
        try {
            $db->setQuery($query);
            $result = $db->loadResult();
            return (!empty($result) ? $result : '');
        }
        catch (\RuntimeException $e) {
            return '';
        }
	}

	public static function fixModifiedFieldNameInCalculation($oldFieldName, $newFieldName, $calFieldId) {
		$defaultValue = self::getDefaultValueFromDb($calFieldId);
		$pattern = '/\['.strtoupper($oldFieldName).']/';
		$defaultValue = preg_replace($pattern, '['.strtoupper($newFieldName).']', $defaultValue);
		self::saveDefaultValue($calFieldId, $defaultValue);
	}

	public static function getRestrictions($id): array {
		$db = Factory::getContainer()->get(DatabaseInterface::class);
		$query = $db->createQuery();
		// select restriction
		$query->select('restrictions')
			->from('#__visfields')
            ->where($db->qn('id') . ' = ' . $id);
        try {
            $db->setQuery($query);
            $result = $db->loadResult();
            return (!empty($result) ? VisformsHelper::registryArrayFromString($result) : array());
        }
        catch (\RuntimeException $e) {
            return  array();
        }
	}

	public static function saveRestriction($id, $value) {
		$db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->createQuery();
        $query->update($db->quoteName('#__visfields'))
            ->set($db->quoteName('restrictions') . " = " . $db->quote($value))
            ->where($db->quoteName('id') . " = " . $id);
		try {
            $db->setQuery($query);
            $db->execute();
        }
        catch (\RuntimeException $e) {}
	}

	public static function setRestrictsFromDb($id, $fid = null): array {
		$db = Factory::getContainer()->get(DatabaseInterface::class);
		$query = $db->createQuery();
		$query->select($db->quoteName(array('defaultvalue', 'name')))
			->from('#__visfields')
			->where($db->qn('id') . ' = ' . $id);
        try {
            $db->setQuery($query);
            $result = $db->loadObject();
            if (empty($result)) {
                return array();
            }
        }
        catch (\RuntimeException $e) {
            return array();
        }
		// convert the default value field to an array
		$defaultValues = VisformsHelper::registryArrayFromString($result->defaultvalue);
		$name = $result->name;
		return self::setRestricts($id, $defaultValues, $name, $fid);
	}

	public static function setRestricts($id, $defaultValues = array(), $name = '', $fid = null): array {
		$restricts = array();
		foreach ($defaultValues as $dfName => $dfValue) {
			foreach (self::$restrictRestrictionMap as $rName => $rValue) {
				if ((strpos($dfName, $rName) > 0)) {
					$className = 'Visolutions\\Component\\Visforms\\Administrator\\Visfield\\Restriction\\' . self::$restrictRestrictionMap[$rName];
					if (class_exists($className)) {
						$o = new $className($dfValue, $id, $name, $fid);
						$newRestricts = $o->getRestricts();
						if (!empty($newRestricts) && is_array($newRestricts)) {
							foreach ($newRestricts as $newRestrict) {
								array_push($restricts, $newRestrict);
							}
						}
					}
				}
			}
		}
		return $restricts;
	}

	public static function removeRestriction($restricts) {
		while (!empty($restricts)) {
			// pop the first ID off the stack
			$deletedRestriction = array_shift($restricts);
			// extract params in database field restrictions
			$restrictions = self::getRestrictions($deletedRestriction['restrictedId']);
			// if deletedRestriction is set, remove it
			foreach ($restrictions as $r => $v) {
				if ($r == $deletedRestriction['type']) {
					foreach ($v as $index => $restrictorId) {
						if ($restrictorId == $deletedRestriction['restrictorId']) {
							unset($restrictions[$r][$index]);
						}
					}
				}
			}
			foreach ($restrictions as $r => $v) {
				if ((is_array($v)) && (count($v) == 0)) {
					unset($restrictions[$r]);
				}
			}

			if (isset($restrictions) && is_array($restrictions)) {
				$restrictions = VisformsHelper::registryStringFromArray($restrictions);
			}
			// save the changed deletedRestriction
			self::saveRestriction($deletedRestriction['restrictedId'], $restrictions);
		}
	}

	public static function setRestriction($restricts) {
		while (!empty($restricts)) {
			// pop the first ID off the stack
			$newRestriction = array_shift($restricts);
			// extract params in database field restrictions
			$restrictions = self::getRestrictions($newRestriction['restrictedId']);
			// check if newRestriction type already exists, if not, create it as array
			if (!array_key_exists($newRestriction['type'], $restrictions)) {
				$restrictions[$newRestriction['type']] = array();
			}
			// add newRestriction of this type to restrictions of field, if the newRestriction already exists it is just overridden with the same value
			$restrictions[$newRestriction['type']][$newRestriction['restrictorName']] = $newRestriction['restrictorId'];
			if (isset($restrictions) && is_array($restrictions)) {
				$restrictions = VisformsHelper::registryStringFromArray($restrictions);
			}
			self::saveRestriction($newRestriction['restrictedId'], $restrictions);
		}
	}

	public static function getRemovedOptionIds($data) {
		$oldDefaultValues = VisformsHelper::registryArrayFromString(self::getDefaultValueFromDb($data['id']));
		$oldOptions = $oldDefaultValues['f_' . $data['typefield'] . '_list_hidden'];
		$options = $data['defaultvalue']['f_' . $data['typefield'] . '_list_hidden'];
		if ($oldOptions === $options) {
			// options have not been changed
			return false;
		}
		if (empty($oldOptions)) {
			$oldOptions = array();
		}
		else {
			$oldOptions = VisformsHelper::registryArrayFromString($oldOptions);
		}
		if (empty($options)) {
			$options = array();
		}
		else {
			$options = VisformsHelper::registryArrayFromString($options);
		}
		$oldOptionsIds = array_map(function ($element) {
			return $element['listitemid'];
		}, $oldOptions);
		$optionsIds = array_map(function ($element) {
			return $element['listitemid'];
		}, $options);
		return array_values(array_diff($oldOptionsIds, $optionsIds));
	}

	// $data[id] = id of select, radio, multieelect from which an option was removed
	// $id = id of field which is listed in the restrictions of the select, radio, mulitcheckbox and which may have a condition that uses the deleted option
	public static function removeDeletedOptionsDependencies($fieldName, $id, $deletedOptionsId, $data) {
		// try to run not to much code
		$restrictedId = $data['id'];
		$name = $data['name'];
		$oldDefaultValues = self::getDefaultValueFromDb($id);
		$oldDefaultValuesArray = VisformsHelper::registryArrayFromString($oldDefaultValues);
		$usedRemovedShowWhens = array();
		$usedRemovedOptionsIds = array();
		// simple check, if removed option is used as condition in this specific field
		foreach ($deletedOptionsId as $optionId) {
			$search = '"field' . $restrictedId . '__' . $optionId . '"';
			if (!str_contains($oldDefaultValues, $search)) {
				continue;
			}
			// removed options was used as condition, store information, so that we can sanitize db with as little effort as possible
			$usedRemovedShowWhens[] = 'field' . $restrictedId . '__' . $optionId;
			$usedRemovedOptionsIds[] = $optionId;
		}
		// only if option was used, sanitize conditional field and restrictions.
		if (!empty($usedRemovedShowWhens)) {
			$removeRestriction = true;
			// sanitize default values of conditional field and store them in db
			$showWhenValues = array();
			foreach ($oldDefaultValuesArray as $key => $value) {
				if (str_contains($key, '_showWhen')) {
					$showWhenValues = $value;
					break;
				}
			}
			// remove deleted showWhen values from showWhenValues Array and find out, if conditional field can be removed from restriction in select, radio, multiselect
			$newShowWhenValues = array_diff($showWhenValues, $usedRemovedShowWhens);
			if (!empty($newShowWhenValues)) {
				// check if all showWhenValues for the modified select, radio, multicheckbox have been removed
				foreach ($newShowWhenValues as $keyName => $keyValue) {
					if (str_contains($keyValue, 'field' . $restrictedId . '__')) {
						$removeRestriction = false;
						break;
					}
				}
				$oldDefaultValuesArray[$key] = $newShowWhenValues;
			}
			else {
				unset($oldDefaultValuesArray[$key]);
				$removeRestriction = true;
			}
			$newDefaultValues = VisformsHelper::registryStringFromArray($oldDefaultValuesArray);
			self::saveDefaultValue($id, $newDefaultValues);
			// sanitize restrictions of (select, radio, multicheckbox field) and store them in db
			if ($removeRestriction === true) {
				$restrictions = self::getRestrictions($restrictedId);
				$oldUsedAsShowWhenRestriction = (!empty($restrictions['usedAsShowWhen'])) ? $restrictions['usedAsShowWhen'] : array();
				if (!empty($oldUsedAsShowWhenRestriction)) {
					unset($oldUsedAsShowWhenRestriction[$fieldName]);
				}
				if (!empty($oldUsedAsShowWhenRestriction)) {
					$restrictions['usedAsShowWhen'] = $oldUsedAsShowWhenRestriction;
				}
				else {
					unset($restrictions['usedAsShowWhen']);
				}
				if (!empty($restrictions)) {
					$restrictions = VisformsHelper::registryStringFromArray($restrictions);
				}
				else {
					$restrictions = '';
				}
				self::saveRestriction($restrictedId, $restrictions);
			}
			// set a message
			Factory::getApplication()->enqueueMessage(Text::sprintf("COM_VISFORMS_OPTION_TOGGLES_DISPLAY", $name, $fieldName), 'notice');
		}
	}

	public static function canDelete($id, $name): bool {
		$restrictions = self::getRestrictions($id);
		if (!(empty($restrictions))) {
			foreach ($restrictions as $r => $value) {
				foreach ($value as $fieldName => $fieldId) {
					Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_HAS_RESTRICTIONS', $name, Text::_(self::$restrictionTypeNameMap[$r]), $fieldName), 'warning');
				}
			}
			return false;
		}
		return true;
	}

	public static function canSaveEditOnlyField($id, $name):bool {
		$restrictions = self::getRestrictions($id);
		if (!(empty($restrictions))) {
			foreach ($restrictions as $r => $value) {
				foreach ($value as $fieldName => $fieldId) {
					Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_HAS_SAVE_RESTRICTIONS', $name, Text::_(self::$restrictionTypeNameMap[$r]), $fieldName, Text::_('COM_VISFORMS_IS_EDIT_ONLY_FIELD_LABEL')), 'warning');
				}
			}
			return false;
		}
		return true;
	}

	public static function setConditionsInCopiedFields($idMap, $fid) {
		foreach ($idMap as $newId) {
			// get restrictions in new Field
			$oldRestricts = self::setRestrictsFromDb($newId, $fid);
			$newEqualToRestrict = "";
			$newMinDateRestrict = "";
			$newMaxDateRestrict = "";
			$newShowWhenRestricts = array();
			$newReloadTrigger = array();
			if (!empty($oldRestricts)) {
				// replace oldIds in restricts['restrictedId'] with the proper newId value
				$c = count($oldRestricts);
				for ($i = 0; $i < $c; $i++) {
					$oldRestrictId = $oldRestricts[$i]['restrictedId'];
					if (!array_key_exists($oldRestrictId, $idMap)) {
						// used in calculation condition
						continue;
					}
					// replace restrictedId in restrict
					$oldRestricts[$i]['restrictedId'] = $idMap[$oldRestrictId];

					// collect information to sanitize restricts in new fields
					switch ($oldRestricts[$i]['type']) {
						case 'usedAsEqualTo' :
							$newEqualToRestrict = '#field' . $oldRestricts[$i]['restrictedId'];
							break;
						case 'usedAsMinDate' :
							$newMinDateRestrict = '#field' . $oldRestricts[$i]['restrictedId'];
							break;
						case 'usedAsMaxDate' :
							$newMaxDateRestrict = '#field' . $oldRestricts[$i]['restrictedId'];
							break;
						case 'usedInCal' :
							// cal restricts is the calculation equation. we copy form with fields.
							// field names are not changes ==> equation uses field names and is therefore still valid ==> nothing to change in defaultvalue
							break;
						case 'usedAsShowWhen' :
							$newShowWhenRestricts[] = $oldRestricts[$i];
							break;
						case 'usedAsReloadTrigger' :
							$newReloadTrigger[] = 'field' . $oldRestricts[$i]['restrictedId'];
							break;
						default :
							// actually these data are invalid, we prevent them from being stored in the defaultvalue again and add a message
							unset ($oldRestricts[$i]);
							Factory::getApplication()->enqueueMessage((Text::_('COM_VISFORMS_CHECK_RESTRICTS_AFTER_BATCH_COPY')));
							break;
					}
				}
				$oldRestricts = array_values($oldRestricts);
			}

			// set and save restrictions
			self::setRestriction($oldRestricts);
			// save new showWhenrestricts
			// create strings from newShowWhenRestricts
			$newShowWhenRestricts = array_map(function ($element) {
				return 'field' . $element['restrictedId'] . '__' . $element['optionId'];
			}, $newShowWhenRestricts);
			// get old values from database
			$db = Factory::getContainer()->get(DatabaseInterface::class);
			$query = $db->createQuery();
			$query->select($db->quoteName(array('defaultvalue', 'typefield')))
				->from($db->quoteName('#__visfields'))
				->where('id = ' . $newId);
            try {
                $db->setQuery($query);
                $result = $db->loadObject();
                // extract default value
                $defaultValue = ((!empty($result)) ? VisformsHelper::registryArrayFromString($result->defaultvalue) : array());
            }
            catch (\RuntimeException $e) {
                $defaultValue = array();
            }

			// reset or remove value in default value
			if (!empty($newShowWhenRestricts)) {
				$defaultValue['f_' . $result->typefield . '_showWhen'] = $newShowWhenRestricts;
			}
			else {
			    // remove option
				unset($defaultValue['f_' . $result->typefield . '_showWhen']);
			}
			if (!empty($newEqualToRestrict)) {
				$defaultValue['f_' . $result->typefield . '_validate_equalTo'] = $newEqualToRestrict;
			}
			else {
			    if (isset($defaultValue['f_' . $result->typefield . '_validate_equalTo'])) {
                    // reset to default
                    $defaultValue['f_' . $result->typefield . '_validate_equalTo'] = "0";
                }
			}
			if (!empty($newMinDateRestrict)) {
				$defaultValue['f_' . $result->typefield . '_minvalidation_type'] = $newMinDateRestrict;
			}
            // Else: Do nothing, keep existing value
			if (!empty($newMaxDateRestrict)) {
				$defaultValue['f_' . $result->typefield . '_maxvalidation_type'] = $newMaxDateRestrict;
			}
            // Else: Do nothing, keep existing value
			if (!empty($newReloadTrigger)) {
				$defaultValue['f_' . $result->typefield . '_reload'] = $newReloadTrigger;
			}
			else {
                if (isset($defaultValue['f_' . $result->typefield . '_reload'])) {
                    // reset to default
                    $defaultValue['f_' . $result->typefield . '_reload'] = "";
                }
			}

			// parse default value as string
			$defaultValue = VisformsHelper::registryStringFromArray($defaultValue);
			// update database
			self::saveDefaultValue($newId, $defaultValue);
		}
	}

	public static function removeRestrictsValues($defaultValue, $fieldName, $msg = true, $register = true, $excludes = array()) {
		if (!empty($register)) {
			$defaultValue = VisformsHelper::registryArrayFromString($defaultValue);
		}

		foreach ($defaultValue as $dfName => $dfValue) {
			if ((strpos($dfName, '_validate_equalTo') > 0) && (str_starts_with($dfValue, '#field'))) {
				$defaultValue[$dfName] = '';
				if ($msg) {
					Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_RESTRICTS_RESET', Text::_('COM_VISFORMS_EQUAL_TO'), $fieldName), 'warning');
				}
			}
			if ((strpos($dfName, '_minvalidation_type') > 0) && (str_starts_with($dfValue, '#field'))) {
				$defaultValue[$dfName] = '';
				if ($msg) {
					Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_RESTRICTS_RESET', Text::_('COM_VISFORMS_MIN_DATE_VALIDATION_TYPE'), $fieldName), 'warning');
				}
			}
			if ((strpos($dfName, '_maxvalidation_type') > 0) && (str_starts_with($dfValue, '#field'))) {
				$defaultValue[$dfName] = '';
				if ($msg) {
					Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_RESTRICTS_RESET', Text::_('COM_VISFORMS_MAX_DATE_VALIDATION_TYPE'), $fieldName), 'warning');
				}
			}
			if ((strpos($dfName, '_showWhen') > 0) && is_array($dfValue)) {
				$defaultValue[$dfName] = '';
				if ($msg) {
					Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_RESTRICTS_RESET', Text::_('COM_VISFORMS_SHOW_WHEN'), $fieldName), 'warning');
				}
			}
			// clear value in equation of field of type calculation
			if ((strpos($dfName, '_equation') > 0) && (!empty($dfValue))) {
				$defaultValue[$dfName] = '';
				if ($msg) {
					Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_RESTRICTS_RESET', Text::_('COM_VISFORMS_FIELD_CALCULATION'), $fieldName), 'warning');
				}
			}
			if (!in_array('_reload', $excludes) && (strpos($dfName, '_reload') > 0) && is_array($dfValue)) {
				$defaultValue[$dfName] = '';
				if ($msg) {
					Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_RESTRICTS_RESET', Text::_('COM_VISFORMS_RELOAD_WHEN'), $fieldName), 'warning');
				}
			}
		}
		if (isset($defaultValue) && is_array($defaultValue) && (!empty($register))) {
			$defaultValue = VisformsHelper::registryStringFromArray($defaultValue);
		}
		return $defaultValue;
	}
}