<?php
/**
 * @author       Aicha Vack
 * @package      Joomla.Administrator
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2021 vi-solutions
 * @since        Joomla 1.6
 */

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Version;
use Joomla\CMS\Language\Text;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\Filesystem\Path;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Installer\Installer;
use Joomla\CMS\Date\Date;
use Joomla\Database\DatabaseInterface;
use Visolutions\Component\Visforms\Administrator\Helper\AefHelper;

class com_visformsInstallerScript {
	// common members for all installation script files
	private $release;
	private $oldRelease;
	private $minimum_joomla_release = 6;
	private $maximum_joomla_release = 6;
	private $min_visforms_version;
	private $name;
	private $loggerName;
	// special members

	private $versionsWithPostflightFunction;
	private $forms;
	private $db;

	// construction

	public function __construct($adapter) {
		$this->initializeLogger($adapter);
		$this->addLogEntry('****** script file loaded: com_visforms component ******', Log::INFO);
		$this->db = Factory::getContainer()->get(DatabaseInterface::class);
	}

	// interface

	public function preflight($route, $adapter) {
		$this->addLogEntry('start preflight', Log::INFO);

		$this->name = $adapter->getManifest()->name;
		$this->release = $adapter->getManifest()->version;
		$this->oldRelease = "";
		$this->minimum_joomla_release = $adapter->getManifest()->attributes()->version;
		$this->min_visforms_version = $adapter->getManifest()->vfminversion;
		$max_downgrade_version = (!empty($this->getLastCompatibleVersion())) ? $this->getLastCompatibleVersion() : $this->release;

		// list all updates with special post flight functions here
		$this->versionsWithPostflightFunction = array('5.2.0', '6.0.1');
		$this->forms = $this->getForms();
		$jversion = new Version;
		$date = new Date('now');
		$app = Factory::getApplication();
		$this->addLogEntry('*** start ' . $route . ' of extension ' . $this->name . ' ' . $this->release . ': ' . $date . ' ***', Log::INFO);

		// abort if system requirements are not met
		if ($route != 'uninstall') {
            if ($jversion::MAJOR_VERSION < $this->minimum_joomla_release) {
                $msg = Text::_('COM_VISFORMS_WRONG_JOOMLA_VERSION') . $this->minimum_joomla_release;
                $app->enqueueMessage($msg, 'ERROR');
                $this->addLogEntry($msg, Log::ERROR);
                return false;
            }
			if ($jversion::MAJOR_VERSION > $this->maximum_joomla_release) {
                $msg = Text::sprintf('COM_VISFORMS_WRONG_MAX_JOOMLA_VERSION', $this->maximum_joomla_release);
				$app->enqueueMessage($msg, 'ERROR');
				$this->addLogEntry($msg, Log::ERROR);
				return false;
			}

			// abort if the component being installed is lower than the last downgrade-able version
			if ($route == 'update') {
			    // try to get old version from manifest_cache in #__extentsion; fall back on max_downgrad_version, stored in a visforms database table
				$this->oldRelease = ($this->getExtensionParam('version')) ?? $max_downgrade_version;
				$this->addLogEntry("installed version is: " . $this->oldRelease . " Update version is : " . $this->release, Log::INFO);
				if (version_compare($this->release, $max_downgrade_version, 'lt')) {
				    $msg = Text::sprintf('COM_VISFORMS_WRONG_VERSION_NEW', $this->oldRelease, $this->release);
					$app->enqueueMessage($msg, 'ERROR');
					$this->addLogEntry($msg, Log::ERROR);
					return false;
				}

                // abort if the installed version is to old
                if (version_compare($this->oldRelease, $this->min_visforms_version, 'lt')) {
				    $msg = Text::sprintf('COM_VISFORMS_INCOMPATIBLE_VERSION_NEW', $this->min_visforms_version, $this->oldRelease, $this->release);
	                $app->enqueueMessage($msg, 'ERROR');
                    $this->addLogEntry($msg, Log::ERROR);
                    return false;
                }

                // abort if we have a 4.0.x Version
                if (version_compare($this->oldRelease, '4.0.0', 'ge') && version_compare($this->oldRelease, '4.1.0', 'lt')) {
                    $msg = Text::sprintf('COM_VISFORMS_INCOMPATIBLE_VERSION_NEW', $this->min_visforms_version, $this->oldRelease, $this->release);
                    $app->enqueueMessage($msg, 'ERROR');
                    $this->addLogEntry($msg, Log::ERROR);
                    return false;
                }

				// set permissions for css files (which might be edited through backend and set to readonly) so they can be updated
				$files = array(
				    'joomla-alert.css',
					'jquery.searchtools.css', 'jquery.searchtools.min.css',
					'visdata.css', 'visdata.min.css',
					'visforms.bt5.css', 'visforms.bt5.min.css',
					'visforms.css', 'visforms.min.css',
					'visforms.default.css', 'visforms.default.min.css',
					'visforms.tooltip.css', 'visforms.tooltip.min.css',
					'visforms.uikit3.css', 'visforms.uikit3.min.css');
				foreach ($files as $cssfile) {
					@chmod(Path::clean(JPATH_ROOT . '/media/com_visforms/css/' . $cssfile), 0755);
				}
			}
			else {
				$this->addLogEntry("*** start Install: " . $date . " ***", Log::INFO);
				$this->addLogEntry("version is: " . $this->release, Log::INFO);
			}
		}

		if ($route == 'uninstall') {
			$language = Factory::getApplication()->getLanguage();
			$language->load('com_visforms', JPATH_ROOT  , 'en-GB', true);
			$language->load('com_visforms', JPATH_ROOT  , null, true);
		}

		$this->addLogEntry('end preflight', Log::INFO);
		return true;
	}

	public function postflight($route, $adapter) {
		$this->addLogEntry('start postflight', Log::INFO);

		if ($route == 'update') {
			// run specific component adaptation for specific update versions
			if (!empty($this->oldRelease) && (version_compare($this->oldRelease, '5.0.0', 'ge'))) {
				foreach ($this->versionsWithPostflightFunction as $versionWithDatabaseChanges) {
					if (version_compare($this->oldRelease, $versionWithDatabaseChanges, 'lt')) {
						$postFlightFunctionPostfix = str_replace('.', '_', $versionWithDatabaseChanges);
						$postFlightFunctionName = 'postFlightForVersion' . $postFlightFunctionPostfix;
						if (method_exists($this, $postFlightFunctionName)) {
							$this->$postFlightFunctionName();
						}
					}
				}
			}
			// we must check if tables are not yet converted to utf8mb4 every time, because the conversion can only be performed if the mysql engine supports utf8mb4
			$this->convertTablesToUtf8mb4();
		}

		if ($route == 'install') {
			$this->createFolder(array('images', 'visforms'));
			$this->setLastCompatibleVersion($this->release);
		}

		if ($route == 'install' || $route == 'update') {
			// set "add" parameter in forms menu item in administration
			$db = $this->db;
            $where = $db->quoteName('menutype') . ' = ' . $db->quote('main') . ' AND ' . $db->quoteName('client_id') . ' = 1 AND ' . $db->quoteName('link') . ' = ' . $db->quote('index.php?option=com_visforms&view=visforms') . ' AND ' . $db->quoteName('alias') . ' = ' . $db->quote('com-visforms-submenu-forms');
		    $this->setParams(array("menu-quicktask" => "index.php?option=com_visforms&task=visform.add"), 'menu', 'params', $where);
        }

		$this->addLogEntry('end postflight', Log::INFO);
		return true;
	}

	public function uninstall( $adapter) {
		$this->addLogEntry('start uninstall', Log::INFO);

		$this->forms = $this->getForms();

		$date = new Date('now');
		$this->addLogEntry('*** start uninstall of extension Visforms: ' . $date . ' ***', Log::INFO);

		// delete all visforms related tables in database
		$dataTables = $this->getPrefixFreeDataTableList();
		if (!empty($dataTables)) {
			$this->addLogEntry("*** trying to delete data tables ***", Log::INFO);
			foreach ($dataTables as $tn) {
			    $this->dropTable($tn);
			}
		}
		$visTables = array('#__visfields', '#__visforms', '#__visverificationcodes',
            '#__visforms_lowest_compat_version', '#__visforms_utf8_conversion', '#__visforms_spambot_attempts',
            '#__viscreator', '#__vispdf');
		foreach ($visTables as $visTable) {
		    $this->dropTable($visTable);
        }

		// delete folders in image folder
		$this->addLogEntry("*** trying to delete custom files and folders ***", Log::INFO);
		$folder = JPATH_ROOT .  '/images/visforms';
		if (file_exists($folder)) {
			try {
				$result = Folder::delete($folder);
				if ($result) {
					$this->addLogEntry("folder successfully removed: " . $folder, Log::INFO);
				}
				else {
					$this->addLogEntry('problems removing folder: ' . $folder, Log::ERROR);
				}
			}
			catch (RuntimeException $e) {
				$this->addLogEntry('problems removing folder: ' . $folder . ', ' . $e->getMessage(), Log::ERROR);
			}

		}

		// delete visuploads folder
		$folder = JPATH_ROOT . '/visuploads';
		if (file_exists($folder)) {;
			try {
				$result = Folder::delete($folder);
				// $this->status->folders[] = array('folder' => $folder, 'result' => $result[0]);
				if ($result) {
					$this->addLogEntry("folder successfully removed: " . $folder, Log::INFO);
				}
				else {
					$this->addLogEntry('problems removing folder: ' . $folder, Log::ERROR);
				}
			}
			catch (RuntimeException $e) {
				$this->addLogEntry('problems removing folder: ' . $folder . ', ' . $e->getMessage(), Log::ERROR);
			}
		}

		$this->addLogEntry('end uninstall', Log::INFO);
	}

	// implementation

	private function getExtensionParam($name, $eid = 0) {
		$db = $this->db;
		$query = $db->createQuery();
		$query->select($db->quoteName('manifest_cache'));
		$query->from($db->quoteName('#__extensions'));
		// check if an extension id is given. If yes we want a parameter from this extension
		if ($eid != 0) {
			$query->where($db->quoteName('extension_id') . ' = ' . $db->quote($eid));
		}
		else {
			// we want a parameter from component visforms
			$query->where($this->getComponentWhereStatement());
		}
		try {
			$db->setQuery($query);
			$result = $db->loadResult();
			if (empty($result)) {
				return '';
			}
			$manifest = json_decode($result, true);
			if (isset($manifest[$name])) {
				return $manifest[$name];
			}
			return '';
		}
		catch (RuntimeException $e) {
			$message = Text::sprintf('COM_VISFORMS_UNABLE_TO_GET_VALUE_OF_PARAM', $name) . " " . Text::sprintf('COM_VISFORMS_DB_FUNCTION_FAILED', $e->getMessage());
			Factory::getApplication()->enqueueMessage($message, 'warning');
			$this->addLogEntry('unable to get value of param ' . $name . ', ' . $e->getMessage(), Log::ERROR);
		}

		return false;
	}

	private function getComponentWhereStatement() {
		// create where statement to select visforms component record in #__extensions table
		$db = $this->db;
		$where = $db->quoteName('type') . ' = ' . $db->quote('component') . ' AND ' . $db->quoteName('element') . ' = ' . $db->quote('com_visforms') . ' AND ' . $db->quoteName('name') . ' = ' . $db->quote('visforms');
		return $where;
	}

	private function getForms() {
		$db = $this->db;
		$query = $db->createQuery();
		$query->select('*')
			->from($db->qn('#__visforms'));
		try {
			$db->setQuery($query);
			return $db->loadObjectList();
		}
		catch (RuntimeException $e) {
			$this->addLogEntry('unable to load form list from database: ' . $e->getMessage(), Log::INFO);
			return false;
		}
	}

	private function getLowerCaseTableList() {
		$db = $this->db;
		$tablesAllowed = $db->getTableList();
		if (!empty($tablesAllowed)) {
			return array_map('strtolower', $tablesAllowed);
		}
		else {
			return false;
		}
	}

	private function getPrefixFreeDataTableList () {
		$prefixFreeTableList = array();
		$forms = $this->forms;
		if (empty($forms)) {
			return $prefixFreeTableList;
		}
		$db = $this->db;
		$tableList = $this->getLowerCaseTableList();
		if (empty($tableList)) {
			return $prefixFreeTableList;
		}
		foreach ($forms as $form) {
			$tnfulls = array(strtolower($db->getPrefix() . "visforms_" . $form->id), strtolower($db->getPrefix() . "visforms_" . $form->id . "_save"));
			foreach ($tnfulls as $tnfull) {
				if (in_array($tnfull, $tableList)) {
					$prefixFreeTableList[] = str_replace(strtolower($db->getPrefix()), "#__", $tnfull);
				}
			}
		}

		return $prefixFreeTableList;
	}

	private function getLastCompatibleVersion() {
		$this->addLogEntry('trying to get last compatible version sequence', Log::INFO);
		$db = $this->db;
		$query = $db->createQuery();
		$query->select($db->qn('vfversion'))
			->from($db->qn('#__visforms_lowest_compat_version'));
		try {
			$db->setQuery($query);
			return $db->loadResult();
		}
		catch (Exception $e) {
			$this->addLogEntry("unable to get last compatible version sequence from db: " . $e->getMessage(), Log::INFO);
			return false;
		}
	}

	private function setLastCompatibleVersion($version) {
		$this->addLogEntry('trying to set lowest compatible version sequence.', Log::INFO);
		$db = $this->db;
		$lowestCompatVersion = $this->getLastCompatibleVersion();
		// Fix Table #__visforms_lowest_compat_version has no recordset yet (as a result of an incomplete installation due to an inclomplete script)
		if ($lowestCompatVersion === faLse || is_null($lowestCompatVersion)) {
			try {
				$db->setQuery('INSERT INTO ' . $db->quoteName('#__visforms_lowest_compat_version')
					. ' (' . $db->quoteName('vfversion') . ') VALUES (' . $db->q($version) . ')')->execute();
			}
			catch (Exception $e) {
				$this->addLogEntry("unable to create record set in table lowest compatible version sequence: " . $e->getMessage(), Log::ERROR);
			}
		}
		// update existing record
		else {
			try {
				$db->setQuery('UPDATE ' . $db->quoteName('#__visforms_lowest_compat_version')
					. ' SET ' . $db->quoteName('vfversion') . ' = ' . $db->q($version))->execute();
			} catch (Exception $e) {
				$this->addLogEntry("unable to set lowest compatible version sequence from db: " . $e->getMessage(), Log::ERROR);
			}
		}
	}

	private function setParams($param_array, $table, $fieldName, $where = "", $key = 'id') {
		if (count($param_array) > 0) {
			$this->addLogEntry("*** trying to add params to table: #__" . $table . " ***", Log::INFO);
			$db = $this->db;
			$query = $db->createQuery();
			$query
				->select($db->quoteName(array($key, $fieldName)))
				->from($db->quoteName('#__' . $table));
			if ($where != "") {
				$query->where($where);
			}
			$results = new stdClass();
			try {
				$db->setQuery($query);
				$results = $db->loadObjectList();
				$this->addLogEntry(count($results) . ' recordsets to process', Log::INFO);
			}
			catch (RuntimeException $e) {
				$this->addLogEntry('unable to load param fields, ' . $e->getMessage(), Log::ERROR);
			}
			if ($results) {
				foreach ($results as $result) {
					$params = json_decode($result->$fieldName, true);
					// add the new variable(s) to the existing one(s)
					foreach ($param_array as $name => $value) {
						$params[(string)$name] = (string)$value;
					}
					// store the combined new and existing values back as a JSON string
					$paramsString = json_encode($params);
					try {
                        $query = $db->createQuery();
                        $query->update($db->quoteName('#__' . $table))
                            ->set($db->quoteName($fieldName) . ' = ' . $db->quote($paramsString))
                            ->where(array($db->quoteName($key) . ' = ' . $result->$key));
                        $db->setQuery($query);
						$db->execute();
						$this->addLogEntry("params successfully added", Log::INFO);
					}
					catch (RuntimeException $e) {
						$this->addLogEntry('problems with adding params ' . $e->getMessage(), Log::ERROR);
					}
				}
			}
		}
	}

	// miscellaneous

	private function dropTable($table) {
		$db = $this->db;
		try {
			$db->setQuery("drop table if exists $table");
			$db->execute();
			$this->addLogEntry('Table dropped: ' . $table, Log::INFO);
		}
		catch (RuntimeException $e) {
			$this->addLogEntry('unable to drop table: '.$table.', ' . $e->getMessage(), Log::ERROR);
		}
    }

    private function postFlightForVersion5_2_0() {
        $this->addLogEntry('*** Perform postflight for Version 5.2.0 ***', Log::INFO);
        $this->addLogEntry("*** Try to update email field options  ***", Log::INFO);
        $db = $this->db;
        $query = $db->createQuery();
        $query
            ->select($db->quoteName(array('id', 'defaultvalue')))
            ->from($db->quoteName('#__visfields'))
            ->where($db->quoteName('typefield') . ' = ' . $db->q('email'));
        $results = new stdClass();
        try {
            $db->setQuery($query);
            $results = $db->loadObjectList();
            $this->addLogEntry(count($results) . ' recordsets to process', Log::INFO);
        }
        catch (RuntimeException $e) {
            $this->addLogEntry('Unable to update email field options, ' . $e->getMessage(), Log::ERROR);
        }
        if ($results) {
            foreach ($results as $result) {
                $params = json_decode($result->defaultvalue, true);
                if (!empty(AefHelper::checkAEF())) {
                    $params['f_email_mailexists_nevv'] = "1";
                }
                if (isset($params['f_email_fillwith']) && $params['f_email_fillwith'] === 'inival') {
                    $params['f_email_fillwith'] = '0';
                }
                // store the combined new and existing values back as a JSON string
                $paramsString = json_encode($params);
                try {
                    $query = $db->createQuery();
                    $query
                        ->update($db->quoteName('#__visfields'))
                        ->set($db->quoteName('defaultvalue') . ' = ' . $db->quote($paramsString))
                        ->where(array($db->quoteName('id') . ' = ' . $result->id));
                    $db->setQuery($query);
                    $db->execute();
                    $this->addLogEntry(" Email field options updated for field with id " . $result->id, Log::INFO);
                }
                catch (RuntimeException $e) {
                    $this->addLogEntry('Problems updating email field options for field with id ' . $result->id . ': ' . $e->getMessage(), Log::ERROR);
                }
            }
        }
        $this->addLogEntry("*** Try to update text field options  ***", Log::INFO);
        // Adjust option values of fillwith option
        // In hidden field uniqueid = 1, user name = 2, username = 3
        // In text field types user name = 1, username = 2
        // Convert values in text field
        $query = $db->createQuery();
        $query
            ->select($db->quoteName(array('id', 'defaultvalue')))
            ->from($db->quoteName('#__visfields'))
            ->where($db->quoteName('typefield') . ' = ' . $db->q('text'));
        $results = new stdClass();
        try {
            $db->setQuery($query);
            $results = $db->loadObjectList();
            $this->addLogEntry(count($results) . ' recordsets to process', Log::INFO);
        }
        catch (RuntimeException $e) {
            $this->addLogEntry('Unable to update text field options, ' . $e->getMessage(), Log::ERROR);
        }
        if ($results) {
            foreach ($results as $result) {
                $params = json_decode($result->defaultvalue, true);
                if (isset($params['f_text_fillwith'])) {
                    if ($params['f_text_fillwith'] === '2') {
                        $params['f_text_fillwith'] = '3';
                    }
                    else if ($params['f_text_fillwith'] === '1') {
                        $params['f_text_fillwith'] = '2';
                    }
                    else {
                        // nothing changed, no saving in database necessary
                        continue;
                    }
                }
                else {
                    // nothing changed, no saving in database necessary
                    continue;
                }
                // store the combined new and existing values back as a JSON string
                $paramsString = json_encode($params);
                try {
                    $query = $db->createQuery();
                    $query
                        ->update($db->quoteName('#__visfields'))
                        ->set($db->quoteName('defaultvalue') . ' = ' . $db->quote($paramsString))
                        ->where(array($db->quoteName('id') . ' = ' . $result->id));
                    $db->setQuery($query);
                    $db->execute();
                    $this->addLogEntry("Text field options updated for field with id " . $result->id, Log::INFO);
                }
                catch (RuntimeException $e) {
                    $this->addLogEntry('Problems updating text field options for field with id ' . $result->id . ': ' . $e->getMessage(), Log::ERROR);
                }
            }
        }
        // Move uniquevalues validation settings into defaultvalues
        $this->addLogEntry("*** Try to move uniquevalues validation settings into defaultvalues  ***", Log::INFO);
        $uvFieldTypes = array('text','email','url','tel','number','date','select','radio','checkbox','multicheckbox','password','textarea','selectsql','radiosql','multicheckboxsql');
        foreach ($uvFieldTypes as $uvFieldType) {
            $query = $db->createQuery();
            $query
                ->select($db->quoteName(array('id', 'defaultvalue', 'uniquevaluesonly')))
                ->from($db->quoteName('#__visfields'))
                ->where($db->quoteName('typefield') . ' = ' . $db->q($uvFieldType));
            $results = new stdClass();
            try {
                $db->setQuery($query);
                $results = $db->loadObjectList();
                $this->addLogEntry(count($results) . ' ' . $uvFieldType . ' recordsets to process', Log::INFO);
            }
            catch (RuntimeException $e) {
                $this->addLogEntry('Unable to move ' . $uvFieldType . ' uniquevalues validation settings into defaultvalues, ' . $e->getMessage(), Log::ERROR);
            }
            if ($results) {
                foreach ($results as $result) {
                    $params = json_decode($result->defaultvalue, true);
                    $params['f_'.$uvFieldType.'_uniquevaluesonly'] = (!empty($result->uniquevaluesonly)) ? $result->uniquevaluesonly : '0';

                    // store the combined new and existing values back as a JSON string
                    $paramsString = json_encode($params);
                    try {
                        $query = $db->createQuery();
                        $query
                            ->update($db->quoteName('#__visfields'))
                            ->set($db->quoteName('defaultvalue') . ' = ' . $db->quote($paramsString))
                            ->where(array($db->quoteName('id') . ' = ' . $result->id));
                        $db->setQuery($query);
                        $db->execute();
                        $this->addLogEntry(ucfirst($uvFieldType) . " field options updated for field with id " . $result->id, Log::INFO);
                    }
                    catch (RuntimeException $e) {
                        $this->addLogEntry('Problems updating ' . $uvFieldType . ' field options for field with id ' . $result->id . ': ' . $e->getMessage(), Log::ERROR);
                    }
                }
            }
        }
        // Move frontend display option 'show link' for upload field from 'defaultvalues' to root
        $this->addLogEntry("*** Try to move frontend data view link settings for upload fields  ***", Log::INFO);
        $query = $db->createQuery();
        $query
            ->select($db->quoteName(array('id', 'defaultvalue')))
            ->from($db->quoteName('#__visfields'))
            ->where($db->quoteName('typefield') . ' = ' . $db->q('file'));
        $results = new stdClass();
        try {
            $db->setQuery($query);
            $results = $db->loadObjectList();
            $this->addLogEntry(count($results) . ' recordsets to process', Log::INFO);
        }
        catch (RuntimeException $e) {
            $this->addLogEntry('Unable to move frontend data view link settings for upload fields, ' . $e->getMessage(), Log::ERROR);
        }
        if ($results) {
            foreach ($results as $result) {
                $params = json_decode($result->defaultvalue, true);
                $result->showlink = (isset($params['f_file_showlink']) && !empty((int) $params['f_file_showlink'])) ? (int) $params['f_file_showlink'] : 0;
                unset($params['f_file_showlink']);
                if (isset($params['f_file_link_text_type'])) {
                   $result->link_text_type = (int) $params['f_file_link_text_type'];
                   unset($params['f_file_link_text_type']);
                }
                if (isset($params['f_file_link_text'])) {
                    $result->link_text = $params['f_file_link_text'];
                    unset($params['f_file_link_text']);
                }
                $result->defaultvalue = json_encode($params);
                try {
                    $db->updateObject('#__visfields', $result, 'id');
                    $this->addLogEntry("Frontend data view link settings moved for field with id " . $result->id, Log::INFO);
                }
                catch (\RuntimeException $e) {
                    $this->addLogEntry('Problems moving data view link settings for field with id ' . $result->id . ': ' . $e->getMessage(), Log::ERROR);
                }
            }
        }
        // Move frontend display option 'show link' for URL field from 'defaultvalues' to root
        $this->addLogEntry("*** Try to move frontend data view link settings for URL fields  ***", Log::INFO);
        $query = $db->createQuery();
        $query
            ->select($db->quoteName(array('id', 'defaultvalue')))
            ->from($db->quoteName('#__visfields'))
            ->where($db->quoteName('typefield') . ' = ' . $db->q('url'));
        $results = new stdClass();
        try {
            $db->setQuery($query);
            $results = $db->loadObjectList();
            $this->addLogEntry(count($results) . ' recordsets to process', Log::INFO);
        }
        catch (RuntimeException $e) {
            $this->addLogEntry('Unable to move frontend data view link settings for URL fields, ' . $e->getMessage(), Log::ERROR);
        }
        if ($results) {
            foreach ($results as $result) {
                $params = json_decode($result->defaultvalue, true);
                $result->urlaslink = (!empty((int) $params['f_url_urlaslink'])) ? (int) $params['f_url_urlaslink'] : 0;
                unset($params['f_url_urlaslink']);
                if (isset($params['f_url_link_text'])) {
                    $result->url_link_text = $params['f_url_link_text'];
                    unset($params['f_url_link_text']);
                }
                if (isset($params['f_url_link_pre_text'])) {
                    $result->link_pre_text = $params['f_url_link_pre_text'];
                    unset($params['f_url_link_pre_text']);
                }
                if (isset($params['f_url_link_post_text'])) {
                    $result->link_post_text = $params['f_url_link_post_text'];
                    unset($params['f_url_link_post_text']);
                }
                $result->defaultvalue = json_encode($params);
                try {
                    $db->updateObject('#__visfields', $result, 'id');
                    $this->addLogEntry("Frontend data view link settings moved for field with id " . $result->id, Log::INFO);
                }
                catch (\RuntimeException $e) {
                    $this->addLogEntry('Problems moving data view link settings for field with id ' . $result->id . ': ' . $e->getMessage(), Log::ERROR);
                }
            }
        }
        $this->dropColumns(array('uniquevaluesonly'), 'visfields');
        $this->setLastCompatibleVersion('5.2.0');
    }

	private function postFlightForVersion6_0_1() {
		$this->addLogEntry('*** Perform postflight for Version 6.0.1 ***', Log::INFO);
		// remove visforms component parameter downloadid, which is no longer supported
		// downloadid is stored in update_site_record
		$db = $this->db;
		$where = $db->quoteName('element') . ' = ' . $db->quote('com_visforms') . ' AND ' . $db->quoteName('client_id') . ' = 1 AND ' . $db->quoteName('type') . ' = ' . $db->quote('component') . ' AND ' . $db->quoteName('name') . ' = ' . $db->quote('visforms');
		$this->setParams(array("downloadid" => ""), 'extensions', 'params', $where, 'extension_id');
		$this->setLastCompatibleVersion('6.0.1');
	}

	private function createFolder($folders = array()) {
		$this->addLogEntry("*** trying to create folders ***", Log::INFO);
		// create visforms folder in image directory and copy an index.html into it
		$folder = JPATH_ROOT;
		foreach ($folders as $name) {
			$folder .= '/' . $name;
		}

		if (($folder != JPATH_ROOT) && !(file_exists($folder))) {
			try {
				$result = Folder::create($folder);
				if ($result) {
					$this->addLogEntry("folder successfully created: " . $folder, Log::INFO);
				}
				else {
					$this->addLogEntry("problems creating folder: " . $folder, Log::ERROR);
				}
			}
			catch (RuntimeException $e) {
				$this->addLogEntry("problems creating folders, " . $e->getMessage(), Log::ERROR);
			}

			$src = JPATH_ROOT . '/media/com_visforms/index.html';
			$dest = Path::clean($folder .  '/index.html');

			try {
				$result = File::copy($src, $dest);
				if ($result) {
					$this->addLogEntry("file successfully copied: " . $dest, Log::INFO);
				}
				else {
					$this->addLogEntry("problems copying file: " . $dest, Log::ERROR);
				}
			}
			catch (RuntimeException $e) {
				$this->addLogEntry("problems copying files, " . $e->getMessage(), Log::ERROR);
			}
		}
	}

	private function deleteFile($fileToDelete) {
        $oldfile = Path::clean(JPATH_ROOT . $fileToDelete);
        if (file_exists($oldfile)) {
            try {
                File::delete($oldfile);
                $this->addLogEntry($oldfile . " deleted", Log::INFO);
            }
            catch (RuntimeException $e) {
                $this->addLogEntry('unable to delete ' . $oldfile . ': ' . $e->getMessage(), Log::INFO);
            }
        }
        else {
            $this->addLogEntry($oldfile . " does not exist.", Log::INFO);
        }
    }

	private function deleteFolder($folderToDelete) {
        $folder = Path::clean(JPATH_ROOT . $folderToDelete);
        if (file_exists($folder)) {
            try {
                Folder::delete($folder);
                $this->addLogEntry($folder . "deleted", Log::INFO);
            }
            catch (RuntimeException $e) {
                $this->addLogEntry('unable to delete ' . $folder . ': ' . $e->getMessage(), Log::INFO);
            }
        }
        else {
            $this->addLogEntry($folder . " does not exist.", Log::INFO);
        }
    }

	private function addColumns($columnsToAdd = array(), $table = "visforms") {
		if (count($columnsToAdd) > 0) {
			$this->addLogEntry("*** trying to add new fields to table: #__" . $table . " ***", Log::INFO);
			$this->addLogEntry(count($columnsToAdd) . " fields to add", Log::INFO);
			$db = $this->db;
			foreach ($columnsToAdd as $columnToAdd) {
				// we need at least a column name
				if (!(isset($columnToAdd['name'])) || ($columnToAdd['name'] == "")) {
					continue;
				}
				$queryStr = ("ALTER TABLE " . $db->quoteName('#__' . $table) . "ADD COLUMN " . $db->quoteName($columnToAdd['name']) .
					((isset($columnToAdd['type']) && ($columnToAdd['type'] != "")) ? " " . $columnToAdd['type'] : " text") .
					((isset($columnToAdd['length']) && ($columnToAdd['length'] != "")) ? "(" . $columnToAdd['length'] . ")" : "") .
					((isset($columnToAdd['attribute']) && ($columnToAdd['attribute'] != "")) ? " " . $columnToAdd['attribute'] : "") .
					((isset($columnToAdd['notNull']) && ($columnToAdd['notNull'] == true)) ? " not NULL" : "") .
					((isset($columnToAdd['default']) && ($columnToAdd['default'] !== "")) ? " DEFAULT " . $db->quote($columnToAdd['default']) : " DEFAULT ''"));
				try {
					$db->setQuery($queryStr);
					$db->execute();
					$this->addLogEntry("field added: " . $columnToAdd['name'], Log::INFO);
				}
				catch (RuntimeException $e) {
					$this->addLogEntry("unable to add field: " . $columnToAdd['name'] . ', ' . $e->getMessage(), Log::ERROR);
				}
			}
		}
	}

	private function dropColumns($columnsToDrop = array(), $table = "visforms") {
		$this->addLogEntry("*** trying to drop fields from table #__" . $table . " ***", Log::INFO);
		if (count($columnsToDrop) > 0) {
			$this->addLogEntry(count($columnsToDrop) . " fields to drop", Log::INFO);
			$db = $this->db;
			foreach ($columnsToDrop as $columnToDrop) {
				$queryStr = ("ALTER TABLE " . $db->quoteName('#__' . $table) . "DROP COLUMN " . $db->quoteName($columnToDrop));
				try {
					$db->setQuery($queryStr);
					$db->execute();
					$this->addLogEntry("field successfully dropped: " . $columnToDrop, Log::INFO);
				}
				catch (RuntimeException $e) {
					$this->addLogEntry("problems dropping field: " . $columnToDrop . ', ' . $e->getMessage(), Log::ERROR);
				}
			}
		}
		else {
			$this->addLogEntry('no fields to drop', Log::INFO);
		}
	}

	private function enableExtension($extWhere) {
		$db = $this->db;
		$query = $db->createQuery();
		$query->update($db->quoteName('#__extensions'))
			->set($db->quoteName('enabled') . " = 1")
			->where($extWhere);
		try {
			$db->setQuery($query);
			$db->execute();
			$this->addLogEntry("extension successfully enabled", Log::INFO);
		}
		catch (RuntimeException $e) {
			$this->addLogEntry("unable to enable extension " . $e->getMessage(), Log::ERROR);
		}
	}

	private function convertTablesToUtf8mb4() {
	    // Joomla! will use character set utf8 as default, if utf8mb4 is not supported
        // if we have successfully converted to utf8md4, we set a flag in the database
		$db = $this->db;
		$serverType = $db->getServerType();
		if ($serverType != 'mysql') {
			return;
		}
		// as this is only performed on mysql databases, we can write standard mysql directly
		try {
			$db->setQuery('SELECT ' . $db->quoteName('converted')
				. ' FROM ' . $db->quoteName('#__visforms_utf8_conversion')
			);
			$convertedDB = $db->loadResult();
		}
		catch (Exception $e) {
			// Render the error message from the Exception object
			$this->addLogEntry("unable to run sql query: " . $e->getMessage(), Log::ERROR);
			return;
		}

		if ($db->hasUTF8mb4Support()) {
			$converted = 2;
		}
		else {
			$converted = 1;
		}

		if ($convertedDB == $converted) {
			return;
		}
		$tablelist = $db->getTableList();
		foreach ($tablelist as $table) {
			if ((str_contains($table, '_visforms')) || (str_contains($table, '_visfields')) || (strpos($table, '_viscreator') !== false) || (str_contains($table, '_vispdf'))) {
				if (!$this->runQuery('ALTER TABLE ' . $table . ' CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci')) {
				    $converted = 0;
                }
				if (!$this->runQuery('ALTER TABLE ' . $table . ' DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci')){
				    $converted = 0;
                }

			}
			if (str_contains($table, '_visverificationcode')) {
			    // table has a key on a varchar field. This may result in data loss on conversion.
                // Therefore we must drop the key, enlarge column and set the key later again.
                // Character set of key column is set to utf8mb4_bin not utf8mb4_unicode_ci
				if (!$this->runQuery('ALTER TABLE ' . $table . ' DROP KEY `idx_email`')) {
					$converted = 0;
				}
				if (!$this->runQuery('ALTER TABLE ' . $table . '  MODIFY `email` varchar(400) NOT NULL DEFAULT ""')) {
					$converted = 0;
				}
				if (!$this->runQuery('ALTER TABLE ' . $table . ' CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci')) {
					$converted = 0;
				}
				if (!$this->runQuery('ALTER TABLE ' . $table . '  MODIFY `email` varchar(400) CHARACTER SET utf8mb4 COLLATE utf8mb4_bin NOT NULL DEFAULT ""')) {
					$converted = 0;
				}
				if (!$this->runQuery('ALTER TABLE ' . $table . ' DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci')){
					$converted = 0;
				}
				if (!$this->runQuery('ALTER TABLE ' . $table . ' ADD KEY `idx_email` (`email`(100))')) {
					$converted = 0;
				}
			}
		}
        try {
	        $db->setQuery('UPDATE ' . $db->quoteName('#__visforms_utf8_conversion')
		        . ' SET ' . $db->quoteName('converted') . ' = ' . $converted . ';')->execute();
        }
        catch (Exception $e) {
	        $this->addLogEntry("unable to run sql query: " . $e->getMessage(), Log::ERROR);
        }
    }

	private function runQuery($sql) {
		$this->addLogEntry('trying to run sql query: ' . $sql, Log::INFO);
		$db = $this->db;
		$query = $sql;
		try {
			$db->setQuery($query);
			$db->execute();
			return true;
		}
		catch (Exception $e) {
			$this->addLogEntry("unable to run sql query: " . $e->getMessage(), Log::ERROR);
			return false;
		}
	}

	private function cmp($a, $b) {
		if (strlen($a) == strlen($b)) {
			return 0;
		}

		return (strlen($a) > strlen($b)) ? 1 : -1;
	}

	// logging

	private function initializeLogger($adapter) {
		$this->loggerName               = 'visforms';
		$options['format']              = "{CODE}\t{MESSAGE}";
		$options['text_entry_format']   = "{PRIORITY}\t{MESSAGE}";
		$options['text_file']           = 'visforms_update.php';
		try {
			Log::addLogger($options, Log::ALL, array($this->loggerName, 'jerror'));
		}
		catch (RuntimeException $e) {}
	}

	private function addLogEntry($message, $code = Log::ERROR) {
		try {
			Log::add($message, $code, $this->loggerName);
		}
		catch (RuntimeException $exception) {}
	}
}