<?php
/**
 * Visforms field text business class
 *
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2012 vi-solutions
 * @since        Joomla 1.6
 */
namespace  Visolutions\Component\Visforms\Site\Lib\Business;

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\String\StringHelper;
use Visolutions\Component\Visforms\Site\Lib\Message\DefaultMessage;
use Visolutions\Component\Visforms\Site\Lib\Message\MinLenMessage;
use Visolutions\Component\Visforms\Site\Lib\Message\MaxLenMessage;
use Visolutions\Component\Visforms\Site\Lib\Message\EqualToMessage;
use Visolutions\Component\Visforms\Site\Lib\Message\RequiredMessage;
use Visolutions\Component\Visforms\Administrator\Helper\SqlHelper;
use Visolutions\Component\Visforms\Site\Lib\Validation\MinNumberValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\MaxNumberValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\EqualToValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\CustomValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\NotEmptyValidation;

class TextFieldBusiness extends Business
{
	public function getFields() {
		$this->setField();
		return $this->fields;
	}

	protected function setField() {
        $this->setCustomJs();
		$this->setIsDisabled();
		if (isset($this->field->dataSource) && $this->field->dataSource == 'post') {
			$this->validatePostValue();
		}
		$this->addShowWhenForForm();
	}

	protected function validatePostValue(): void {
		// rules for text are: minlength, maxlength, equalTo, custom validation, unique field
		// update $this->field with value from $this->fields
		$this->updateField();
		$valid = true;
		// validate value from sql (option toggle_value_from_sql = 1)
		// edit view: only if option sql_process_in_edit_view is 1
		if (!empty($this->field->toggle_value_from_sql)) {
			// Edit View option sql_process_in_edit_view = 0
			if (($this->isRedisplayEditTask && empty($this->field->sql_process_in_edit_view))) {
				// nothing to do
			}
			// do not validate the value in attribute_value for disabled fields, if attribute_value and configurationDefault are exactly identical
			else if ((!empty($this->field->isDisabled) && ($this->field->attribute_value === $this->field->configurationDefault))) {
				// nothing to do
			}
			// Form View, always validate sql value
			else {
				$value = $this->field->attribute_value;
				try {
					$sqlHelper = new SqlHelper($this->field->sql, (isset($this->form->context) ? $this->form->context : ''));
					$sqlValue = $sqlHelper->getItemsFromSQL('loadResult');
				}
				catch (\Exception $e) {
					$sqlValue = '';
				}
				// validate that submitted values equals the value we expect, if we process the sql statement with all the submitted user inputs
				$validation = new EqualToValidation(array('value' => $value, 'cvalue' => $sqlValue));
				if (!$validation->validate()) {
					// invalid value
					$valid = false;
					$message = new DefaultMessage($this->field->label, $this->field->custom_php_error);
					$error = $message->getMessage();
					// attach error to form
					$this->setErrors($error);
				}
			}
		}
		// only to perform when the value is not empty
		if ($this->field->attribute_value != "") {
            // do not validate the value in attribute_value for disabled fields, if attribute_value and configurationDefault are exactly identical
            if ((!empty($this->field->isDisabled) && ($this->field->attribute_value === $this->field->configurationDefault))) {
                return;
            }
			// check for right minlength
			if ((isset($this->field->validate_minlength)) && ($this->field->validate_minlength != '')) {
				$mincount = $this->field->validate_minlength;
				$count = StringHelper::strlen($this->field->attribute_value);
				$validation = new MinNumberValidation(array('count' => $count, 'mincount' => $mincount));
                if (!$validation->validate()) {
					// invalid value
					$valid = false;
                    $message = new MinLenMessage($this->field->label, $this->field->custom_php_error, array('mincount' => $mincount));
                    $error = $message->getMessage();
					// attach error to form
					$this->setErrors($error);
				}
			}

			// check for right maxlength
			if ((isset($this->field->attribute_maxlength)) && ($this->field->attribute_maxlength != '')) {
				$maxcount = $this->field->attribute_maxlength;
				$count = StringHelper::strlen($this->field->attribute_value);
                $validation = new MaxNumberValidation(array('count' => $count, 'maxcount' => $maxcount));
                if (!$validation->validate()) {
					// invalid value
					$valid = false;
                    $message = new MaxLenMessage($this->field->label, $this->field->custom_php_error, array('maxcount' => $maxcount));
                    $error = $message->getMessage();
					// attach error to form
					$this->setErrors($error);
				}
			}

			// perform equalTo validation
			if ((isset($this->field->validate_equalTo)) && ($this->field->validate_equalTo != '0')) {
				$value = $this->field->attribute_value;
				$id = str_replace("#field", "", $this->field->validate_equalTo);

				foreach ($this->fields as $equalToField) {
					if ($equalToField->id == $id) {
					    $validation = new EqualToValidation(array('value' => $value, 'cvalue' => $equalToField->attribute_value));
                        if (!$validation->validate()) {
							// invalid value
							$valid = false;
                            $message = new EqualToMessage($this->field->label, $this->field->custom_php_error, array('equalToLabel' => $equalToField->label));
                            $error = $message->getMessage();
							// attach error to form
							$this->setErrors($error);
							break;
						}
					}
				}
			}

			// perform custom validation
			$regex = isset($this->field->customvalidation) ? "/" . $this->field->customvalidation . "/" : "";
			if ($regex != "") {
			    $validation = new CustomValidation(array('value' => $this->field->attribute_value, 'regex' => $regex));
                if (!$validation->validate()) {
					// invalid value
					$valid = false;
                    $message = new DefaultMessage($this->field->label, $this->field->custom_php_error);
                    $error = $message->getMessage();
					// attach error to form
					$this->setErrors($error);
				}
			}
			// validate unique field value in database
			$this->validateUniqueValue();
		}

		//at least one validation failed
		if (!$valid) {
			$this->field->isValid = false;
		}
	}

	public function validateRequired() {
		if (isset($this->field->dataSource) && $this->field->dataSource == 'post') {
			// check that a value is set if field is required
			if (isset($this->field->attribute_required)) {
				// only for fields that are not disabled
				if (empty($this->field->isDisabled)) {
				    $validation = new NotEmptyValidation(array('value' => $this->field->attribute_value));
					if (!$validation->validate()) {
						$this->field->isValid = false;
                        $message = new RequiredMessage($this->field->label, $this->field->custom_php_error);
                        $error = $message->getMessage();
						// attach error to form
						$this->setErrors($error);
					}
				}
			}
		}
		return $this->field;
	}

    protected function setCustomJs() {
        if (empty($this->field->toggle_value_from_sql)) {
            return true;
        }
        // no value reload, if we do not process SQL in Edit View but use stored values as input.
        if (!empty($this->field->noEditSqlProcess)) {
            return true;
        }
        // add record id as cid to event.data, needed if we are in an edit view
        $cid = 0;
        if ($this->form->displayState === $this->isEditTask || $this->form->displayState === $this->isRedisplayEditTask) {
            $cid = Factory::getApplication()->getInput()->getCmd('cid', 0);
        }
        $cidJs = ", cid: " . $cid;
        // Reload once, after form is loaded in order to set the proper value in the edit view
        $this->setOnloadReloadJs($cidJs, 'reloadValue');
        // set onchange reload event handler
        $this->reloadTriggerFields = isset($this->field->reload) ? $this->field->reload : array();
        if (empty($this->reloadTriggerFields)) {
            return true;
        }
        $this->removeDuplicateReloadTriggerFields();
        // foreach trigger field add a onchange handler which reloads the value for this field
        foreach ($this->reloadTriggerFields as $trigger) {
            foreach ($this->fields as $triggerField) {
                $this->setReloadJs($triggerField, $trigger, $cidJs, 'reloadValue');
            }
        }
    }

	/**
	 * we always use the configuration defaults as field "value" (attribute value, attribute selected, attribute checked or text in textarea)
	 * only then, we can reset the field properly
	 * we use javascript to set field "value state" (val(), prop selected, checked...) to the proper value (user input, configuration default...)
	 */
	public function setFieldValueProperties() {
		// stored (validated) "userinput" in new parameter
		$this->field->userInput = $this->getUserInputForJs();
		// set value, which is first displayed to the configuration defaults
		$this->field->attribute_value = $this->field->configurationDefault;
		// only used in business calculation if the field is disabled. Use the configuration default then.
		$this->field->calculationValue = ($this->field->configurationDefault !== '') ? $this->field->configurationDefault : ((isset($this->field->unchecked_value)) ? $this->field->unchecked_value : 0);
		return $this->field;
	}

	private function getUserInputForJs() {
		$value = $this->field->attribute_value;
		if ($this->isEditTask) {
			// use configuartion default as default, if field is disabled
			if (!empty($this->field->isDisabled)) {
				$value = $this->field->configurationDefault;
			}
		}
		if ($this->isRedisplayEditTask) {
			if (!empty($this->field->isDisabled)) {
				// if field was originally not disabled use dbValue
				$fieldsDisabledState = Factory::getApplication()->getUserState('com_visforms.fieldsDisabledState.' . $this->form->context, null);
				if (!empty($fieldsDisabledState) && (is_array($fieldsDisabledState)) && (empty($fieldsDisabledState[$this->field->name])) && isset($this->field->editValue)) {
					$value = $this->field->editValue;
				} else {
					$value = $this->field->configurationDefault;
				}
			}
		}
		if (!empty($value)) {
            $value = str_replace('@', '&#64', $value);
        }
		return $value;
	}
}