<?php
/**
 * Visforms field date class
 *
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2012 vi-solutions
 * @since        Joomla 1.6
 */
namespace  Visolutions\Component\Visforms\Site\Lib\Field;

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\User\User;
use Joomla\CMS\User\UserHelper;
use Visolutions\Component\Visforms\Administrator\Helper\AefHelper;
use Visolutions\Component\Visforms\Site\Model\VisformsModel as VisformsModelSite;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Plugin\PluginHelper;
use Visolutions\Component\Visforms\Site\Event\Visforms\VisfieldBeforeCreateEvent;
use Visolutions\Component\Visforms\Site\Lib\Message\DateMessage as ErrorMessage;
use Visolutions\Component\Visforms\Administrator\Helper\SqlHelper;
use Visolutions\Component\Visforms\Site\Lib\Validation\DateValidation;

class DateField extends TextField
{

	public function __construct($field, $form) {
		parent::__construct($field, $form);
		// store potentiall query Values for this field in the session
		$this->setQueryValue();
		$this->postValue = $this->input->post->get($field->name, '', 'STRING');
		if ($this->postValue === '0000-00-00 00:00:00') {
			$this->postValue = '';
		}
	}
	protected function setField(): void {
		// preprocessing field
		$this->extractDefaultValueParams();
		$this->extractGridSizesParams();
		$this->extractRestrictions();
		$this->mendBooleanAttribs();
        $this->prepareFieldForNoEditSqlProcess();
        $this->unsetDeselectedEditViewValidations();
		$this->cleanFieldProperties();
        $this->setCustomPhpErrorMsg();
        PluginHelper::importPlugin('visforms');
        $dispatcher = Factory::getApplication()->getDispatcher();
        $event = new VisfieldBeforeCreateEvent('onVisfieldBeforeCreate', [
            'subject' => $this->field,
            'context' => 'com_visforms.field',
            'form' => $this->form
        ]);
        $dispatcher->dispatch('onVisfieldBeforeCreate', $event);
		$this->setIsConditional();
		$this->addDateFormatsToField();
		$this->setMinDate();
		$this->setMaxDate();
		$this->fillWith();
		// just to make sure that we do not store an invalid attribute value from field configuration as reset value, because invalid date values break the form!
		$this->setInvalidAttributeValueToEmpty();
		$this->removeInvalidQueryValues();
		$this->setEditValue();
		$this->setConfigurationDefault();
		$this->setEditOnlyFieldDbValue();
		$this->setFieldDefaultValue();
		$this->setDbValue();
		$this->setRedirectParam();
		$this->setCustomJs();
		$this->setFieldsetCounter();
		$this->setEnterKeyAction();
		$this->setShowRequiredAsterix();
        $this->replaceLabelWithCustomText();
	}

	protected function setFieldDefaultValue(): void {
		$field = $this->field;
		if ($this->form->displayState === VisformsModelSite::$displayStateIsNewEditData) {
			// edit value may not set (= NULL) when field was disabled (previous Visforms versions), field was unpublished or field was not created when user inputs were stored
			// we cannot use invalid date values and just display them, because they cause a fatal error on form display
			if (isset($this->field->editValue)) {
				// invalid date values break the form, so check for valid value
                $validation = new DateValidation(array('value' => $this->field->editValue, 'format' => $this->field->dateFormatPhp));
				if ($validation->validate()) {
					$this->field->attribute_value = $this->field->editValue;
				}
			}
			// else use field configuration default value
			$this->field->dataSource = 'db';
			return;
		}
		// if we have a POST Value, we use this
		if ((count($_POST) > 0) && isset($_POST['postid']) && ($_POST['postid'] == $this->form->id)) {
			// this will create a error message on form display when user input is invalid
			$valid = $this->validateUserInput('postValue');
			// $_POST is not set if field was disabled when form was submitted
			// we cannot use invalid date values and just display them, because they cause a fatal error on form display
			if (isset($_POST[$field->name]) && ($valid === true)) {
				$this->field->attribute_value = $this->postValue;
			} // use default values
			else {
				$this->field->attribute_value = $this->field->configurationDefault;
			}
			// else don't change the configuration default value
			$this->field->dataSource = 'post';
			return;
		}

		// if we have a GET Value and field may use GET values, we uses this
		if (isset($field->allowurlparam) && ($field->allowurlparam == true)) {
			$urlparams = Factory::getApplication()->getUserState('com_visforms.urlparams.' . $this->form->context, null);
			if (!empty($urlparams) && (is_array($urlparams)) && (isset($urlparams[$this->field->name]))) {
				$queryValue = $urlparams[$this->field->name];
			}
			if (isset($queryValue)) {
				$this->field->attribute_value = $queryValue;
				$this->field->dataSource = 'query';
			}
		}
		// Nothing to do
	}

	protected function validateUserInput($inputType): bool {
		$value = $this->$inputType;
		// Some empty values are valid but 0 is not
		if ((!isset($value)) || ($value === '')) {
			return true;
		}
		// if a value is set we test it has a valid date format
        $validation = new DateValidation(array('value' => $value, 'format' => $this->field->dateFormatPhp));
		if (!$validation->validate()) {
            // invalid date format - set field->isValid to false
            $this->field->isValid = false;
            // get the Error Message
            $message = new ErrorMessage($this->field->label, $this->field->custom_php_error, array('format' => $this->field->dateFormatPhp));
            $error = $message->getMessage();
            $this->setErrorMessageInForm($error);
            // we cannot use the value!
            return false;
		}
		return true;
	}

	private function addDateFormatsToField() {
		$this->field->dateFormatPhp = '';
		$this->field->dateFormatJs = '';
		if (isset($this->field->format)) {
			// get dateformat for php and for javascript
			$dformat = explode(";", $this->field->format);
			if (count($dformat) == 2) {
				$this->field->dateFormatPhp = $dformat[0];
				$this->field->dateFormatJs = $dformat[1];
			}
		}
	}

	protected function removeInvalidQueryValues() {
		$type = $this->type;
		$app = Factory::getApplication();
		$urlparams = $app->getUserState('com_visforms.urlparams.' . $this->form->context);
		if (empty($urlparams) || !is_array($urlparams) || !isset($urlparams[$this->field->name])) {
			return;
		}
		$queryValue = $urlparams[$this->field->name];
		// empty string is a valid value (= field value is not set)
		if (($queryValue !== '')) {
		    $validation = new DateValidation(array('value' => $queryValue, 'format' => $this->field->dateFormatPhp));
			$valid = $validation->validate();
			if (empty($valid)) {
				// remove invalid queryValue ulrparams array and set urlparams to Null if the array is empty
				unset($urlparams[$this->field->name]);
				if (!(count($urlparams) > 0)) {
					$urlparams = null;
				}
				$app->setUserState('com_visforms.urlparams.' . $this->form->context, $urlparams);
			}
		}
	}

	protected function setInvalidAttributeValueToEmpty() {
	    $validation = new DateValidation(array('value' => $this->field->attribute_value, 'format' => $this->field->dateFormatPhp));
		if (!$validation->validate()) {
			$this->field->attribute_value = "";
		}
	}

	protected function fillWith(): void {
        $field = $this->field;
        if ((!isset($field->fillwith))) {
            return;
        }
        if ((($field->fillwith) !== '0')) {
            // process fillwith sql first
            if ($field->fillwith == 'sql') {
                try {
                    $sqlHelper = new SqlHelper($field->defaultvalue_sql);
                    $value = $sqlHelper->getItemsFromSQL('loadResult');
                    $this->field->attribute_value = HTMLHelper::_('date', $value, $field->dateFormatPhp);
                }
                catch (\Exception $e) {
                    $this->field->attribute_value = '';
                }
                return;
            }
            // process fill with user data
            // if we edit stored user inputs we want to use the user profile of the user who submitted the form as default fill with values
            if (($this->form->displayState === VisformsModelSite::$displayStateIsNewEditData) || ($this->form->displayState === VisformsModelSite::$displayStateIsRedisplayEditData)) {
                $data = $this->form->data;
                $userId = $data->created_by;
                if (!empty($userId)) {
                    $user = new User($userId);
                }
            }
            // use user profile of logged-in user
            else {
                $user = Factory::getApplication()->getIdentity();
                $userId = $user->id;
            }
            if ($userId != 0) {
                preg_match('/^CF(\d+)$/', $field->fillwith, $matches);
                if ($matches && !empty($matches[1])) {
                    $customfieldid = $matches[1];
                    $value = HTMLHelper::_('visforms.getCustomUserFieldValue', $customfieldid, $user);
                    // in error case, value === false
                    if ($value !== false) {
                        $this->field->attribute_value = HTMLHelper::_('date', $value, $field->dateFormatPhp);
                        return;
                    }
                }
                $userProfile = UserHelper::getProfile($userId);
                if ((!(empty($userProfile->profile))) && (is_array($userProfile->profile))) {
                    if (!(empty($userProfile->profile[$field->fillwith]))) {
                        $this->field->attribute_value = HTMLHelper::_('date', $userProfile->profile[$field->fillwith], $field->dateFormatPhp);
                        return;
                    }
                }
            }
            // no value found, set attribute_value to ''
            $this->field->attribute_value = '';
            return;
        }
        // if we have a special default value set in field declaration we use this
        if (($field->attribute_value === '') && (isset($field->daydate) && ((int) $field->daydate === 1))) {
        	$date = empty($field->daydate_shift) ? 'now' : $field->daydate_shift .' days';
            $this->field->attribute_value = HTMLHelper::_('date', $date, $field->dateFormatPhp);
        }
    }

	protected function setMinDate() {
		if (isset($this->field->minvalidation_type) && !empty($this->field->minvalidation_type)) {
			switch ($this->field->minvalidation_type) {
				case "fix" :
					// use original value of $this->field->mindate
					break;
				case "1" :
					// use (shifted) current date
					$min = empty($this->field->dynamic_min_shift) ? 'now' : $this->field->dynamic_min_shift .' days';
					$this->field->mindate = HTMLHelper::_('date', $min, $this->field->dateFormatPhp);
					break;
				default:
				    // Due to showon implementation, there might be a value stored, which is not correct
                    $this->field->mindate = "";
					break;
			}
		}
		else {
			$this->field->mindate = "";
		}
	}

	protected function setMaxDate() {
		if (isset($this->field->maxvalidation_type) && !empty($this->field->maxvalidation_type)) {
			switch ($this->field->maxvalidation_type) {
				case "fix" :
					// use original value of $this->field->maxdate
					break;
				case "1" :
					// use (shifted) current date
					$max = empty($this->field->dynamic_max_shift) ? 'now' : $this->field->dynamic_max_shift .' days';
					$this->field->maxdate = HTMLHelper::_('date', $max, $this->field->dateFormatPhp);
					break;
				default:
				    // Due to Joomla showon implementation, there might be a value stored, which is not correct
                    $this->field->maxdate = "";
					break;
			}
		}
		else {
			$this->field->maxdate = "";
		}
	}

	protected function cleanFieldProperties() {
		$hasSub = AefHelper::checkAEF();
		if (empty($hasSub)) {
			unset($this->field->mindate);
			unset($this->field->maxdate);
			unset($this->field->daydate_shift);
			unset($this->field->dynamic_min_shift);
			unset($this->field->dynamic_max_shift);
			if (isset($this->field->minvalidation_type) && (str_contains($this->field->minvalidation_type, '#field'))) {
				$this->field->minvalidation_type = '';
			}
			if (isset($this->field->maxvalidation_type) && (str_contains($this->field->maxvalidation_type, '#field'))) {
				$this->field->maxvalidation_type = '';
			}
		}
	}

    protected function unsetDeselectedEditViewValidations(): void {
	    parent::unsetDeselectedEditViewValidations();
        if ($this->isEditOrSaveEditTask ) {
            if (isset($this->field->minvalidation_type) && !empty($this->field->minvalidation_nevv)) {
                $this->field->minvalidation_type = '';
            }
            if (isset($this->field->maxvalidation_type) && !empty($this->field->maxvalidation_nevv)) {
                $this->field->maxvalidation_type = '';
            }
        }
    }
}