<?php
/**
 * Component for Spambotcheck plugin
 * @author       Ingmar Vack
 * @package      Joomla.Administrator
 * @subpackage   com_spambotcheck
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2021 vi-solutions
 * @since        Joomla 4.0
 */
use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Filesystem\Path;
use Joomla\CMS\Version;

// no direct access
defined('_JEXEC') or die;

class pkg_spambotcheckInstallerScript {
	private string $loggerName;
	private string $versionInstalled;
	private string $versionToInstall;
	private string $versionMinimumJoomla;

	// construction

	public function __construct($adapter) {
		$this->initializeLogger($adapter);
		$this->addLogEntry('*** starting package script ***', Log::INFO);
		// return value is null if no prior package version is installed
		$this->versionInstalled = (string) $this->getExtensionParameter('manifest_cache', 'version', 'package', 'pkg_spambotcheck');
		$this->addLogEntry('package version installed = ' . ($this->versionInstalled ?: 'not installed'), Log::INFO);
		if ($adapter->getManifest()) {
			$this->versionToInstall = (string) $adapter->getManifest()->version;
			$this->addLogEntry('package version in manifest = ' . ($this->versionToInstall ?: 'package manifest version not found'), Log::INFO);
			$this->versionMinimumJoomla = (string) $adapter->getManifest()->attributes()->version;
		}
	}

	// interface

	public function preflight($route, $adapter): bool {
		if($route !== 'uninstall') {
			$jversion = new Version();
			// abort if the current Joomla release is older
			if (isset($this->versionMinimumJoomla) && version_compare($jversion->getShortVersion(), $this->versionMinimumJoomla, 'lt')) {
				$text = Text::_('PKG_USER_SPAMBOTCHECK_WRONG_JOOMLA_VERSION') . $this->versionMinimumJoomla;
				$app  = Factory::getApplication();
				$app->enqueueMessage($text, 'warning');
				$this->addLogEntry($text);
				return false;
			}
		}

		// move/rename old files
		if($route == 'update') {
			$this->moveOldFiles();
			$this->deleteOldFiles();
		}

		return true;
	}

	public function postflight($route, $adapter): bool {
		if ($route == 'install' || $route == 'update') {
			$manifest = $adapter->getParent()->manifest;
			$packages = $manifest->xpath('files/file');
			if (!empty($packages)) {
				$this->deleteUpdateSites($packages);
			}
		}
		return true;
	}

	public function install($adapter): bool {
		return true;
	}

	public function uninstall($adapter): bool {
		return true;
	}

	public function update($adapter): bool {
		return true;
	}

	// implementation

	// files and directories

	function moveOldFiles() {
		$this->addLogEntry('*** try to move old files ***', Log::INFO);
		// rename view files with wrong casing
		$moves = array(
			'/administrator/components/com_spambotcheck/src/View/Help/HTMLView.php' => '/administrator/components/com_spambotcheck/src/View/Help/HtmlView.php',
			'/administrator/components/com_spambotcheck/src/View/Logs/HTMLView.php' => '/administrator/components/com_spambotcheck/src/View/Logs/HtmlView.php',
			'/administrator/components/com_spambotcheck/src/View/Users/HTMLView.php' => '/administrator/components/com_spambotcheck/src/View/Users/HtmlView.php',
		);
		$this->moveFiles($moves);
	}

	private function deleteOldFiles() {
		// list all files and folder which have to be removed on update
		$files = array(
			'/administrator/components/com_spambotcheck/src/View/Users/index.html',
			//'/administrator/components/com_spambotcheck/src/View/Help/HTMLView.php',
			//'/administrator/components/com_spambotcheck/src/View/Logs/HTMLView.php',
			//'/administrator/components/com_spambotcheck/src/View/Users/HTMLView.php',
		);
		$folders = array(
			'/administrator/components/com_visforms/helpers/html',
			'/administrator/components/com_visforms/controllers',
			'/administrator/components/com_visforms/Controller',
			'/administrator/components/com_visforms/Extension',
			'/administrator/components/com_visforms/Field',
			'/administrator/components/com_visforms/models',
			'/administrator/components/com_visforms/Model',
			'/administrator/components/com_visforms/Service',
			'/administrator/components/com_visforms/tables',
			'/administrator/components/com_visforms/Table',
			'/administrator/components/com_visforms/views',
			'/administrator/components/com_visforms/View',
			'/administrator/components/com_visforms/lib/placeholder',
			'/components/com_visforms/controllers',
			'/components/com_visforms/Controller',
			'/components/com_visforms/Field',
			'/components/com_visforms/helpers/route',
			'/components/com_visforms/models',
			'/components/com_visforms/Model',
			'/components/com_visforms/Service',
			'/components/com_visforms/views',
			'/components/com_visforms/View',
			'/modules/mod_visforms/helper',
			'/modules/mod_visforms/Helper',
		);
		$this->addLogEntry('*** try to delete old files and folders ***', Log::INFO);
		$this->deleteFiles($files);
		$this->deleteFolders($folders);
	}

	private function deleteFiles($files = array()) {
		foreach ($files as $file) {
			$oldFile = Path::clean(JPATH_ROOT . $file);
			if (File::exists($oldFile)) {
				try {
					File::delete($oldFile);
					$this->addLogEntry($oldFile . ' successfully deleted', Log::INFO);
				}
				catch (RuntimeException $e) {
					$this->addLogEntry('Deleting ' . $oldFile . ' failed: ' . $e->getMessage(), Log::INFO);
				}
			}
		}
	}

	private function deleteFolders($folders = array()) {
		foreach ($folders as $folder) {
			$oldFolder = Path::clean(JPATH_ROOT . $folder);
			if (Folder::exists($oldFolder)) {
				try {
					Folder::delete($oldFolder);
					$this->addLogEntry($oldFolder . ' successfully deleted', Log::INFO);
				}
				catch (RuntimeException $e) {
					$this->addLogEntry('Deleting ' . $oldFolder . ' failed: ' . $e->getMessage(), Log::INFO);
				}
			}
		}
	}

	private function moveFiles($moves) {
		foreach ($moves as $fromFile => $toFile) {
			$from   = Path::clean(JPATH_ROOT . $fromFile);
			$to     = Path::clean(JPATH_ROOT . $toFile);
			if (File::exists($from)) {
				try {
					File::move($from, $to);
					$this->addLogEntry("$from successfully moved to $to", Log::INFO);
				}
				catch (RuntimeException $e) {
					$this->addLogEntry("failed moving $from to $to: " . $e->getMessage(), Log::INFO);
				}
			}
		}
	}

	// miscellaneous

	function getExtensionParameter($field = 'params', $name = null, $type = 'plugin', $element = 'spambotcheck') {
		// return value is null if no prior package version is installed (found)
		$db     = Factory::getDbo();
		$query  = $db->getQuery(true);
		$query->select($db->qn($field))
			->from($db->qn('#__extensions'))
			->where($db->qn('type') . ' = ' . $db->q($type) . ' AND ' . $db->qn('element') . ' = ' . $db->q($element));
		$db->setQuery($query);
		try {
			$params = json_decode($db->loadResult(), true);
			if (isset($name)) {
				return $params[$name];
			}
			return $params;
		}
		catch (RuntimeException $e) {
			$this->addLogEntry("unable to get element '$element' of type '$type' own '$field' parameter " . ($name ?? '') . ' from database', JLog::ERROR);
		}
	}

	private function deleteUpdateSites($packages) {
		$db = Factory::getDbo();
		// remove upload site information for all extensions from database
		foreach ($packages as $package) {
			$type = (string) $package->attributes()->type;
			$name = (string) $package->attributes()->id;
			$group = (!empty($package->attributes()->group)) ? (string) $package->attributes()->group : '';
			$id = $this->getExtensionId($type, $name, $group, 0);
			if (!empty($id)) {
				$update_site_ids = $this->getUpdateSites($id);
				if (!empty($update_site_ids)) {
					$update_sites_ids_a = implode(',', $update_site_ids);
					$query = $db->getQuery(true);
					$query->delete($db->quoteName('#__update_sites'));
					$query->where($db->quoteName('update_site_id') . ' IN (' . $update_sites_ids_a . ')');
					try {
						$db->setQuery($query);
						$db->execute();
					}
					catch (RuntimeException $e) {
						$this->addLogEntry("Problems deleting record sets in #__update_sites : " . $e->getMessage(), Log::INFO);
					}
					$query = $db->getQuery(true);
					$query->delete($db->quoteName('#__update_sites_extensions'));
					$query->where($db->quoteName('extension_id') . ' = ' . $id);
					try {
						$db->setQuery($query);
						$db->execute();
					}
					catch (RuntimeException $e) {
						$this->addLogEntry("Problems deleting record sets in #__update_sites_extensions : " . $e->getMessage(), Log::INFO);
					}
				}
			}
		}
	}

	private function getExtensionId($type, $name, $group = '', $client_id = 0) {
		$db = Factory::getDbo();
		$where = $db->quoteName('type') . ' = ' . $db->quote($type) . ' AND ' . $db->quoteName('element') . ' = ' . $db->quote($name);
		$query = $db->getQuery(true)
			->select($db->quoteName('extension_id'))
			->from($db->quoteName('#__extensions'))
			->where($where);
		try {
			$db->setQuery($query);
			$id = $db->loadResult();
		}
		catch (RuntimeException $e) {
			$this->addLogEntry('Unable to get extension_id: ' . $name . ', ' . $e->getMessage(), Log::INFO);
			return false;
		}
		return $id;
	}

	private function getUpdateSites($extension) {
		$db = Factory::getDbo();
		$query = $db->getQuery(true)
			->select($db->quoteName('update_site_id'))
			->from($db->quoteName('#__update_sites_extensions'))
			->where($db->quoteName('extension_id') . ' = ' . $extension);
		try {
			$db->setQuery($query);
			$update_site_ids = $db->loadColumn();
		}
		catch (RuntimeException $e) {
			$this->addLogEntry('Unable to get update sites id: ' . $extension . ', ' . $e->getMessage(), Log::INFO);
			return false;
		}
		return $update_site_ids;
	}

	// logging

	private function initializeLogger($adapter) {
		$this->loggerName = (string) $adapter->getManifest()->loggerName;
		$options['format']              = "{CODE}\t{MESSAGE}";
		$options['text_entry_format']   = "{PRIORITY}\t{MESSAGE}";
		$options['text_file']           = 'spambotcheck_update.php';
		try {
			Log::addLogger($options, Log::ALL, array($this->loggerName, 'jerror'));
		}
		catch (RuntimeException $e) {}
	}

	private function addLogEntry($message, $level = Log::ERROR) {
		try {
			Log::add($message, $level, $this->loggerName);
		}
		catch (RuntimeException $exception)
		{
			// prevent installation routine from failing due to problems with logger
		}
	}
}