<?php /** @noinspection MissingSinceTagDocInspection */
/**
 * SpambotCheckImpl for Spambotcheck plugin
 *
 * @author		 vi-solutions, Aicha Vack & Ingmar Vack
 * @package		 User SpambotCheck - check for possible spambots during register and login
 * @subpackage   com_spambotcheck
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2021 vi-solutions
 * @since        Joomla 4.0
 */
namespace Visolutions\Plugin\User\Spambotcheck\Helper;

use Joomla\Registry\Registry;

// no direct access
defined('_JEXEC') or die;

class SpambotCheckProvider {
    // string SPAMBOT_FALSE (not a spambot), SPAMBOT_TRUE (spambot) + description text
    public string $identifierTag    = '';
    // to access the plugin parameter
    private Registry $params;
    // original values
    private string $orgEmail        = '';
    private string $orgIP           = '';
    private string $orgUsername     = '';
    // fields to work with
    private string $email           = '';
    private string $IP              = '';
    private string $username        = '';
    private string $reverseIP       = '';
    private bool $isXMLAvailable    = false;

    /**
    * Constructor
    * @access	public
    * @param   Registry     plugin parameters
    * @param   string       email to check
    * @param   string       IP to check
    * @param   string       username to check
    *
    * @return	void
    */
    public function __construct(&$params, $email, $IP, $username) {
        // to access the plugin parameter
        $this->params       = $params;
        // original values
        $this->orgEmail     = $email;
        $this->orgIP        = $IP;
        $this->orgUsername  = $username;
        // fields to work with
        $this->email        = $email;
        $this->IP           = $IP;
        $this->username     = $username;
        // all DNSBL take a reverse(ed) IP
        $this->reverseIP    = implode('.', array_reverse(explode('.', $IP)));
        // As of PHP 5.0, the SimpleXML functions are part of the PHP core. 
        // There is no installation needed to use these functions.
        // (https://www.w3schools.com/PHP/php_xml_simplexml.asp)
        $this->isXMLAvailable = (phpversion() > '5' && class_exists('SimpleXMLElement') == true);
    }
    
    /**
     * This method checks the field identifierTag if a spammer identification is made (is or is not a spammer).
     *
     * @access	private
     * @return	boolean
     *
     */
    private function isIdentified(): bool {
        if (str_contains($this->identifierTag, 'SPAMBOT_TRUE') || str_contains($this->identifierTag, 'SPAMBOT_FALSE')) {
            return true;
        }
        
        return false;
    }
    
    /**
     * This method dispatches calls to 
     * - prepare the parameters
     * - check against SpambotChek own listings
     * - check against online providers
     * 
     * The final result is saved in field sIdentifierTag.
     *
     * @access	private
     * @return	void
     */
    public function checkOwnListingsAndSpambotProvider() {
        // file_get_contents function and cURL are available ?
        if (!SpambotCheckHelper::isCUrlAvailable() && !function_exists('file_get_contents')) {
            $this->identifierTag = 'SPAMBOT_FALSE';
            return;
        }
        
        // set to empty string if invalid or if not to check, clean username
        $this->checkUserDataFormatsAndActions();
		if ($this->isIdentified()) {
            return;
        }
        
        // check against SpambotCheck own listings (black and white, mail and IP)
        $this->checkOwnListings();
		if ($this->isIdentified()) {
            return;
        }
        
        // this is it: check against the online providers
        $this->checkSpambotProviders();
    }

    /**
     * This method prepares the parameters, checks for valid input formats and cleans the username string
     *
     * @access	private
     * @return	void
     */
    private function checkUserDataFormatsAndActions() {
        // set to empty string if it is not to check
        if (!$this->params->get('spbot_check_email', 1)) {
            $this->email = '';
        }
        if (!$this->params->get('spbot_check_ip', 1)) {
            $this->IP = '';
        }
        if (!$this->params->get('spbot_username', 0)) {
            $this->username = '';
        }

        // set to empty string if invalid
        $this->email = SpambotCheckHelper::isValidEmail($this->email);
        $this->IP    = SpambotCheckHelper::isValidIP($this->IP);
        
        // clean username
        $this->username = SpambotCheckHelper::cleanUsername($this->username);

        if ($this->email == '' && $this->IP == '' && $this->username == '') {
            // nothing to check
            $this->identifierTag = 'SPAMBOT_FALSE';
        }
    }
    
    /**
     * This method dispatches calls to 
     * - check against SpambotChek own IP listing
     * - check against SpambotChek own Email whitelist
     * - check against SpambotChek own Email blacklist
     *
     * @access	private
     * @return	void
     */
    private function checkOwnListings() {
        // check against SpambotCheck own listings (black and white, mail and IP)
        
        // check against own IP listing white
        $this->checkOwnListingsIPWhite();
        if ($this->isIdentified()) {
            return;
        }

        // check against own IP listing black
        $this->checkOwnListingsIPBlack();
        if ($this->isIdentified()) {
            return;
        }
        
        // check against own Email whitelist
        $this->checkOwnListingsEMailWhite();
        if ($this->isIdentified()) {
            return;
        }

        // check against own Email blacklist
        $this->checkOwnListingsEMailBlack();
    }
    
    /**
     * This method checks against SpambotChek own IP white listing.
     *
     * @access	private
     * @return	void
     */
    private function checkOwnListingsIPWhite() {
        $IPs = $this->params->get('spbot_whitelist_ip', '');
        if ($this->IP != '' && $IPs != '' && (str_contains($IPs, $this->IP))) {
            // IP in whitelist
            $this->identifierTag = 'SPAMBOT_FALSE';
        }
    }

    /**
     * This method checks against SpambotChek own IP black listing.
     *
     * @access	private
     * @return	void
     */
    private function checkOwnListingsIPBlack() {
        $IPs = $this->params->get('spbot_blacklist_ip', '');
        if ($this->IP != '' && $IPs != '' && (str_contains($IPs, $this->IP))) {
	        // IP in blacklist
	        $spbot_bl_log_to_db = $this->params->get('spbot_bl_log_to_db', 0);
	        if ($spbot_bl_log_to_db) {
		        $this->params->set('spbot_log_to_db', 1);
		        SpambotCheckHelper::logSpammerToDB($this->orgEmail, $this->orgIP, $this->orgUsername, 'Blacklist-IP', '', '', '', $this->params);
	        }
	        $this->identifierTag = 'IP in Backlist' . ' SPAMBOT_TRUE';
        }
    }
    
    /**
     * This method checks against SpambotChek own Email whitelist.
     *
     * @access	private
     * @return	void
     */
    private function checkOwnListingsEMailWhite() {
        // read plugin parameter settings
        $whitelist = $this->params->get('spbot_whitelist_email', '');

        // clean and check email whitelist
        if (($whitelist = SpambotCheckHelper::cleanEMailWhitelist($whitelist)) == '') {
            return;
        }
        
        // generic whitelist
        if ($this->params->get('allow_generic_email_check', false)) {
            // split all whitelist emails into an array
            $whitelist = explode(',', $whitelist);
            // check if whitelist entry is a valid email domain-port (@mail.com)
            $count = count($whitelist);
            for ($i = 0; $i < $count; $i++) {
                $regex = '/\@(([a-zA-Z0-9\-])+\.)+([a-zA-Z0-9]{2,4})+$/';
                if (preg_match($regex, $whitelist[$i])) {
                    if (str_contains($this->email, $whitelist[$i])) {
                        // email in generic whitelist
                        $this->identifierTag = 'SPAMBOT_FALSE';
                    }
                }
            }
        }
        // whitelist
        else if (($this->email != '') && (str_contains($whitelist, $this->email))) {
            // email in whitelist
            $this->identifierTag = 'SPAMBOT_FALSE';
        }
    }
    
    /**
     * This method checks against SpambotChek own Email blacklist.
     *
     * @access	private
     * @return	void
     */
    private function checkOwnListingsEMailBlack() {
        // process email blacklists
        if ($this->email == '') {
            return;
        }
        
        // read plugin parameter settings
        $emailBlacklist = $this->params->get('spbot_blacklist_email', '');

        // clean and check email blacklist
        if (($emailBlacklist = SpambotCheckHelper::cleanEMailBlacklist($emailBlacklist)) == '') {
            return;
        }
        
        // blacklist emails into an array
        $emailBlacklist = explode(',', $emailBlacklist);
        $count = count($emailBlacklist);
        // for each blacklist entry
        for ($i = 0; $i < $count; $i++) {
            // check if valid email domain-port (@mail.com)
            $regex = '/\@(([a-zA-Z0-9\-])+\.)+([a-zA-Z0-9]{2,4})+$/';
            if (preg_match($regex, $emailBlacklist[$i])) {
                // valid email domain port
                if (str_contains($this->email, $emailBlacklist[$i])) {
	                // email in blacklist
	                $spbot_bl_log_to_db = $this->params->get('spbot_bl_log_to_db', 0);
	                if ($spbot_bl_log_to_db) {
	                    $this->params->set('spbot_log_to_db', 1);
	                    SpambotCheckHelper::logSpammerToDB($this->orgEmail, $this->orgIP, $this->orgUsername, 'Blacklist-Email', '', '', '', $this->params);
                    }
                    $this->identifierTag = 'E-Mail in Backlist' . ' SPAMBOT_TRUE';
                }
            }
        }
    }
    
    /**
     * This method dispatches calls to all online spambot sites as configured.
     *
     * @access	private
     * @return	void
     */
    private function checkSpambotProviders() {
        //store some information about spammer in plugin parameter
        $this->params->set('isSpamIp', 0);

        // StopForumSpam
        $this->checkSpambotProvider_StopForumSpam();
		if ($this->isIdentified()) {
            return;
        }
        
        // finished if no IP
        if ($this->IP == '') {
            $this->identifierTag = 'SPAMBOT_FALSE';
            return;
        }
        
        // now check all DNSBL (domain name service black-list)
        // 
        // ProjectHoneyPot
        $this->checkSpambotProvider_ProjectHoneyPot();
		if ($this->isIdentified()) {
            return;
        }
        
        // checkSpambotProvider_Sorbs
        $this->checkSpambotProvider_Sorbs();
		if ($this->isIdentified()) {
            return;
        }
        
        // SpamCop.net
        $this->checkSpambotProvider_SpamCop();
		if ($this->isIdentified()) {
            return;
        }
        
        // not a spambot
        $this->identifierTag = 'SPAMBOT_FALSE';
    }

	/**
	 * This method checks against the provider StopForumSpam.
	 *
	 * @access    private
	 * @return    void
	 * @throws \Exception
	 */
    private function checkSpambotProvider_StopForumSpam() {
        // we need the xml parser to work with the API call results
        if ($this->isXMLAvailable == false) {
            return;
        }
        
        // provider enabled ?
        if (($this->params->get('spbot_stopforumspam', 0)) != 1) {
            return;
        }
        
        $maxAllowedFrequency = $this->params->get('spbot_stopforumspam_max_allowed_frequency', 0);
        
        // build URL
        $URL = 'https://www.stopforumspam.com/api?';
        if ($this->email != '') {
            $URL .= 'email=' . $this->email . "&";
        }
        if ($this->IP != '') {
            $URL .= 'ip=' . $this->IP . "&";
        }
        if ($this->username != '') {
            $URL .= 'username=' . $this->username . "&";
        }
        // remove last '&'
        $URL = substr($URL, 0, -1);
        
        // call URL & check result 
        $rawReturn = SpambotCheckHelper::getURL($URL);
        if (str_contains($rawReturn, 'rate limit exceeded')) {
            // Added due to SFS introducing a query limit
            // https://www.stopforumspam.com/forum/t573-Rate-Limiting
            // this isn't really a spammer - we add the entry nevertheless to track the limit-issue of StopForumSpam
            SpambotCheckHelper::logSpammerToDB($this->orgEmail, $this->orgIP, $this->orgUsername, 'StopForumSpam', $URL, $rawReturn, 'rate limit exceeded', $this->params);
            
            return;
        }
        
        if (!str_starts_with($rawReturn, '<')) {
            return;
        }
            
        // Read the result into a SimpleXML and investigate it
        $element = new \SimpleXMLElement($rawReturn);
        $parsedReturn = '';

        // At least one issue (email, ip, username) should be reported more than MinReportFrequency
        $frequencies = array();
        foreach ($element->frequency as $frequency) {
            $frequencies[] = (int) $frequency;
        }
        
        // $bStopforumspam_MaxAllowedFrequency reached? -> we have a spambot
        if (max($frequencies) >= $maxAllowedFrequency) {
            $isMail = false;
            $isIP = false;
            $isUsername = false;
            $count = 0;
            foreach ($element->type as $type) {
                switch ((string) $type) {
                    case 'email':
                        if ($element->appears[$count] == 'yes') {
                            $isMail = TRUE;
                            $parsedReturn .= 'EMail: frequency=' . $element->frequency[$count] . ', last_seen=' . $element->lastseen . '; ';
                        }
                        break;
                    case 'ip':
                        if ($element->appears[$count] == "yes") {
                            $isIP = TRUE;
                            $parsedReturn .= 'IP: frequency=' . $element->frequency[$count] . ', last_seen=' . $element->lastseen . '; ';
                            $this->params->set('isSpamIp', 1);
                        }
                        break;
                    case 'username':
                        if ($element->appears[$count] == "yes") {
                            $isUsername = TRUE;
                            $parsedReturn .= 'UserName: frequency=' . $element->frequency[$count] . ', last_seen=' . $element->lastseen . '; ';
                        }
                        break;
                }
                $count = $count + 1;
            }

            if ($isMail || $isIP || $isUsername) {
                SpambotCheckHelper::logSpammerToDB($this->orgEmail, $this->orgIP, $this->orgUsername, 'StopForumSpam', $URL, $rawReturn, $parsedReturn, $this->params);
                $this->identifierTag = 'StopForumSpam (' . $parsedReturn . ') SPAMBOT_TRUE';
            }
        }
    }
    
     /**
     * This method checks against the provider ProjectHoneyPot.
     * The final result is saved in field sIdentifierTag.
     *
     * @access	private
     * @return	void
     */
    private function checkSpambotProvider_ProjectHoneyPot() {
        // provider enabled ?
        if (($this->params->get('spbot_projecthoneypot', 0)) != 1) {
            return;
        }
        
        // Project Honey Pot API key is available
        // note: this key is required if you wish to query Projecthoneypot.org
        $apiKey = $this->params->get('spbot_projecthoneypot_api_key', '');
        if (strlen($apiKey) != 12) {
            return;
        }
        
        $maxAllowedThreatScore = $this->params->get('spbot_projecthoneypot_max_allowed_threat_rating', 0);
        $lookup = $apiKey . '.' . $this->reverseIP . '.dnsbl.httpbl.org.';
        $lookupResult = gethostbyname($lookup);
        
        if ($lookup == $lookupResult) {
            return;
        }
        
        $parts = explode('.', $lookupResult);
        $days          = $parts[1];
        $threatScore   = $parts[2];
        $visitorType   = $parts[3]; // let's see what PHP says about this IP

        if ($threatScore < $maxAllowedThreatScore) {
            return;
        }
        
        $isSpammer = true;
        switch ($visitorType) {
            case "0":
                $visitorType = "Search Engine";
                $isSpammer = false;
                break;
            case "1":
                $visitorType = "Suspicious";
                if ($threatScore < 25) {
                    $isSpammer = false;
                }
                break;
            case "2":
                $visitorType = "Harvester";
                $isSpammer = true;
                break;
            case "3":
                $visitorType = "Suspicious &amp; Harvester";
                $isSpammer = true;
                break;
            case "4":
                $visitorType = "Comment Spammer";
                $isSpammer = true;
                break;
            case "5":
                $visitorType = "Suspicious &amp; Comment Spammer";
                $isSpammer = true;
                break;
            case "6":
                $visitorType = "Harvester &amp; Comment Spammer";
                $isSpammer = true;
                break;
            case "7":
                $visitorType = "Suspicious &amp; Harvester &amp; Comment Spammer";
                $isSpammer = true;
                break;
        }
        
        if ($isSpammer == true) {
            $this->params->set('isSpamIp', 1);
            $parsedResponse = 'VisitorType=' . $visitorType . ', ThreatScore=' . $threatScore . ', DaysSinceLastActivity=' . $days . '';
            SpambotCheckHelper::logSpammerToDB($this->orgEmail, $this->orgIP, $this->orgUsername, 'ProjectHoneyPot', $lookup, $lookupResult, $parsedResponse, $this->params);
            $this->identifierTag = 'ProjectHoneyPot (RawResponse=' . $lookupResult . ', ' . $parsedResponse . ')' . ' SPAMBOT_TRUE';
        }
    }
    
     /**
     * This method checks against the provider SORBS.
     * The final result is saved in field sIdentifierTag.
     *
     * @access	private
     * @return	void
     */
    private function checkSpambotProvider_Sorbs() {
        // provider enabled ?
        if (($this->params->get('spbot_sorbs', 0)) != 1) {
            return;
        }
        
        $lookup = $this->reverseIP . '.l1.spews.dnsbl.sorbs.net.';
        $lookupResult = gethostbyname($lookup);
        if ($lookup != $lookupResult) {
            SpambotCheckHelper::logSpammerToDB($this->orgEmail, $this->orgIP, $this->orgUsername, 'Sorbs', $lookup, $lookupResult, '', $this->params);
            $this->identifierTag = 'Sorbs (RawResponse=' . $lookupResult . ') SPAMBOT_TRUE';
        }

        $lookup = $this->reverseIP . '.problems.dnsbl.sorbs.net.';
        $lookupResult = gethostbyname($lookup);
        if ($lookup != $lookupResult) {
            $this->params->set('isSpamIp', 1);
            SpambotCheckHelper::logSpammerToDB($this->orgEmail, $this->orgIP, $this->orgUsername, 'Sorbs', $lookup, $lookupResult, '', $this->params);
            $this->identifierTag = 'Sorbs (RawResponse=' . $lookupResult . ') SPAMBOT_TRUE';
        }
    }
    
     /**
     * This method checks against the provider SpamCop.
     * The final result is saved in field sIdentifierTag.
     * 
     * @access	private
     * @return	void
     */
    private function checkSpambotProvider_SpamCop() {
        // provider enabled?
        if (($this->params->get('spbot_spamcop', 0)) != 1) {
            return;
        }
        
        $lookup = $this->reverseIP . '.bl.spamcop.net.';
        $lookupResult = gethostbyname($lookup);
        if ($lookupResult == '127.0.0.2') {
            $this->params->set('isSpamIp', 1);
            SpambotCheckHelper::logSpammerToDB($this->orgEmail, $this->orgIP, $this->orgUsername, 'SpamCop', $lookup, $lookupResult, '', $this->params);
            $this->identifierTag = 'SpamCop (RawResponse=' . $lookupResult . ') SPAMBOT_TRUE';
        }
    }
}