<?php
/**
 * Users model for Spambotcheck
 *
 * @author       Ingmar Vack
 * @package      Joomla.Administrator
 * @subpackage   com_spambotcheck
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2021 vi-solutions
 * @since        Joomla 4.0
 */
namespace Visolutions\Component\Spambotcheck\Administrator\Model;

use DateTimeZone;
use Joomla\CMS\Access\Access;
use Joomla\CMS\Factory;
use Joomla\CMS\Date\Date;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Language\Text;

use Visolutions\Plugin\User\Spambotcheck\Helper\SpambotCheckHelper;

// no direct access
defined('_JEXEC') or die;

class UsersModel extends ListModel {
	public function __construct($config = array(), MVCFactoryInterface $factory = null) {
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = array(
				'id', 'a.id',
				'user_id', 'a.user_id',
				'ip', 'a.ip',
				'hits', 'a.hits',
				'suspicious', 'a.suspicious',
				'trust', 'a.trust',
				'note', 'a.note',
				'name', 'b.name',
				'username', 'b.username',
				'email', 'b.email',
				'range', 'b.registerDate',
				'block', 'b.block',
				'activation', 'b.activation',
				'groupname', 'ug.title',
			);
		}

		parent::__construct($config, $factory);
	}

	protected function populateState($ordering = null, $direction = null) {
		// todo: change list ordering to SPC table fields (was simply copied from VF)
		$search = $this->getUserStateFromRequest($this->context.'.filter.search', 'filter_search');
		$this->setState('filter.search', $search);
		
		$trust = $this->getUserStateFromRequest($this->context.'.filter.trust', 'filter_trust', '');
		$this->setState('filter.trust', $trust);

		$suspicious = $this->getUserStateFromRequest($this->context.'.filter.suspicious', 'filter_suspicious', '');
		$this->setState('filter.suspicious', $suspicious);
		
		$block = $this->getUserStateFromRequest($this->context.'.filter.block', 'filter_block', '');
		$this->setState('filter.block', $block);
		
		$activation = $this->getUserStateFromRequest($this->context.'.filter.activation', 'filter_activation', '');
		$this->setState('filter.activation', $activation);
		
		$range = $this->getUserStateFromRequest($this->context.'.filter.range', 'filter_range');
		$this->setState('filter.range', $range);

		// list state information
		parent::populateState('a.id', 'asc');
	}

	protected function getStoreId($id = '') {
		// compile the store id
		$id	.= ':'.$this->getState('filter.search');
		$id	.= ':'.$this->getState('filter.suspicious');
		$id	.= ':'.$this->getState('filter.trust');
		$id	.= ':'.$this->getState('filter.block');
		$id	.= ':'.$this->getState('filter.activation');
		$id .= ':'.$this->getState('filter.range');

		return parent::getStoreId($id);
	}


	protected function getListQuery() {
		// create a new query object
		$db		= $this->getDbo();
		$query	= $db->getQuery(true);

		// select the required fields from the table
		$query->select(
			$this->getState(
				'list.select',
				'a.*, b.id as bid, b.name as name, b.email as email, b.username as username, b.registerDate as registerdate, b.block as block, b.activation as activation, ' .
				'm.user_id as muser_id, m.group_id as mgroup_id, ug.id as ugid, ug.title as groupname'
			)
		);
		$query->from('#__user_spambotcheck AS a');
		$query->join('LEFT', '#__users AS b ON b.id=a.user_id');
		$query->join('LEFT', '#__user_usergroup_map AS m ON m.user_id=b.id');
		$query->join('LEFT', '#__usergroups AS ug ON ug.id=m.group_id');
		$query->group('a.id,a.user_id,a.ip,a.hits,a.suspicious,a.trust,a.note,b.name,b.username,b.email,b.block,b.registerDate,b.activation');

		// filter by trust state
		$trust = $this->getState('filter.trust');
		if (is_numeric($trust)) {
			$query->where('a.trust = ' . (int) $trust);
		}
		
		// filter by suspicious state
		$suspicious = $this->getState('filter.suspicious');
		if (is_numeric($suspicious)) {
			$query->where('a.suspicious = ' . (int) $suspicious);
		}
		
		// filter by block state
		$block = $this->getState('filter.block');
		if (is_numeric($block)) {
			$query->where('b.block = ' . (int) $block);
		}

		// if the model is set to check the activated state, add to the query
		$active = $this->getState('filter.activation');
		if (is_numeric($active)) {
			if ($active == '0') {
				// might be '' or '0'
				$query->where($query->length('b.activation').' != 32');
			}
			elseif ($active == '1') {
				$query->where($query->length('b.activation').' = 32');
			}
		}

		// filter by search in text fields
		$search = $this->getState('filter.search');
		if (!empty($search)) {
			$search = $db->quote('%'.$db->escape($search, true).'%');
			$query->where('('. $db->qn('a.ip') . ' LIKE '.$search.' OR ' . $db->qn('b.username') .' LIKE '.$search.' OR ' . $db->qn('b.email') . ' LIKE '.$search.' OR ' . $db->qn('b.name') .' LIKE '.$search.')');
		}

		// apply the range filter
		$range = $this->getState('filter.range');
		if ($range != '' && $range != '*') {
			SpambotCheckHelper::addRangeToQuery('b.registerDate', $range, $db, $query);
		}

		// add the list ordering clause
		$orderCol	= $this->state->get('list.ordering', 'a.id');
		$orderDir	= $this->state->get('list.direction', 'asc');
		$query->order($db->escape($orderCol.' '.$orderDir));

		return $query;
	}
	
	public function trust(&$pks, $value = 0) {
        $table = $this->getTable();
        $pks = (array) $pks;

		 // attempt to change the state of the records
        if (!$table->trust($pks, $value)) {
            $this->setError($table->getError());
            return false;
        }

        // clear the component's cache
        $this->cleanCache();
 
        return true;
	}
	
	public function delete(&$pks) {
		$pks        = (array) $pks;
		$table      = $this->getTable();
		$userTable  = $this->getTable('User', 'JTable');
		$user	    = Factory::getUser();

		// check if I am a Super Admin
		$iAmSuperAdmin	= $user->authorise('core.admin');

		if (in_array($user->id, $pks)) {
			$this->setError(Text::_('COM_USERS_USERS_ERROR_CANNOT_DELETE_SELF'));
			return false;
		}

		// iterate the items to delete each one
		foreach ($pks as $i => $pk) {
			if ($table->load($pk)) {
				// access checks
				$allow = $user->authorise('core.delete', 'com_users');
				// don't allow non-super-admin to delete a super admin
				$allow = (!$iAmSuperAdmin && Access::check($pk, 'core.admin', 'com_users')) ? false : $allow;
			
				$userIp = SpambotCheckHelper::getTableFieldValue('#__user_spambotcheck', 'ip', 'user_id', $pk);
				if ($allow) {
					if (!$table->delete($pk)) {
						$this->setError($table->getError());
						return false;
					}
				}
				else {
					// prune items that you can't change
					unset($pks[$i]);
					// todo: verify this new error handling
					// JError::raiseWarning(403, \JText::_('JERROR_CORE_DELETE_NOT_PERMITTED'));
					Factory::getApplication()->enqueueMessage(Text::_('JERROR_CORE_DELETE_NOT_PERMITTED'));
					break;
				}
			}
			else {
				$this->setError($table->getError());
				return false;
			}
			
			// clean up user_spambotcheck fields
			SpambotCheckHelper::cleanUserSpambotTable($userIp, $pk);
			
			if ($userTable->load($pk)) {
				if (!$userTable->delete($pk)) {
					$this->setError($userTable->getError());
					return false;
				}
			}
			else {
				$this->setError($userTable->getError());
				return false;
			}
		}

		// clear the component's cache
		$this->cleanCache();

		return true;
	}

	public function getItems() {
		// get a storage key
		$store = $this->getStoreId();

		// try to load the data from internal storage
		if (empty($this->cache[$store])) {
			$items = parent::getItems();

			// bail out on an error or empty list
			if (empty($items)) {
				$this->cache[$store] = $items;

				return $items;
			}

			// joining the groups with the main query is a performance hog
			// find the information only on the result set

			// first pass: get list of the user id's and reset the counts
			$userIds = array();
			foreach ($items as $item) {
				$userIds[] = (int) $item->user_id;
				$item->group_count = 0;
				$item->group_names = '';
			}

			// get the counts from the database only for the users in the list
			$db     = $this->getDbo();
			$query  = $db->getQuery(true);

			// Join over the group mapping table.
			$query
				->select(array($db->qn('map.user_id'), 'COUNT('. $db->qn('map.group_id'). ') AS '. $db->qn('group_count')))
				->from($db->qn('#__user_usergroup_map') . ' AS '. $db->qn('map'))
				->where($db->qn('map.user_id') .' IN ('.implode(',', $userIds).')')
				->group($db->qn('map.user_id'))
				// join over the user groups table
				->join('LEFT', $db->qn('#__usergroups') .' AS ' . $db->qn('g2') .  'ON '. $db->qn('g2.id') . ' = '. $db->qn('map.group_id'));
			$db->setQuery($query);

			// load the counts into an array indexed on the user id field
			try {
				$userGroups = $db->loadObjectList('user_id');
			}
			catch (\RuntimeException $e) {
				return false;
			}

			// second pass: collect the group counts into the master items array
			foreach ($items as &$item) {
				if (isset($userGroups[$item->user_id])) {
					$item->group_count = $userGroups[$item->user_id]->group_count;
					// group_concat in other databases is not supported
					$item->group_names = $this->getUserDisplayedGroups($item->user_id);
				}
			}

			// add the items to the internal cache
			$this->cache[$store] = $items;
		}

		return $this->cache[$store];
	}
	
	private function getUserDisplayedGroups($user_id) {
		$db     = Factory::getDbo();
		$query  = $db->getQuery(true);
		$query
			->select($db->qn('title'))
			->from($db->qn('#__usergroups') . ' AS ' . $db->qn('ug'))
			->join('LEFT', $db->qn('#__user_usergroup_map') . ' AS ' . $db->qn('map') . ' ON' . $db->qn('ug.id') . ' = ' . $db->qn('map.group_id'))
			->where($db->qn('map.user_id') . ' = ' .$user_id);

		try {
			// SQL errors
			$db->setQuery($query);
			// DBMS errors
			$result = $db->loadColumn();
			return implode("\n", $result);
		}
		catch (\RuntimeException $e) {
			return '';
		}
	}
}