<?php
/**
 * Spambotcheck plugin
 *
 * @author		 vi-solutions, Aicha Vack & Ingmar Vack
 * @package		 User SpambotCheck - check for possible spambots during register and login
 * @subpackage   com_spambotcheck
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2021 vi-solutions
 * @since        Joomla 4.0
 */

namespace Visolutions\Plugin\User\Spambotcheck\Extension;

use Joomla\Database\DatabaseAwareTrait;
use stdClass;
use RuntimeException;

use Joomla\CMS\Application\CMSWebApplicationInterface;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Router\Route;
use Joomla\Utilities\ArrayHelper;
use Visolutions\Plugin\User\Spambotcheck\Helper\SpambotCheckHelper;
use Visolutions\Plugin\User\Spambotcheck\Helper\SpambotCheckProvider;

use Joomla\Event\DispatcherInterface;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Event\User\LoginEvent;
use Joomla\CMS\Event\User\BeforeSaveEvent;
use Joomla\CMS\Event\User\AfterSaveEvent;
use Joomla\CMS\Event\User\AfterDeleteEvent;
use Joomla\CMS\Event\Privacy\CollectCapabilitiesEvent;
use Joomla\CMS\Mail\MailerFactoryInterface;
use Joomla\CMS\Mail\MailHelper;

// no direct access
defined('_JEXEC') or die;

class Spambotcheck extends CMSPlugin implements SubscriberInterface {

	use DatabaseAwareTrait;
	protected bool $componentInstalled;

    public function __construct(DispatcherInterface $subject, $config) {
        parent::__construct($subject, $config);
        // load the translation
        $this->loadLanguage();
        $this->componentInstalled = SpambotCheckHelper::checkComponentInstalled();
    }

	public static function getSubscribedEvents(): array {
		// return an array of events this subscriber will listen to
		return [
			'onUserBeforeSave' => 'onUserBeforeSave',
			'onUserAfterSave' => 'onUserAfterSave',
			'onUserAfterDelete' => 'onUserAfterDelete',
			'onUserLogin' => 'onUserLogin',
			'onPrivacyCollectAdminCapabilities' => 'onPrivacyCollectAdminCapabilities',
		];
	}

	public function onUserBeforeSave(BeforeSaveEvent $event): bool {
		$isNew = $event->getIsNew();

        // only site and not administrator
        // only a new user
		/** @var $app CMSWebApplicationInterface */
		$app = Factory::getApplication();
		if ( !$app->isClient('site') || !$isNew) {
            return true;
        }

        $input = $app->getInput();
        $this->params->set('current_action', 'Register');
		$data = $event->getData();
        $user = array(
	        "fullname"  => $data['name'] ?? $input->post->get('string', 'name', ''),
	        "username"  => $data['username'] ?? $input->post->get('string', 'username', ''),
	        "email"     => $data['email1'] ?? $input->post->get('string', 'email1', '')
        );
        $spamString = "";
        
        if ( !$this->isSpammer($user, $spamString)) {
            // not a spammer
            return true;
        }
        
        // check if users have lately registered with this IP
        if ($this->params->get('isSpamIp', 0) == 1) {
            if ($this->componentInstalled) {
                SpambotCheckHelper::flagUserWithSpamUserIp();
            }
        }
        
        // send email notification to all sys-admins
        $this->sendMailToAdmin($user, $spamString, Text::_('PLG_USER_SPAMBOTCHECK_EMAIL_SUBJECT_REGISTER_PREVENTION_TXT'));
        
        // redirect us to the old page and display an error notification to the user
        $message = Text::_('PLG_USER_SPAMBOTCHECK_USER_REGISTRATION_SPAM_TXT');
        Log::add($message, Log::ERROR, 'jerror');
        $app->redirect('index.php');
        $app->close();
        
        return false;
    }

	public function onUserAfterSave(AfterSaveEvent $event): bool {
		$user = $event->getUser();
		$isNew = $event->getIsNew();
		$result = $event->getSavingResult();

		$userId = ArrayHelper::getValue($user, 'id', 0, 'int');
        // only for new users that were saved successfully in database
        if ($userId && $isNew && $result) {
            if ($this->componentInstalled) {
                // always insert the new user into users_spambot table, even after backend creation
                if (!SpambotCheckHelper::logUserData($userId)) {
                    // no message
                    return false;
                }
	            if (Factory::getApplication()->isClient('site')) {
		            // check if a user has already registered using the same IP
		            SpambotCheckHelper::checkIpSuspicious($user, $this->params);
		            // check for suspicious email addresses
		            SpambotCheckHelper::checkEmailSuspicious($user);
	            }
            }
        }

        return true;
    }

	public function onUserAfterDelete(AfterDeleteEvent $event): bool {
		$user = $event->getUser();
		$success = $event->getDeletingResult();
        $userId = ArrayHelper::getValue($user, 'id', 0, 'int');
        if ($userId && $success) {
            if ($this->componentInstalled) {
                // get and remember IP of deleted user
                $userIp = SpambotCheckHelper::getTableFieldValue('#__user_spambotcheck', 'ip', 'user_id', $userId);
                // delete row in table user_spambotcheck
                $db = $this->getDatabase();
                $query = $db->createQuery();
                $conditions = array(
                    $db->quoteName('user_id') . ' = ' . $db->quote($userId)
                );

	            /** @noinspection SqlResolve */
                $query->delete($db->quoteName('#__user_spambotcheck'));
                $query->where($conditions);

	            try {
	                $db->setQuery($query);
	                $db->execute();
                }
                catch (RuntimeException $e) {}

                // clean up user_spambotcheck fields
                SpambotCheckHelper::cleanUserSpambotTable($userIp, $userId);
            }
        }
        
        return true;
    }

    public function onUserLogin(LoginEvent $event): bool {
		// todo: why is the IP not logged when user already exist
		// todo: is the IP logged when user tries to register the first time or logs in the first time
	    $user = $event->getAuthenticationResponse();
		$options = $event->getOptions();
        // don't monitor log-ins
        if (!($this->params->get('spbot_monitor_events', 'RL') == 'RL')) {
            return true;
        }

        // is user trusted and not to check?
        $userId = SpambotCheckHelper::getTableFieldValue('#__users', 'id', 'email', $user['email']);
        if ($this->componentInstalled && SpambotCheckHelper::getTableFieldValue('#__user_spambotcheck', 'trust', 'user_id', $userId) == 1) {
            return true;
        }
        
        $this->params->set('current_action', 'Login');
        $spamString = "";
        
        // not a spammer ?
        if ( !$this->isSpammer($user, $spamString)) {
            return true;
        }

        // this is a spammer
        if (($spamString != "") && (!str_contains($spamString, 'E-Mail in Backlist')) && (!str_contains($spamString, 'IP in Backlist'))) {
            // set user to suspicious if not already done
            // get value of note field
	        if ($this->componentInstalled) {
		        $noteValue = SpambotCheckHelper::getTableFieldValue('#__user_spambotcheck', 'note', 'user_id', $userId);
		        if (!str_contains($noteValue, '4: User flagged; ')) {
			        $note = '4: User flagged; ';
			        // create an object for the record we are going to update.
			        $object = new stdClass();
			        $object->user_id = $userId;
			        // add a note
			        $object->note = $noteValue . $note;
			        // set suspicious state
			        $object->suspicious = 0;
			        // update their details in the users table using user_id as the primary key.
			        try {
				        $this->getDatabase()->updateObject('#__user_spambotcheck', $object, array('user_id'));
			        }
			        catch (RuntimeException $e) {}
		        }
		        // check if users have lately registered with this IP
		        if ($this->params->get('isSpamIp', 0) == 1) {
			        SpambotCheckHelper::flagUserWithSpamUserIp($userId);
		        }
	        }
        }

        // user is already logged in by task done in plgUserJoomla::onUserLogin
        // enforce a logout operation by resetting fields in session table to a guest user
		SpambotCheckHelper::updateSessionAndSessionRecord();

        // send email notification to all sys-admins
        if (($spamString != "") && (!str_contains($spamString, 'E-Mail in Backlist'))) {
            $this->sendMailToAdmin($user, $spamString, Text::_('PLG_USER_SPAMBOTCHECK_EMAIL_SUBJECT_LOGIN_PREVENTION_TXT'));
        }

        // redirect us to the old page and display an error notification to the user
        Log::add(Text::_('PLG_USER_SPAMBOTCHECK_USER_LOGIN_SPAM_TXT'), Log::ERROR, 'jerror');
		/** @var CMSWebApplicationInterface $app */
	    $app = Factory::getApplication();
	    $app->redirect(Route::_($options['return']));
        $app->close();

        return false;
    }

	public function onPrivacyCollectAdminCapabilities(CollectCapabilitiesEvent $event): array {
		$this->loadLanguage();
		return array(
			Text::_('PLG_USER_SPAMBOTCHECK_PRIVACY') => array(
				Text::_('PLG_USER_SPAMBOTCHECK_PRIVACY_INFORMATION'),
			)
		);
	}

	// implementation

	private function isSpammer($user, &$spamString): bool {
        // don't check admins
        if (SpambotCheckHelper::userIsAdmin($user)) {
            return false;
        }

        // check for spammer
        $SpambotCheck = new SpambotCheckProvider($this->params, $user['email'], SpambotCheckHelper::getIP(), $user['username']);
        $SpambotCheck->checkOwnListingsAndSpambotProvider();
        if ($SpambotCheck->identifierTag == false || strlen($SpambotCheck->identifierTag) == 0 || !str_contains($SpambotCheck->identifierTag, "SPAMBOT_TRUE")) {
            // not a spammer
            $spamString = "";
            return false;
        }

        // if we get here we have to deal with a spammer
        $spamString = $SpambotCheck->identifierTag;

		return true;
    }

	private function sendMailToAdmin(&$user, &$spamString, $subjectAddString) {
        if (!$this->params->get('spbot_email_notifications', 1)) {
            // -> no admin notifications
            return;
        }

        // get Super User Groups
        $superUserGroups = SpambotCheckHelper::getSuperUserGroups();
        if (!(count($superUserGroups) > 0)) {
            // something went wrong with finding superadmins: don't sent mails to everybody
            return;
        }

	    $db = $this->getDatabase();
        // only send notifications for selected types
        $type = $this->params->get('current_action');
        $notificationType = $this->params->get('email_notification_type');

        if (($notificationType == "RL") || ($notificationType == "R" && $type == "Register") || ($notificationType == "L" && $type == "Login")) {
            $name       = $user['fullname'];
            $username   = $user['username'];
            $email      = $user['email'];
            $sPostersIP = SpambotCheckHelper::getIP();

	        $app = Factory::getApplication();
	        $siteName = $app->getCfg('sitename');
	        $mailFrom = $app->getCfg('mailfrom');
	        $fromName = $app->getCfg('fromname');

	        // get all super administrator
            // create where statement for SQL
            $where  = "";
            $length = count($superUserGroups);
	        for ($i = 0; $i < $length; $i++) {
	            $where .= $db->qn('map.group_id') .' = ' . $superUserGroups[$i];
	            if ($i < $length - 1) {
	                $where .= ' OR ';
	            }
	        }

	        $query = $db->createQuery();
	        /** @noinspection SqlResolve */
	        $query
	            ->select(array($db->qn('u.name') . ' AS ' . $db->qn('name'), $db->qn('u.email') . 'AS' . $db->qn('email'), $db->qn('u.sendEmail') . 'AS' . $db->qn('sendEmail')))
	            ->from($db->qn('#__users'))
	            ->join('LEFT', $db->qn('#__user_usergroup_map') . ' AS ' . $db->qn('map') . ' ON ('.  $db->qn('map.user_id') . ' =' . $db->qn('u.id') .')' )
	            ->join('LEFT', $db->qn('#__usergroups') . ' AS ' . $db->qn('g') . ' ON ('.  $db->qn('map.group_id') . ' =' . $db->qn('g.id') .')' )
                ->where($where);
	        try {
	            $db->setQuery($query);
	            $rows = $db->loadObjectList();
            }
            catch (RuntimeException $e) {
            	return;
            }

            // send notification to all administrators
            $subject = sprintf(Text::_('PLG_USER_SPAMBOTCHECK_ACCOUNT_DETAILS_FOR_TXT'), $name, $siteName) . $subjectAddString;
            $subject = html_entity_decode($subject, ENT_QUOTES);
            foreach ($rows as $row) {
                if ($row->sendEmail) {
                    $message = sprintf(Text::_('PLG_USER_SPAMBOTCHECK_SEND_EMAIL_TO_ADMIN_TXT'), $row->name, $siteName, $type, $name, $email, $username, $sPostersIP, $spamString);
                    $message = html_entity_decode($message, ENT_QUOTES);
                    $mailer = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
                    // clean the email data
                    $subject = MailHelper::cleanSubject($subject);
                    $message = MailHelper::cleanBody($message);
                    $mailer->sendMail($mailFrom, $fromName, $row->email, $subject, $message);
                }
            }
        }
        else {
            // no admin notification
        }
    }
}