<?php
/**
 * Spambotcheck plugin
 *
 * @author		 vi-solutions, Aicha Vack & Ingmar Vack
 * @package		 User SpambotCheck - check for possible spambots during register and login
 * @subpackage   com_spambotcheck
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2021 vi-solutions
 * @since        Joomla 4.0
 */
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Version;

// no direct access
defined('_JEXEC') or die;

class PlguserspambotcheckInstallerScript {
	private string $loggerName;
	private string $versionInstalled;
	private string $versionToInstall;
	private string $versionMinimumJoomla;
	private stdClass $status;

    // construction

	public function __construct($adapter) {
		$this->initializeLogger($adapter);
		// return value is null if no prior package version is installed
		$this->versionInstalled = (string) $this->getExtensionParameter('manifest_cache', 'version');
		$this->addLogEntry('plugin version installed = ' . ($this->versionInstalled ?: 'not installed'), Log::INFO);
		if ($adapter->getManifest()) {
			$this->versionToInstall = (string) $adapter->getManifest()->version;
			$this->addLogEntry('plugin version in manifest = ' . ($this->versionToInstall ?: 'plugin manifest version not found'), Log::INFO);
			$this->versionMinimumJoomla = (string) $adapter->getManifest()->attributes()->version;
		}
		// holds data for user messages
		$this->status = new stdClass();
		$this->status->plugins = array();
		$this->status->modules = array();
		$this->status->components = array();
		$this->status->tables = array();
		$this->status->folders = array();
	}

    // interface

	public function preflight($route, $adapter): bool {
		if($route !== 'uninstall') {
			$jversion = new Version();
			// abort if the current Joomla release is older
			if (isset($this->versionMinimumJoomla) && version_compare($jversion->getShortVersion(), $this->versionMinimumJoomla, 'lt')) {
				$text = Text::_('PLG_USER_SPAMBOTCHECK_WRONG_JOOMLA_VERSION') . $this->versionMinimumJoomla;
				$app  = Factory::getApplication();
				$app->enqueueMessage($text, 'warning');
				$this->addLogEntry($text);
				return false;
			}
		}
        return true;
	}

	public function postflight($route, $adapter): bool {
		// run version specific update code
		if ($route == 'update') {
			if (isset($this->versionInstalled) && version_compare($this->versionInstalled, '1.3.12', 'lt'))
				$this->postFlightForVersion_1_3_12();
			if (isset($this->versionInstalled) && version_compare($this->versionInstalled, '1.3.13', 'lt'))
				$this->postFlightForVersion_1_3_13();
			$this->convertTablesToUtf8mb4();
		}

        // enable the plugin
        if('uninstall' !== $route) {
	        $db         = Factory::getDbo();
	        $conditions = array(
		        $db->qn('type') . ' = ' . $db->q('plugin'),
		        $db->qn('element') . ' = ' . $db->quote('spambotcheck'),
		        $db->qn('folder') . ' = ' . $db->quote('user')
	        );
	        $fields     = array($db->qn('enabled') . ' = 1');

	        $query = $db->getQuery(true);
	        $query->update($db->quoteName('#__extensions'))->set($fields)->where($conditions);
	        try {
		        // SQL errors
		        $db->setQuery($query);
		        // DBMS errors
		        $db->execute();
		        // message to user
		        $app = Factory::getApplication();
		        $app->enqueueMessage(Text::_('PLG_USER_SPAMBOTCHECK_ENABLED'), 'message');
	        }
	        catch (RuntimeException $e) {
		        $text = Text::_('PLG_USER_SPAMBOTCHECK_NOT_ENABLED');
		        $app  = Factory::getApplication();
		        $app->enqueueMessage($text, 'warning');
		        $this->addLogEntry($text, Log::WARNING);
	        }
        }
        return true;
	}

	public function install($adapter): bool {
		// give a warning if cURL is not enabled on system; plugin will not be able to identify spammer
		$extension = 'curl';
		if (!extension_loaded($extension)) {
			$text = Text::_('PLG_USER_SPAMBOTCHECK_CURL_MISSING');
			$app  = Factory::getApplication();
			$app->enqueueMessage($text, 'warning');
			$this->addLogEntry($text, Log::WARNING);
		}
        return true;
	}

	public function uninstall($adapter): bool {
        // delete own table #__spambot_attempts
		$db = Factory::getDBO();
		if ($db) {
			$attempts       = $db->getPrefix() . '_spambot_attempts';
			$tablesAllowed  = $db->getTableList();
			if (!in_array($attempts, $tablesAllowed)) {
				$db->setQuery("drop table if exists #__spambot_attempts");
				try {
					$db->execute();
				}
				catch (Exception $e) {
					$this->addLogEntry('Could not delete database table #__spambot_attempts', Log::WARNING);
					return false;
				}
			}
		}
		return true;
	}

    // implementation

	private function postFlightForVersion_1_3_12() {
		$this->addLogEntry('plugin performing postflight for Version 1.3.12', Log::INFO);
		// inspect value of parameter spbot_monitor_events: value format changed from 0/1 to R/RL
		$name   = 'spbot_monitor_events';
		$params = $this->getExtensionParameter();
		if (array_key_exists($name, $params)) {
			if ($params[$name] === '0')
				$params[$name] = 'R';
			if ($params[$name] === '1')
				$params[$name] = 'RL';
		}
		// add new parameters
		$params['spbot_blacklist_email']    = '';
		$params['spbot_bl_log_to_db']       = '0';
		$params['spbot_suspicious_time']    = '12';
		$params['spbot_allowed_hits']       = '3';
		$this->setExtensionParameters($params);
	}

	private function postFlightForVersion_1_3_13() {
		$this->addLogEntry('plugin performing postflight for Version 1.3.13', Log::INFO);
		// get all extension parameters
		$params = $this->getExtensionParameter();
		// remove spambusted.com related parameter
		$name = 'spbot_spambusted';
		if (array_key_exists($name, $params)) {
			unset($params[$name]);
		}
		// remove deprecated spacer parameter (used to structure the parameter ui layout)
		$name = '@spacer';
		if (array_key_exists($name, $params)) {
			unset($params[$name]);
		}
		// add new parameters
		$name   = 'spbot_projecthoneypot_api_key';
		$param  = $this->getExtensionParameter('params', $name);
		$value  = is_string($param) && $param != '' ? '1' : '0';
		$params['spbot_projecthoneypot'] = $value;
		// set all parameters
		$this->setExtensionParameters($params);
	}

	private function getExtensionParameter($field = 'params', $name = null) {
		// read the existing component value(s)
		$db     = Factory::getDbo();
		$query  = $db->getQuery(true);
		$query
            ->select($db->qn($field))
			->from($db->qn('#__extensions'))
			->where($db->qn('name') . ' = ' . $db->q('User - SpambotCheck'));
		try {
            // SQL errors
			$db->setQuery($query);
            // DBMS errors
			$params = json_decode($db->loadResult(), true);
			if (isset($name)) {
				return $params[$name];
			}
			return $params;
		}
		catch (RuntimeException $e) {
			$this->addLogEntry('unable to get Plugin User - SpambotCheck parameter ' . (isset($name) ? $name : "") . ' from database');
        }
        return array();
	}

	private function setExtensionParameters($params, $field = 'params') {
		// write the existing component value(s)
		$db             = Factory::getDbo();
		$paramsString   = json_encode($params);
		$conditions     = array(
			$db->qn('type') . ' = ' . $db->q('plugin'),
			$db->qn('element') . ' = ' . $db->quote('spambotcheck'),
			$db->qn('folder') . ' = ' . $db->quote('user')
		);
		$fields = array($db->qn($field) . ' = ' . $db->q($paramsString));

		$query = $db->getQuery(true);
		$query->update($db->quoteName('#__extensions'))->set($fields)->where($conditions);
		try {
			$db->setQuery($query);
			$db->execute();
		}
		catch (Exception $e) {
			$this->addLogEntry('unable to set User - SpambotCheck parameter in database');
		}
	}

	private function installationResults($route, $status) {
		$language = Factory::getLanguage();
		$language->load('plg_spambotcheck');
		$rows = 0;
		$version = ($route == 'install')
			? Text::_('PLG_USER_SPAMBOTCHECK_INSTALL_VERSION') . $this->versionToInstall
			: Text::_('PLG_USER_SPAMBOTCHECK_UPDATE_VERSION') . $this->versionInstalled . Text::_('PLG_USER_SPAMBOTCHECK_TO') . $this->versionToInstall;
		// info: install info or new feature text for update
		$info = ($route == 'install')
			? Text::_('PLG_USER_SPAMBOTCHECK_INSTALL_INFO')
			: Text::_('PLG_USER_SPAMBOTCHECK_NEW_FEATURE'); ?>

        <div class="span12" style="font-weight:normal">
            <p><strong><?php echo $version; ?></strong></p>
            <p><?php echo Text::_('PLG_USER_SPAMBOTCHECK_INSTALL_MESSAGE'); ?></p>
            <img src="<?php echo JURI::base(); ?>/components/com_spambotcheck/images/logo-banner.png" alt=""
                 align="right"/>
            <h2><?php echo Text::_('PLG_USER_SPAMBOTCHECK_INSTALLATION_STATUS'); ?></h2>
            <table class="adminlist table table-striped">
                <thead><tr>
                    <th class="title" style="text-align: left;"><?php echo Text::_('PLG_USER_SPAMBOTCHECK_HEADER_PLUGIN'); ?></th>
                    <th width="40%" style="text-align: left;"><?php echo Text::_('PLG_USER_SPAMBOTCHECK_HEADER_STATUS'); ?></th>
                </tr></thead>
                <tbody>
                <tr class="row0">
                    <td><p><strong><?php echo Text::_('PLG_USER_SPAMBOTCHECK_PLUGIN_NAME'); ?></strong></p>
                        <p><?php echo Text::_('PLG_USER_SPAMBOTCHECK_DESC'); ?></p>
                    </td>
                    <td><p><strong><?php echo Text::_('PLG_USER_SPAMBOTCHECK_INSTALLED'); ?></strong></p>
                        <p><?php echo $info; ?></p>
                    </td>
                </tr>
				<?php if (count($status->components)): ?>
                    <tr>
                        <th><?php echo Text::_('PLG_USER_SPAMBOTCHECK_HEADER_COMPONENT'); ?></th>
                        <th></th>
                    </tr><?php
                    foreach ($status->components as $component):
						// no empty strings
						is_string($component['name']) && $component['name'] != "" ? $name = $component['name'] : $name = "";
						is_string($component['desc']) && $component['desc'] != "" ? $desc = $component['desc'] : $desc = "";
						is_string($component['feature']) && $component['feature'] != "" ? $feature = $component['feature'] : $feature = "";
						is_string($component['info']) && $component['info'] != "" ? $info = $component['info'] : $info = "";
						is_string($component['message']) && $component['message'] != "" ? $message = $component['message'] : $message = "";
						// success or error text
						if ($component['result']) {
							$state = Text::_('PLG_USER_SPAMBOTCHECK_INSTALLED');
							$style = '';
							// info: install info or new feature text for update
							$info = ($route == 'install') ? $info : $feature;
						}
						else {
							$state = Text::_('PLG_USER_SPAMBOTCHECK_NOT_INSTALLED');
							// info: coloured error message
							$style = ' style="color: red"';
							// info: error message generated during coder execution
							$info = $message;
						} ?>
                        <tr class="row<?php echo(++$rows % 2); ?>">
                            <td class="key"><p><strong><?php echo $name; ?></strong></p>
                                <p><?php echo $desc; ?></p>
                            </td>
                            <td><p><strong<?php echo $style; ?>><?php echo $state; ?></strong></p>
                                <p><?php echo $info; ?></p>
                            </td>
                        </tr>
					<?php endforeach; ?>
				<?php endif; ?>
                </tbody>
            </table>
        </div><?php
	}

	private function uninstallationResults($status) {
		$language = Factory::getLanguage();
		$language->load('plg_spambotcheck');
		$rows = 0; ?>
        <div class="span12" style="font-weight:normal">
            <h2><?php echo Text::_('PLG_USER_SPAMBOTCHECK_REMOVAL_STATUS'); ?></h2>
            <table class="adminlist table table-striped">
                <thead><tr>
                    <th class="title"
                        style="text-align: left;"><?php echo Text::_('PLG_USER_SPAMBOTCHECK_HEADER_PLUGIN'); ?></th>
                    <th style="text-align: left;"><?php echo Text::_('PLG_USER_SPAMBOTCHECK_HEADER_STATUS'); ?></th>
                </tr></thead>
                <tbody>
                <tr class="row0">
                    <td class="key"><?php echo Text::_('PLG_USER_SPAMBOTCHECK_PLUGIN_NAME'); ?></td>
                    <td><strong><?php echo Text::_('PLG_USER_SPAMBOTCHECK_REMOVED'); ?></strong></td>
                </tr><?php
                if (count($status->components)): ?>
                    <tr>
                        <th><?php echo Text::_('PLG_USER_SPAMBOTCHECK_HEADER_COMPONENT'); ?></th>
                        <th></th>
                    </tr><?php
                    foreach ($status->components as $component):
						// no empty strings
						is_string($component['name']) && $component['name'] != "" ? $name = $component['name'] : $name = "";
						// success or error text
						if ($component['result']) {
							$state = Text::_('PLG_USER_SPAMBOTCHECK_REMOVED');
							$style = '';
						}
						else {
							$state = Text::_('PLG_USER_SPAMBOTCHECK_NOT_REMOVED');
							// info: coloured error message
							$style = ' style="color: red"';
						} ?>
                        <tr class="row<?php echo(++$rows % 2); ?>">
                            <td class="key"><?php echo $name; ?></td>
                            <td><p><strong<?php echo $style; ?>><?php echo $state; ?></strong></p>
                        </tr>
					<?php endforeach; ?>
				<?php endif; ?>
                </tbody>
            </table>
        </div><?php
	}

	private function convertTablesToUtf8mb4() {
		// Joomla! will use character set utf8 as default, if utf8mb4 is not supported
		// if we have successfully converted to utf8md4, we set a flag in the database
		$db = Factory::getDbo();
		$serverType = $db->getServerType();
		if ($serverType != 'mysql') {
			return;
		}
        $pluginParams = $this->getExtensionParameter();
		$convertedDB = 0;
		
		if (is_array($pluginParams) && isset($pluginParams['utf8_conversion'])) {
		    $convertedDB = (int) $pluginParams['utf8_conversion'];
        }

		if ($db->hasUTF8mb4Support()) {
			$converted = 2;
		}
		else {
			$converted = 1;
		}

		if ($convertedDB == $converted) {
			return;
		}
		$tableList = array('#__spambot_attempts', '#__user_spambotcheck');
		foreach ($tableList as $table) {
		    $query = 'ALTER TABLE ' . $table . ' CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci';
			try {
			    $db->setQuery($query);
			    $db->execute();
            }
            catch (RuntimeException $e) {
	            $converted = 0;
            }
		}
		$pluginParams['utf8_conversion'] = (string) $converted;
		$this->setExtensionParameters($pluginParams);
	}

	// logging

	private function initializeLogger($adapter) {
		$this->loggerName = (string) $adapter->getManifest()->loggerName;
		$options['format']              = "{CODE}\t{MESSAGE}";
		$options['text_entry_format']   = "{PRIORITY}\t{MESSAGE}";
		$options['text_file']           = 'spambotcheck_update.php';
		try {
			Log::addLogger($options, Log::ALL, array($this->loggerName, 'jerror'));
		}
		catch (RuntimeException $e) {}
	}

    private function addLogEntry($message, $level = Log::ERROR) {
		try {
			Log::add($message, $level, $this->loggerName);
		}
		catch (RuntimeException $exception)
		{
			// prevent installation routine from failing due to problems with logger
		}
	}
}