<?php /** @noinspection MissingSinceTagDocInspection */
/**
 * General Helper for Spambotcheck
 *
 * @author		 vi-solutions, Aicha Vack & Ingmar Vack
 * @package		 User SpambotCheck - check for possible spambots during register and login
 * @subpackage   Spambotchek Helper
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2021 vi-solutions
 * @since        Joomla 4.0
 */
namespace Visolutions\Plugin\User\Spambotcheck\Helper;

defined('_JEXEC') or die;

use RuntimeException;

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Access\Access;
use Joomla\CMS\User\User;
use Joomla\CMS\User\UserHelper;
use DateTimeZone;
use Joomla\CMS\Date\Date;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\Database\Exception\ExecutionFailureException;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;
use Joomla\Database\QueryInterface;
use Joomla\Session\SessionInterface;
use Joomla\Utilities\ArrayHelper;

class SpambotCheckHelper {
    public static function cleanEMailWhitelist($list): string {
        if ($list != '') {
            // delete blanks
            $list = str_replace(' ', '', $list);
            // delete ',' at string end
            while ($list[strlen($list) - 1] == ',') {
                $list = substr($list, 0, strlen($list) - 1);
            }
        }
        return $list;
    }

    public static function cleanEMailBlacklist($list): string {
        if ($list != '') {
            // delete blanks
            $list = str_replace(' ', '', $list);
            // delete ',' at string end
            while ($list[strlen($list) - 1] == ',') {
                $list = substr($list, 0, strlen($list) - 1);
            }
        }
        return $list;
    }

    public static function cleanUsername($name): string {
        if ($name != '') {
            $name = addslashes(htmlentities($name));
            $name = urlencode($name);
            $name = str_replace(" ", "%20", $name); // no spaces
        }

        return $name;
    }

    public static function isCUrlAvailable(): bool {
        $extension = 'curl';
        if (extension_loaded($extension)) {
            return true;
        }

        return false;
    }

    public static function isURLOnline($URL): bool {
        // check, if curl is available
        if (self::isCUrlAvailable()) {
            // check if url is online
            $curl = @curl_init($URL);
            curl_setopt($curl, CURLOPT_TIMEOUT, 10);
            curl_setopt($curl, CURLOPT_FAILONERROR, 1);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
            @curl_exec($curl);
            if (curl_errno($curl) != 0) {
	            curl_close($curl);
                return false;
            }
            curl_close($curl);
            return true;
        }

        // curl is not loaded, this won't work
        return false;
    }

    public static function getURL($URL): string {
        if (self::isURLOnline($URL) == false) {
            return 'Unable to connect to server';
        }

		if (function_exists('file_get_contents') && ini_get('allow_url_fopen') == true) {
		    // use file_get_contents
		    $returnURL = @file_get_contents($URL);
	    }
	    else {
		    // use cURL (if available)
		    if (self::isCUrlAvailable()) {
			    $curl = @curl_init();
			    curl_setopt($curl, CURLOPT_URL, $URL);
			    curl_setopt($curl, CURLOPT_VERBOSE, 1);
			    curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
			    curl_setopt($curl, CURLOPT_HEADER, 0);
			    $returnURL = @curl_exec($curl);
			    curl_close($curl);
		    }
		    else {
			    return 'Unable to connect to server';
		    }
	    }

	    return $returnURL;
    }

    public static function isValidIP(string $IP): string {
	    if ($IP != '') {
            $regex = "'\b(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b'";
            return preg_match($regex, $IP) ? $IP : '';
        }

        return '';
    }

	public static function isValidEmail($email): string {
        if ($email != '') {
            $regex = '/^([a-zA-Z0-9_\.\-\+%])+\@(([a-zA-Z0-9\-])+\.)+([a-zA-Z0-9]{2,4})+$/';
            return preg_match($regex, $email) ? $email : '';
        }

        return '';
    }

    // Usage example:
    // ---------------
    // logSpammerToDB('test@test.com', '12.12.12.12', 'username', 'ProjectHoneyPot', '127.41.11.5', 'ThreatScore=11, DaysSinceLastActivity=41', $plgParams)
    //
	public static function logSpammerToDB($email, $IP, $username, $engine, $request, $rawReturn, $parsedReturn, &$params): bool {
		// check plugin settings: save spambots to DB
		if (!$params->get('spbot_log_to_db', 0)) {
            return false;
        }

        // change empty vars to "NULL"
        if ($email == '') {
            $email = 'NULL';
        }
        if ($IP == '') {
            $IP = 'NULL';
        }
        if ($username == '') {
            $username = 'NULL';
        }

        // trim anything that could screw up SQL
        $email   = str_replace(array("0x", ",", "%", "'", "\r\n", "\r", "\n"), "", $email);
        //$sEmail = mysql_real_escape_string($sEmail);

        $IP      = str_replace(array("0x", ",", "%", "'", "\r\n", "\r", "\n"), "", $IP);
        //$sIP = mysql_real_escape_string($sIP);

        $username = str_replace(array("0x", ",", "%", "'", "\r\n", "\r", "\n"), "", $username);


        // add DB record
        $db      = Factory::getContainer()->get(DatabaseInterface::class);
        $date    = gmdate("Y-m-d H:i:s", time());
        $action  = $params->get('current_action', '-');
		$query   = $db->createQuery();
		$columns = array('action', 'email', 'ip', 'username', 'engine', 'request', 'raw_return', 'parsed_return', 'attempt_date');
		$values  = array($db->quote($action), $db->quote($email), $db->quote($IP), $db->quote($username), $db->quote($engine), $db->quote($request), $db->quote($rawReturn), $db->quote($parsedReturn), $db->quote($date));
		/** @noinspection SqlResolve */
		$query
			->insert($db->quoteName('#__spambot_attempts'))
			->columns($db->quoteName($columns))
			->values(implode(',', $values));
		try {
	        $db->setQuery($query);
	        $db->execute();
        }
        catch (RuntimeException $e) {
            return false;
        }

        return true;
    }

    public static function userIsAdmin($user) {
        if ($userId = UserHelper::getUserId($user['username'])) {
            $db     = Factory::getContainer()->get(DatabaseInterface::class);
            $query  = $db->createQuery();
            $query
	            ->select($db->qn('g.id'))
	            ->from($db->qn('#__usergroups') . ' AS ' . $db->qn('g'))
	            ->join('LEFT', $db->qn('#__user_usergroup_map') . ' AS ' . $db->qn('map') . ' ON ' . $db->qn('map.group_id') . ' = ' . $db->qn('g.id'))
	            ->where($db->qn('map.user_id') .'=' . $db->quote($userId));
			//$query = 'SELECT g.id AS group_id FROM `#__usergroups` AS g LEFT JOIN `#__user_usergroup_map` AS map ON map.group_id=g.id WHERE map.user_id=' . $db->quote($userid);

	        // A user can be member of more than one user groups
	        try {
		        $db->setQuery($query);
		        $userGroups = $db->loadObjectList();
	        }
	        catch (RuntimeException $e) {
	        	return false;
	        }

			// check if any of these groups have admin rights
            foreach ($userGroups as $group) {
                $groupId = $group->id;
                if (Access::checkGroup($groupId, 'core.admin') == 1) { // user is admin
                    return true;
                }
            }
            return false;
        }
        return false;
    }

    public static function getSuperUserGroups(): array {
	    // store superuser groups later here
	    $superUserGroups = array();

	    // get all user groups from database
	    $db     = Factory::getContainer()->get(DatabaseInterface::class);
	    $query  = $db->createQuery();
	    $query->select($db->qn('id'));
	    /** @noinspection SqlResolve */
	    $query->from($db->qn('#__usergroups'));
	    try {
	        $db->setQuery($query);
	        $groups = $db->loadColumn();
        }
        catch (RuntimeException $e) {
	        return ($superUserGroups);
        }

        foreach ($groups as $value) {
            // check if group has superuser rights (core.admin)
            $SuperAdmin = Access::checkGroup($value, 'core.admin');
            if ($SuperAdmin == 1) {
                // store in returned array
                $superUserGroups[] = $value;
            }
        }

        return ($superUserGroups);
    }

    public static function logUserData($userId): bool {
        // create and populate an object
        $user_spambot = new \stdClass;
        $user_spambot->user_id = $userId;
        if (Factory::getApplication()->isClient('site')) {
            $user_spambot->ip = self::getIP();
        } else {
            $user_spambot->ip = "";
            $user_spambot->note = "Backend creation";
            $user_spambot->trust = 1;
        }

        // insert the object into the user_spambot table
	    try {
		    return Factory::getContainer()->get(DatabaseInterface::class)->insertObject('#__user_spambotcheck', $user_spambot);
	    }
	    catch (RuntimeException $e) {
        	return false;
	    }
    }

    /**
     * Method to get a list of users with same IP.
     *
     * @param  string	User Ip from SERVER REMOTE ADDRESS
     * @return  mixed false or object of Array's with user data
     *
     * @since   1.6
     */
    public static function getUsersByIp($ip) {
        if ($ip == '') {
            // user was created by admin in backend or user was already registered when component was installed
            return false;
        }

        $db     = Factory::getContainer()->get(DatabaseInterface::class);
        $query  = $db->createQuery();

	    /** @noinspection SqlResolve */
	    // select all records from spambot user table with .
	    $query
            ->select($db->quoteName(array('a.id', 'a.user_id', 'a.ip', 'a.hits', 'a.note', 'a.trust')))
            ->select($db->quoteName('b.id', 'bid'))
            ->select($db->quoteName('b.registerDate', 'registerDate'))
            ->from($db->quoteName('#__user_spambotcheck', 'a'))
            ->join('INNER', $db->quoteName('#__users', 'b') . ' ON (' . $db->quoteName('a.user_id') . ' = ' . $db->quoteName('b.id') . ')')
            ->where($db->quoteName('ip') . ' = ' . $db->quote($ip))
            ->order($db->quoteName('a.id') . ' asc');
	    // load the results as a list of stdClass objects
	    try {
		    $db->setQuery($query);
		    return $db->loadObjectList();
	    }
	    catch (RuntimeException $e) {
		    return false;
	    }
    }

    /**
     * Method to calculate the difference between to timestamps
     *
     * @param $firstDate      Date
     * @param $secondDate     Date
     *
     * @return  int in seconds
     *
     * @since   1.6
     */
    public static function getDateDiff(Date $firstDate, Date $secondDate): int {
        // check that we have two dateTime strings
        if (!strtotime($secondDate)) {
            if (is_numeric($secondDate)) {
                // we assume we have a unix timestamp and convert it
                $secondDate = new Date($secondDate);
            } else {
                // we can set the registration date of the new user to now and create a proper Date
                $secondDate = new Date();
            }
        }

        if (!strtotime($firstDate)) {
            if (is_numeric($firstDate)) {
                // we assume we have a unix timestamp and convert it
                $firstDate = new Date($firstDate);
            }
        }

        if ((!strtotime($firstDate)) || (!strtotime($secondDate))) {
            return 0;
        }

        $firstDate  = strtotime($firstDate);
        $secondDate = strtotime($secondDate);

	    return abs($secondDate - $firstDate);
    }

    /**
     * Method to check if a IP is suspicious and update user data.
     *
     * @param $data         array with user data
     * @param $params       Object with plugin params
     *
     * @return  boolean true
     *
     * @since   1.6
     */
    public static function checkIpSuspicious(array $data, object $params): bool {
        $userId         = ArrayHelper::getValue($data, 'id', 0, 'int');
        $userIp         = self::getIP();
        $userRegDate    = ArrayHelper::getValue($data, 'registerDate', new Date(), 'date');
        // object with array of users with same IP
        $sameIPs        = self::getUsersByIp($userIp);
        $allowedHits    = $params->get('spbot_allowed_hits', 2);
        $allowedSeconds = ($params->get('spbot_suspicious_time', 12)) * 60 * 60;

        if ($sameIPs !== false && count($sameIPs) > 1) {
            // we have already an old user with the same IP
            $hits = count($sameIPs);
			// however, the meaning of the suspicious value is confusing
			// $suspicious = 1 --> not suspicious
			// $suspicious = 0 --> is suspicious
            $suspicious = 1;
            foreach ($sameIPs as $pk => $value) {
                // check time difference between first registration with this IP and the actual registration
                if ($pk == 0) {
                    $diff = self::getDateDiff(new Date($value->registerDate), new Date ($userRegDate));
                    if ($diff < $allowedSeconds) {
                        if ($hits > $allowedHits) {
                            // that is suspicious
                            $suspicious = 0;
                        }
                    }
                }

                // update data of old users with same Ip
                if ($userId != $value->user_id) {
                    // create an object for the record we are going to update
                    $object = new \stdClass();
                    $object->id = $value->id;
                    // set hits field
                    $object->hits = $hits;
                    // add a note
                    $object->note = $value->note . '1: ' . $userId . '; ';
                    if (($suspicious == 0) && ($value->trust != 1)) {
                        // set suspicious state
                        $object->suspicious = $suspicious;
                    }
                    // update their details in the users table using id as the primary key
	                try {
		                Factory::getContainer()->get(DatabaseInterface::class)->updateObject('#__user_spambotcheck', $object, array('id'));
	                }
	                catch (RuntimeException $e) {

	                }
                }

                // update data of new user
                if ($userId == $value->user_id) {
                    $note = '';
                    foreach ($sameIPs as $pk1 => $value1) {
                        if ($userId != $value1->user_id) {
                            $note .= '1: ' . $value1->user_id . '; ';
                        }
                    }
                    // create an object for the record we are going to update
                    $object = new \stdClass();
                    $object->id = $value->id;
                    // set hits field
                    $object->hits = $hits;
                    // add a note
                    $object->note = $note;
                    if (($suspicious == 0) && ($value->trust != 1)) {
                        // set suspicious state
                        $object->suspicious = $suspicious;
                    }
                    // update their details in the users table using id as the primary key
	                try {
		                Factory::getContainer()->get(DatabaseInterface::class)->updateObject('#__user_spambotcheck', $object, array('id'));
	                }
	                catch (RuntimeException $e) {}
                }
            }
        }
        return true;
    }

    /**
     * Method to get the value of a specified field using a where condition.
     *
     * @param $table         string Tablename
     * @param $field         string field
     * @param $whereField    string field for where condition
     * @param $value         string value of where condition
     *
     * @return  string 		fieldvalue
     *
     * @since   1.6
     */
    public static function getTableFieldValue(string $table = '#__user_spambotcheck', string $field = 'ip', string $whereField = 'user_id', string $value = '') {
        // get registration IP of deleted user
        if ($field != '' && $whereField != '' && $value != '') {
            $db     = Factory::getContainer()->get(DatabaseInterface::class);
            $query  = $db->createQuery();
            $query->select($db->quoteName($field));
            $query->from($db->quoteName($table));
            $query->where($db->quoteName($whereField) . " = " . $db->quote($value));
	        try {
	            $db->setQuery($query);
	            return $db->loadResult();
            }
            catch (RuntimeException $e) {}
        }
	    return false;
    }

    /**
     * Method to delete parts of note text of nested datasets with same Ip when a user is deleted.
     *
     * @param $userIP    string	IP
     * @param $userID    string    user id
     *
     * @return  void
     *
     * @since   1.6
     */
    public static function cleanUserSpambotTable(mixed $userIP, string $userID) {
        if (SpambotCheckHelper::is_string_non_empty($userIP)) {
            $sameIps = self::getUsersByIp($userIP);
			$db = Factory::getContainer()->get(DatabaseInterface::class);
            foreach ($sameIps as $pk => $value) {
                if ($value->note != '') {
                    $value->note = str_replace('1: ' . $userID . '; ', "", $value->note);
                    // create an object for the record we are going to update
                    $object = new \stdClass();
                    $object->id = $value->id;
                    // set hits field
                    $object->hits = $value->hits - 1;
                    // add a note
                    $object->note = $value->note;
                    // update their details in the users table using id as the primary key
	                try {
		                $db->updateObject('#__user_spambotcheck', $object, array('id'));
	                }
	                catch (RuntimeException $e) {}
                }
            }
        }
    }

    /**
     * Method to check if an email address is suspicious and update #_user_spambotcheck table.
     *
     * @param array $data user
     *
     * @return  void
     *
     * @since   1.6
     */
    public static function checkEmailSuspicious(array $data) {
        $suspicious = 1;
        $userId     = ArrayHelper::getValue($data, 'id', 0, 'int');
        $email      = ArrayHelper::getValue($data, 'email', '', 'string');
        $note       = '';
        if (isset($email) && $email != '') {
            // check vor 3 or more dots left of @
            $regex1 = '/^([^\.]*[\.]){3,}[^\.]*@.*$/';
            if (preg_match($regex1, $email)) {
                // that is suspicious
                $suspicious = 0;
                $note = '2: To many dots; ';
            }
        }

        if ($suspicious == 0) {
            // get value of note field
            $noteValue = self::getTableFieldValue('#__user_spambotcheck', 'note', 'user_id', $userId);
            // create an object for the record we are going to update
            $object = new \stdClass();
            $object->user_id = $userId;
            // add a note
            $object->note = $noteValue . $note;
            // set suspicious state
            $object->suspicious = $suspicious;
            // update their details in the users table using user_id as the primary key
	        try {
		        Factory::getContainer()->get(DatabaseInterface::class)->updateObject('#__user_spambotcheck', $object, array('user_id'));
	        }
	        catch (RuntimeException $e) {

	        }
        }
    }

    /**
     * Method to set an old user to suspicious, if their ip is now listed in spambot databases
     *
     * @param String $userID user Id of user who was prevented from login because listed in online spambot database
     *
     * @return  void
     *
     * @since   1.6
     */
    public static function flagUserWithSpamUserIp(String $userID = '0') {
        $userIP = self::getIP();
        $userRegDate = new Date();

        // object with array of users with same IP
        $sameIPs = self::getUsersByIp($userIP);
        $allowedSeconds = 48 * 60 * 60;

        if ($sameIPs !== false && count($sameIPs) > 0) {
        // we have already an old user with the same IP
            $suspicious = 1;
			$db = Factory::getContainer()->get(DatabaseInterface::class);
            foreach ($sameIPs as $pk => $value) {
                // check time difference between first registration with this IP and the actual registration
                if ($pk == 0) {
                    $diff = self::getDateDiff($value->registerDate, $userRegDate);
                    if ($diff < $allowedSeconds) {
                        // that is suspicious
                        $suspicious = 0;
                    }
                }

				// update data of old users with same Ip if it doesn't already have the error code in note
                if (($userID != $value->user_id) && ($suspicious == 0) && ($value->trust != 1)) {
                    if (!str_contains($value->note, '3: IP flagged; ')) {
                        // create an object for the record we are going to update
                        $object = new \stdClass();
                        $object->id = $value->id;
                        // add a note
                        $object->note = $value->note . '3: IP flagged; ';
                        // set suspicious state
                        $object->suspicious = $suspicious;
                        // update their details in the users table using id as the primary key
	                    try {
		                    $db->updateObject('#__user_spambotcheck', $object, array('id'));
	                    }
	                    catch (RuntimeException $e) {}
                    }
                }
            }
        }
    }

    public static function checkComponentInstalled(): bool {
        $db     = Factory::getContainer()->get(DatabaseInterface::class);
        $query  = $db->createQuery();
        $query->select('COUNT(*)');
	    /** @noinspection SqlResolve */
        $query->from($db->quoteName('#__extensions'));
        $query->where($db->quoteName('element') . " = " . $db->quote('com_spambotcheck') . ' AND ' . $db->quoteName('enabled') . " = " . $db->quote('1'));
	    try {
	        $db->setQuery($query);
	        if (!$db->loadResult()) {
		        return false;
	        }
	        return true;
        }
        catch (RuntimeException $e) {
	        return false;
        }
    }

	public static function updateSessionAndSessionRecord() {
		$config = ComponentHelper::getParams('com_users');
		$defaultUserGroup = $config->get('new_usertype', 2);

		// get the session
		/** @var SessionInterface $session */
		$session = Factory::getApplication()->getSession();

		// create a guest user
		$user = new User();
		$user->id = 0;
		$user->name ='';
		$user->username = '';
		$user->groups = array($defaultUserGroup);

		// first:
		// replace session user with guest user
		$session->set('user', $user);

		// second:
		// store the guest user to the #__session table using the session id of the session created by the spammer
		// thus replacing the logged in spammer with a guest user

		$db = Factory::getContainer()->get(DatabaseInterface::class);
		/** @var QueryInterface $query */
		$query = $db->createQuery();

		$setValues = [
			$db->quoteName('guest') . ' = :guest',
			$db->quoteName('userid') . ' = :user_id',
			$db->quoteName('username') . ' = :username',
		];

		// Bind query values
		$sessionId   = $session->getId();
		$userId      = $user->id;
		$userIsGuest = $user->guest;
		$username    = $user->username;

		$query
			->bind(':session_id', $sessionId)
			->bind(':guest', $userIsGuest, ParameterType::INTEGER)
			->bind(':user_id', $userId, ParameterType::INTEGER)
			->bind(':username', $username);

		$app = Factory::getApplication();
		// todo: possibly clarify if-statement (especially 'shared_session')
		// which is basically copied from \Joomla\CMS\Session\MetadataManager::updateSessionRecord
		if ($app instanceof CMSApplication && !$app->get('shared_session', false)) {
			$clientId = $app->getClientId();
			$setValues[] = $db->quoteName('client_id') . ' = :client_id';
			$query->bind(':client_id', $clientId, ParameterType::INTEGER);
		}

		$query
			->update($db->quoteName('#__session'))
			->set($setValues)
			->where($db->quoteName('session_id') . ' = :session_id');

		try {
			$db->setQuery($query);
			$db->execute();
		}
		catch (ExecutionFailureException $e) {
			// This failure isn't critical, we can go on without the metadata
			return;
		}
	}

	public static function addRangeToQuery(string $field, string $range, DatabaseInterface $db, QueryInterface $query) {
		if ($range == '' || $range == '*') {
			return;
		}

		// get UTC for now
		$dNow = new Date;
		$dStart = clone $dNow;

		switch ($range) {
			case 'past_week':
				$dStart->modify('-7 day');
				break;

			case 'past_1month':
				$dStart->modify('-1 month');
				break;

			case 'past_3month':
				$dStart->modify('-3 month');
				break;

			case 'past_6month':
				$dStart->modify('-6 month');
				break;

			case 'post_year':
			case 'past_year':
				$dStart->modify('-1 year');
				break;

			case 'today':
				// ranges that need to align with local 'days' need special treatment
				$offset	= Factory::getApplication()->getConfig()->get('offset');

				// reset the start time to be the beginning of today, local time
				$dStart	= new Date('now', $offset);
				$dStart->setTime(0, 0, 0);

				// how change the timezone back to UTC
				$tz = new DateTimeZone('GMT');
				$dStart->setTimezone($tz);
				break;
		}

		if ($range == 'post_year') {
			$query->where(
				$db->qn($field) . ' < ' . $db->quote($dStart->format('Y-m-d H:i:s'))
			);
		}
		else {
			$query->where(
				$db->qn($field) . '  >= ' . $db->quote($dStart->format('Y-m-d H:i:s')).
				' AND ' .$db->qn($field) . ' <=' . $db->quote($dNow->format('Y-m-d H:i:s'))
			);
		}
	}

	public static function getIP() {
		return $_SERVER["HTTP_CF_CONNECTING_IP"] ?? $_SERVER['REMOTE_ADDR'];
	}

	public static function is_string_non_empty($val): bool {
		return is_string($val) && $val !== '';
	}
}