<?php
/**
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2020 vi-solutions
 */
namespace Visolutions\Component\Visforms\Administrator\Helper;
// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Event\Table\AfterDeleteEvent;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\SubscriberInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Event\Model;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisfieldSaveJFormExtraDataEvent;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisfieldAfterBatchCopyFormEvent;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisfieldPrepareJFormEvent;
use Joomla\Database\DatabaseInterface;

class FieldPluginHelper extends CMSPlugin implements SubscriberInterface {

    // This helper can be used to create a custom plugin that adds a custom tab with custom configuration options to field configuration
    // See also the comments in the FormPluginHelper

    // must be set in subclass, if custom plugin adds tab to field configuration and stores configuration settings in own data table
    // directory of xml definition file for the additional fields (configuration option) for the field configuration
    protected $formDir;
    // directory of table class file for the additional fields (configuration option) for the field configuration
    // @deprecated 5.4.0: will be removed in Visforms 6.0
    protected $tableDir;
    // table class name for the additional fields (configuration option) for the field configuration
    // @deprecated 5.4.0: will be removed in Visforms 6.0
    protected $plgTableName;
    // name of index field in table
    protected $tableKey;

    // set $hasOwnTableInDatabase to true, if custom plugin which extends this helper, has own datatable in which the custom configuration settings are stored
    // if set to true: make sure, that the custom plugin sets the following class members $formDir, $tableDir, $plgTableName, $tableKey, then!
    // the name of the plugin data table must be '#__'. $this->name
    // plugin data table/form xml definition must have the following fields: id, $tableKey, $this->name.'_params'.
    // id is reference to the id of the field which is extended.
    // make sure id is set properly for the administration parts and the frontend parts"
    // $this->name.'_params' is a fields node in form xml definition
    // make sure, that the names of the additional configuration options do not match any configuration option name, already uses in the field configuration

    // if set to false or required members are not set properly in subclasses, default implementation does not process the respective handler (like onVisfieldPrepareJForm)
    protected $hasOwnTableInDatabase = false;
	// load plugin language files
	protected $autoloadLanguage = true;
    // field id
    protected $fieldId;
    // plugin table class object
    protected $plgTable;

    public function __construct(DispatcherInterface $subject, array $config) {
        parent::__construct($subject, $config);
        // load com_visforms language files (active language and default en-GB)
        $this->loadLanguage('com_visforms', JPATH_SITE);
        $this->setPlgTable();
        if ($this->hasOwnTableInDatabase && !empty($this->params['wasDisabled'])) {
            // plugin is newly enabled
            // clean its data table from form records of forms, which were deleted, while the plugin was disables
            $this->removeDeletedFieldRecords();
            $this->removeDisabledFlag();
        }
        // in administration url parameter id is the field id, in frontend this would be the form id
        // in frontend set fieldId from field->id in handler !!!
        $app =  Factory::getApplication();
        if ($app->isClient('administrator')) {
            $this->fieldId = $app->getInput()->getInt('id', 0);
        }
    }

    public static function getSubscribedEvents(): array {
        return [
            'onTableAfterDelete' => 'onTableAfterDelete',
            'onVisfieldPrepareJForm' => 'onVisfieldPrepareJForm',
            'onVisfieldSaveJFormExtraData' => 'onVisfieldSaveJFormExtraData',
            'onVisformsAfterJFormDelete' => 'onVisformsAfterJFormDelete',
            'onVisfieldAfterBatchCopyForm' => 'onVisfieldAfterBatchCopyForm',
        ];
    }

    // backend actions manage additional configuration settings added by plugin to field configuration in database

    // default event handler implementation
    // add plugin specific options to field configuration to administration edit form
    public function onVisfieldPrepareJForm(VisfieldPrepareJFormEvent $event): void {
        $form = $event->getForm();
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
        $app = Factory::getApplication();
        if ($app->isClient('administrator')) {
            Form::addFormPath($this->formDir);
            $form->loadFile($this->name, false);
            $form = $this->removeFields($form);
            $data = $this->loadFormData($form);
            $form->bind($data);
        }
    }

    // default event handler implementation
    // save values set in plugin specific options in field configuration in plugin data table
    public function onVisfieldSaveJFormExtraData(VisfieldSaveJFormExtraDataEvent $event) : void {
        $data = $event->getData();
        $id = $event->getId();
        $isNew = $event->getIsNew();
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
        $app = Factory::getApplication();
        if ($app->isClient('administrator')) {
            $data['id'] = $id;
            if (isset($data[$this->name . '_params']) && is_array($data[$this->name . '_params'])) {
                $data[$this->name . '_params'] = VisformsHelper::registryStringFromArray($data[$this->name . '_params']);
            }
            $this->saveExtraData($data, $isNew);
        }
    }

    // default event handler implementation
    /// delete plugin specific records if a field is deleted in administration
    public function onVisformsAfterJFormDelete(Model\AfterDeleteEvent $event) : void {
        $context = $event->getContext();
        $table = $event->getItem();
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
        // skip plugin if context is wrong
        if ($context != 'com_visforms.visfield') {
            return;
        }
        $app = Factory::getApplication();
        if ($app->isClient('administrator')) {
            $db = Factory::getContainer()->get(DatabaseInterface::class);
            $query = $db->createQuery();
            $query->delete($db->quoteName('#__' . $this->name))
                ->where($db->quoteName('id') . ' = ' . $table->id);
            try {
                $db->setQuery($query);
                $db->execute();
            }
            catch (\RuntimeException $e) {}
        }
    }

    // default event handler implementation
    // handle plugin specific settings on batch copy of fields
    public function onVisfieldAfterBatchCopyForm(VisfieldAfterBatchCopyFormEvent $event) : void {
        $oldFormId = $event->getPk();
        $newFormId = $event->getNewId();
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
        $app = Factory::getApplication();
        if ($app->isClient('administrator')) {
            $data = $this->getItem($oldFormId, false);
            if (empty($data)) {
                return;
            }
            $data['id'] = $newFormId;
            $this->saveExtraData($data, true);
        }
    }

    // implementation using Joomla onTableAfterDeleteEvent
    // is triggered really often
    public function onTableAfterDelete(AfterDeleteEvent $event) {
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
        $subject = $event->getArgument('subject');
        // check, that we have a delete event on the visfields table.
        if ($subject->getTableName() !== '#__visfields') {
            return;
        }
        $app = Factory::getApplication();
        if ($app->isClient('administrator')) {
            $pks = $event->getArgument('pk'); // array of tablekey => keyvalue pairs
            if (is_array($pks) && isset($pks['id'])) {
                $pk = $pks['id'];
                $db = Factory::getContainer()->get(DatabaseInterface::class);
                $query = $db->createQuery();
                $query->delete($db->quoteName('#__' . $this->name))
                    ->where($db->quoteName('id') . ' = ' . $pk);
                try {
                    $db->setQuery($query);
                    $db->execute();
                }
                catch (\RuntimeException $e) {}
            }
        }
    }

    // implementation helper functions

    protected function loadFormData($form) {
        if (!$this->hasOwnTableInDatabase) {
            return true;
        }
        // $id = $form->get('id');
        $data = $this->getItem();
        if (empty($data)) {
            $data = array();
        }
        return $data;
    }

    protected function getItem($id = null, $extract = true) {
        if (!$this->hasOwnTableInDatabase) {
            return false;
        }
        if (empty($id)) {
            $id = $this->fieldId;
        }
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->createQuery();
        $query->select('*')
            ->from($db->quoteName('#__' . $this->name))
            ->where($db->quoteName('id') . ' = ' . $id);
        try {
            $db->setQuery($query);
            $data = $db->loadAssoc();
        }
        catch (\RuntimeException $e) {
            return false;
        }
        if ($data && $extract && isset($data[$this->name . '_params'])) {
            $data[$this->name . '_params'] = VisformsHelper::registryArrayFromString($data[$this->name . '_params']);
        }
        return $data;
    }

    // save data in database
    protected function saveExtraData($data, $isNew) : void {
        if (!$this->hasOwnTableInDatabase) {
            return;
        }
        $table = $this->plgTable;
        $tableKey = $this->tableKey;
        $app = Factory::getApplication();
        try {
            if (!($isNew)) {
                if (!isset($data[$tableKey])) {
                    return;
                }
                $data[$tableKey] = (int) $data[$tableKey];
                $table->load($data[$tableKey]);
            }
            else {
                unset($data[$tableKey]);
            }
            $table->bind($data);
            if (!$table->check()) {
                $app->enqueueMessage($table->getError(), 'error');
                return;
            }

            if (!$table->store()) {
                $app->enqueueMessage($table->getError(), 'error');
            }
        }
        catch (\InvalidArgumentException | \RuntimeException $e) {
            $app->enqueueMessage($e->getMessage(), 'error');
        }
    }

    // empty implementation
    // can be used in subclass to remove specific fields from the edit form
    protected function removeFields($form) {
        return $form;
    }

    // make sure data stored in data table of the plugin match existing fields
    protected function removeDeletedFieldRecords() {
        // get a list of all existing forms
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->createQuery();
        $query->select($db->qn('id'))
            ->from($db->qn('#__visfields'));
        try {
            $db->setQuery($query);
            $fieldIds = $db->loadColumn();
        }
        catch (\RuntimeException $e) {
            return false;
        }
        // get list of form records stored in plugins own data table
        $query = $db->createQuery();
        $query->select($db->qn('id'))
            ->from($db->qn('#__'. $this->name));
        try {
            $db->setQuery($query);
            $pluginFieldIds = $db->loadColumn();
        }
        catch (\RuntimeException $e) {
            return false;
        }
        if (empty($pluginFieldIds)) {
            return true;
        }
        if (empty($fieldIds)) {
            $fieldIds = array();
        }
        // remove recordsets of forms which no longer exist
        foreach ($pluginFieldIds as $pluginFormId) {
            if (!in_array($pluginFormId, $fieldIds)) {
                $query = $db->createQuery();
                $query->delete($db->quoteName('#__' . $this->name))
                    ->where($db->quoteName('id') . " = " . $pluginFormId);
                try {
                    $db->setQuery($query);
                    $db->execute();
                }
                catch (\RuntimeException $e) {
                    $error = $e->getMessage();
                }
            }
        }
        return true;
    }

    // remove the "wasDisabled" flag from plugin params
    private function removeDisabledFlag () {
        $fieldName = 'params';
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->createQuery();
        $query
            ->select($db->quoteName($fieldName))
            ->from($db->quoteName('#__extensions'))
            ->where($db->qn('element') . ' = ' . $db->q($this->_name) . ' AND ' . $db->qn('type') . ' = ' . $db->q('plugin') . ' AND ' . $db->qn('folder') . ' = ' . $db->q($this->_type));
        try {
            $db->setQuery($query);
            $result = $db->loadResult();
            if (empty($result)) {
                $params = array();
            }
            else {
                $params = json_decode($result, true);
            }
            $params['wasDisabled'] = false;
        }
        catch (\RuntimeException $e) {
            return false;
        }
        $paramsString = json_encode($params);
        try {
            $db->setQuery('UPDATE #__extensions SET ' . $fieldName . ' = ' .  $db->quote($paramsString) . ' WHERE element=' . $db->q($this->_name) . ' and type = ' . $db->q('plugin') .' and folder = ' . $db->q($this->_type));
            $db->execute();

        }
        catch (\RuntimeException $e) {
            return false;
        }
        return true;
    }

    // custom implementation in child classes must set the table if plugins has own table
    protected function setPlgTable() {
        $this->setLegacyTable();
        // ToDo: Visforms 6
        // $this->hasOwnTableInDatabase = false;
        // $this->plgTable = null;
    }

    /* @deprecated 5.4 will be removed in Visforms 6.0
     * Table class must be moved to src/Table directory, refactored and auto loaded
     * Implement setPlgTable() in Plugin
     */
    protected function setLegacyTable() {
        // Plugin does not provide proper values for members, which are used in the functions that modify the admin forms and the database
        if (empty($this->formDir) || empty($this->tableDir) || empty($this->plgTableName) || empty($this->tableKey)) {
            $this->hasOwnTableInDatabase = false;
        }
        else {
            $tableClass = 'Table' . $this->plgTableName;
            $path = $this->tableDir . '/' . (lcfirst($this->plgTableName)) . '.php';
            if (!class_exists($tableClass) && file_exists($path)) {
                require_once($path);
                try {
                    $db = Factory::getContainer()->get(DatabaseInterface::class);
                    $this->plgTable = new $tableClass($db);
                }
                catch (\RuntimeException $e) {
                    $this->hasOwnTableInDatabase = false;
                }
            }
            else {
                $this->hasOwnTableInDatabase = false;
            }
        }
    }
}