<?php
/**
 * Media Helper for Visforms
 *
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2012 vi-solutions
 * @since        Joomla 1.6
 */
namespace Visolutions\Component\Visforms\Administrator\Helper;

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\Filesystem\Path;
use Joomla\Registry\Registry;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Client\ClientHelper;
use Joomla\CMS\Helper\MediaHelper as JoomlaMediaHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\Filesystem\Exception\FilesystemException;

class MediaHelper {

    public static $savemodehigh = 0;
    public static $savemodelow = 1;

    private static function canUpload($file, &$err, $maxfilesize, $allowedextensions): bool {
        if (empty($file['name'])) {
            $err = 'COM_VISFORMS_ERROR_UPLOAD_INPUT';
            return false;
        }
        if (str_replace(' ', '', $file['name']) != $file['name'] || $file['name'] !== File::makeSafe($file['name'])) {
            $err = 'COM_VISFORMS_ERROR_WARNFILENAME';
            return false;
        }
        $filetypes = explode('.', $file['name']);
        if (count($filetypes) < 2) {
            // There seems to be no extension
            $err = 'COM_VISFORMS_ERROR_WARNFILETYPE';
            return false;
        }
        $format = strtolower(File::getExt($file['name']));
        $allowable = explode(',', $allowedextensions);
        if ($format == '' || $format == false || (!in_array($format, $allowable))) {
            $err = 'COM_VISFORMS_ERROR_WARNFILETYPE';
            return false;
        }
        $maxSize = (int)($maxfilesize);
        if (($file['error'] == 1)
            || ($maxSize > 0 && (int)$file['size'] > $maxSize)) {
            $err = 'COM_VISFORMS_ERROR_WARNFILETOOLARGE';
            return false;
        }
        $images = explode(',', "bmp,gif,jpg,jpeg,png");
        if (in_array($format, $images)) { // if its an image run it through getimagesize
            // if tmp_name is empty, then the file was bigger than the PHP limit
            if (!empty($file['tmp_name'])) {
                if (($imginfo = getimagesize($file['tmp_name'])) === FALSE) {
                    $err = 'COM_VISFORMS_ERROR_WARNINVALID_IMG';
                    return false;
                }
            }
            else {
                $err = 'COM_VISFORMS_ERROR_WARNFILETOOLARGE';
                return false;
            }
        }
        $xss_check = file_get_contents($file['tmp_name'], false, null, -1, 256);
        $html_tags = array('abbr', 'acronym', 'address', 'applet', 'area', 'audioscope', 'base', 'basefont', 'bdo', 'bgsound', 'big', 'blackface', 'blink', 'blockquote', 'body', 'bq', 'br', 'button', 'caption', 'center', 'cite', 'code', 'col', 'colgroup', 'comment', 'custom', 'dd', 'del', 'dfn', 'dir', 'div', 'dl', 'dt', 'em', 'embed', 'fieldset', 'fn', 'font', 'form', 'frame', 'frameset', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'head', 'hr', 'html', 'iframe', 'ilayer', 'img', 'input', 'ins', 'isindex', 'keygen', 'kbd', 'label', 'layer', 'legend', 'li', 'limittext', 'link', 'listing', 'map', 'marquee', 'menu', 'meta', 'multicol', 'nobr', 'noembed', 'noframes', 'noscript', 'nosmartquotes', 'object', 'ol', 'optgroup', 'option', 'param', 'plaintext', 'pre', 'rt', 'ruby', 's', 'samp', 'script', 'select', 'server', 'shadow', 'sidebar', 'small', 'spacer', 'span', 'strike', 'strong', 'style', 'sub', 'sup', 'table', 'tbody', 'td', 'textarea', 'tfoot', 'th', 'thead', 'title', 'tr', 'tt', 'ul', 'var', 'wbr', 'xml', 'xmp', '!DOCTYPE', '!--');
        foreach ($html_tags as $tag) {
            // A tag is '<tagname ', so we need to add < and a space or '<tagname>'
            if (stristr($xss_check, '<' . $tag . ' ') || stristr($xss_check, '<' . $tag . '>')) {
                $err = 'COM_VISFORMS_ERROR_WARNIEXSS';
                return false;
            }
        }
        return true;
    }

    // throws RuntimeException with error message
    public static function uploadFiles($visform, $client = 'site') {
        $input = Factory::getApplication()->getInput();
        $files = $input->files->get('jform', array(), 'array');
        $savemode = (empty($visform->savemode)) ? 'cmd' : 'raw';
        // upload files
        $n = count($visform->fields);
        for ($i = 0; $i < $n; $i++) {
            $field = $visform->fields[$i];
            // Request has an fileupload with values
            if ((($client == 'site') && ($field->typefield == 'file') && (isset($_FILES[$field->name]['name'])) && ($_FILES[$field->name]['name'] != ''))
                || (($client == 'admin') && ($field->typefield == 'file') && (isset($_FILES['jform']['name'][$field->name])) && ($_FILES['jform']['name'][$field->name] != ''))) {
                // only upload if field is not disabled
                if (!isset($field->isDisabled) || ($field->isDisabled == false)) {
                    $file = ($client == 'site') ? $input->files->get($field->name, '', $savemode) : $files[$field->name];
                    $allowedextensions = (!empty($field->allowedextensions)) ? $field->allowedextensions : $visform->allowedextensions;
                    // minimum from php ini and Visforms form settings
                    $maxfilesize = self::getMaxFileSizeInByte($field->maxfilesize);
                    if (empty($file)) {
                        throw new \RuntimeException(Text::_('COM_VISFORMS_FILE_NOT_SAFE'));
                    }
                    $folder = (!empty($visform->uploadpath)) ? $visform->uploadpath : '';
                    try {
                        $path = Path::clean(JPATH_ROOT . '/' . $folder);
                        if (!file_exists($path)) {
                            throw new \RuntimeException(Text::_('COM_VISFORMS_UPLOAD_DIRECTORY_DOES_NOT_EXIST'));
                        }
                    }
                    catch (\UnexpectedValueException $e) {
                        throw new \RuntimeException(Text::_('COM_VISFORMS_UPLOAD_DIRECTORY_DOES_NOT_EXIST'));
                    }
                    // Set FTP credentials, if given
                    ClientHelper::setCredentialsFromRequest('ftp');

                    // Make the filename safe
                    $file['name_org'] = $file['name'];
                    $file['name'] = File::makeSafe($file['name']);
                    $file['name'] = str_replace(' ', '-', $file['name']);
                    // Check upload conditions
                    $err = null;
                    if (!self::canUpload($file, $err, $maxfilesize, $allowedextensions)) {
                        // The file can't be upload
                        throw new \RuntimeException(Text::sprintf($err, $file['name_org'], $maxfilesize) . ' (' . $field->label . ')');
                    }
                    else {
                        // get a unique id to rename the uploaded file
                        $fileuid = uniqid('');

                        // rename file
                        $pathInf = pathinfo($file['name']);
                        $ext = $pathInf['extension'];
                        $file['new_name'] = basename($file['name'], "." . $ext) . "_" . $fileuid . "." . $ext;
                        $file['new_name'] = strtolower($file['new_name']);

                        // get complete upload path with filename of renamed file
                        $filepath = Path::clean(JPATH_SITE . DIRECTORY_SEPARATOR . $folder . DIRECTORY_SEPARATOR . $file['new_name']);
                        $file['filepath'] = $filepath;
                        $file['filelink'] = Uri::base() . $folder . '/' . $file['new_name'];


                        // try to upload file
                        if (file_exists($file['filepath'])) {
                            // File exists
                            throw new \RuntimeException(Text::sprintf('COM_VISFORMS_ERROR_FILE_EXISTS', $file['name_org']));
                        }
                        else {
                            try {
                                File::upload($file['tmp_name'], $file['filepath'], false);
                            }
                            catch (FilesystemException $e) {
                                // Error in upload
                                throw new \RuntimeException(Text::sprintf('COM_VISFORMS_ERROR_UNABLE_TO_UPLOAD_FILE', $file['name_org']));
                            }
                        }
                    }
                    foreach ($file as $name => $value) {
                        $visform->fields[$i]->file[$name] = $value;
                    }
                    $dbValue = new \stdClass();
                    $dbValue->folder = $folder;
                    $dbValue->file = $field->file['new_name'];
                    $registry = new Registry($dbValue);
                    $visform->fields[$i]->dbValue = $registry->toString();
                }
            }
        }
    }

    public static function deletefile($path):bool {
        try {
            File::delete(JPATH_ROOT . '/' . $path);
            return true;
        }
        catch (FilesystemException $e) {
            return false;
        }
    }

    public static function copyfile($name, $oldpath, $restore) {
        $newpath = JPATH_ROOT . '/images/visforms_save';
        $orgpath = JPATH_ROOT . '/' . $oldpath;
        if (empty($restore)) {
            $fileexists = file_exists($newpath);
            if (empty($fileexists)) {
                Folder::create($newpath);
            }
            try {
                File::copy($orgpath, $newpath . '/' . $name);
                return true;
            }
            catch (FilesystemException $e) {
                return false;
            }
            catch (\UnexpectedValueException $e) {
                return false;
            }
        }
        else {
            try {
                File::move($newpath . '/' . $name, $orgpath);
                return true;
            }
            catch (FilesystemException $e) {
                return false;
            }
            catch (\UnexpectedValueException $e) {
                return false;
            }
        }
    }

    public static function isImage($file): bool {
        if (empty($file)) {
            return false;
        }
        $saveFile = File::makeSafe($file);
        if (empty($saveFile)) {
            return false;
        }
        $ext = strtolower(File::getExt($saveFile));
        $imageExts = array('jpg', 'png', 'gif', 'bmp', 'jpeg');
        if ((!empty($ext)) && (in_array($ext, $imageExts))) {
            return true;
        }
        return false;
    }

    public static function getFileInfo($dbValue) {
        $file = new \stdClass();
        $file->name = HTMLHelper::_('visforms.getUploadFileName', $dbValue);
        $file->path = HTMLHelper::_('visforms.getUploadFilePath', $dbValue);
        $file->filepath = (!empty($file->path)) ? Path::clean(JPATH_SITE . DIRECTORY_SEPARATOR . $file->path) : '';
        $file->link = HTMLHelper::_('visforms.getUploadFileLink', $dbValue);
        if (empty($file->name) || empty($file->path)) {
            return false;
        }
        $file->isimage = self::isImage($file->name);
        if (!empty($file->isimage)) {
            $info = @getimagesize(JPATH_SITE . '/' . $file->path);
            if ($info) {
                $file->width = @$info[0];
                $file->height = @$info[1];
                if (($info[0] > 60) || ($info[1] > 60)) {
                    $dimensions = JoomlaMediaHelper::imageResize($file->width, $file->height, 60);
                    $file->width = $dimensions[0];
                    $file->height = $dimensions[1];
                }
                if ((!empty($file->width)) && (!empty($file->height))) {
                    $file->displayDimension = 'width="' . $file->width . '" height="' . $file->height . '" ';
                }
            }
        }

        return $file;
    }

     // Small helper function that properly converts any configuration options to their byte representation.
    public static function toBytes($val) {
        switch ($val[strlen($val) - 1]) {
            case 'M':
            case 'm':
                return (int)$val * 1048576;
            case 'K':
            case 'k':
                return (int)$val * 1024;
            case 'G':
            case 'g':
                return (int)$val * 1073741824;
            default:
                return $val;
        }
    }

    public static function getMaxFileSizeInByte($value) {
        $uploadMaxFileSize = self::toBytes(ini_get('upload_max_filesize'));
        return (((int) $uploadMaxFileSize > 0) && (((int) $value === 0) || ($value * 1024) > $uploadMaxFileSize)) ? $uploadMaxFileSize : $value * 1024;
    }
}