<?php
/**
 * Visforms field checkbox class
 *
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2012 vi-solutions
 * @since        Joomla 1.6
 */
namespace  Visolutions\Component\Visforms\Site\Lib\Field;

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Filter\InputFilter;
use Visolutions\Component\Visforms\Site\Event\Visforms\VisfieldBeforeCreateEvent;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\UserHelper;

class CheckboxField extends Field
{
	public function __construct($field, $form) {
		parent::__construct($field, $form);
		// store potential query Values for this field in the session
		$this->setQueryValue();
		// unchecked checkboxes are not send with post
		// checkboxes always have a value different from empty string (enforced in field configuration)
		// we set unchecked checkboxes in postValue with empty string
		$this->postValue = $this->input->post->get($field->name, '', 'STRING');
	}

	protected function setField(): void {
		// preprocessing field
		$this->extractDefaultValueParams();
		$this->extractGridSizesParams();
		$this->extractRestrictions();
		// perform before mending boolean attributes
        $this->setCheckedFromUserFields();
		$this->mendBooleanAttribs();
        $this->unsetDeselectedEditViewValidations();
        $this->setCustomPhpErrorMsg();
        PluginHelper::importPlugin('visforms');
        $dispatcher = Factory::getApplication()->getDispatcher();
        $event = new VisfieldBeforeCreateEvent('onVisfieldBeforeCreate', [
            'subject' => $this->field,
            'context' => 'com_visforms.field',
            'form' => $this->form
        ]);
        $dispatcher->dispatch('onVisfieldBeforeCreate', $event);
		$this->setIsConditional();
		$this->setIsDisplayChanger();
		$this->removeInvalidQueryValues();
		$this->setEditValue();
		$this->setConfigurationDefault();
		$this->setEditOnlyFieldDbValue();
		$this->setFieldDefaultValue();
		$this->setDbValue();
		$this->setRedirectParam();
		$this->setCustomJs();
		$this->setFieldsetCounter();
		$this->mendInvalidUncheckedValue();
		$this->setEnterKeyAction();
		$this->setShowRequiredAsterix();
        $this->replaceLabelWithCustomText();
	}

	protected function setFieldDefaultValue(): void {
		// we never change the attribute value of a checkbox, because this is a fixed value, defined in the field configuration
		// we always set/unset the attribute_checked!
		$field = $this->field;
		if ($this->input->getCmd('task', '') == 'editdata') {
			if (isset($this->field->editValue)) {
			    // we do not check if the stored value matches the the attribute_value
                // attribute value can be changed in the field configuration
                // but if a value is stored for this field this means, that the user has checked the checkbox in the first place.
				if ($this->field->editValue !== '') {
					$this->field->attribute_checked = "checked";
				}
				else {
					// checkbox is not checked
					if (property_exists($this->field, 'attribute_checked')) {
						unset($this->field->attribute_checked);
					}
				}
			}
			// else use configuration defaults
			$this->field->dataSource = 'db';
			return;
		}
		// if we have a POST Value, we use this
		if ((count($_POST) > 0) && isset($_POST['postid']) && ($_POST['postid'] == $this->form->id)) {
			$valid = $this->validateUserInput('postValue');
			// Checkbox was checked, is submitted value correct?
			if ((isset($_POST[$field->name]))) {
				if ($valid === true) {
					if ($this->postValue !== "") {
						$this->field->attribute_checked = "checked";
					}
					else {
						// actually when a $_POST is set, "" is not a valid value therefore we uncheck the box (just as with an invalid value)
						if (property_exists($this->field, 'attribute_checked')) {
							unset($this->field->attribute_checked);
						}
					}
				}
				else {
					if (property_exists($this->field, 'attribute_checked')) {
						unset($this->field->attribute_checked);
					}
				}
			}
			else {
				// checkbox was not submitted with the post (either disabled field or really not checked),
				// we uncheck the field
				// this is necessary because checkboxes can trigger conditional fields if they are checked
				// but in both cases (disabled or really unchecked) the checkbox must not trigger, the display of a depending conditional field
				if (property_exists($this->field, 'attribute_checked')) {
					unset($this->field->attribute_checked);
				}
			}
			// keep configuration default values
			$this->field->dataSource = 'post';
			return;
		}

		// if we have a GET Value and field may use GET values, we use this
		if (isset($field->allowurlparam) && ($field->allowurlparam == true)) {
			$urlparams = Factory::getApplication()->getUserState('com_visforms.urlparams.' . $this->form->context, null);
			if (!empty($urlparams) && (is_array($urlparams)) && (isset($urlparams[$this->field->name]))) {
				$queryValue = $urlparams[$this->field->name];
			}
			if (isset($queryValue)) {
				// only use a query value if it exists; This value is already validated!
				if ($queryValue === "") {
					// checkbox is not checked
					if (property_exists($this->field, 'attribute_checked')) {
						unset($this->field->attribute_checked);
					}
				}
				else {
					$this->field->attribute_checked = "checked";
				}
			}

			$this->field->dataSource = 'query';
			return;
		}
		// Nothing to do
		return;
	}

	protected function setDbValue(): void {
		if (isset($this->field->dataSource) && $this->field->dataSource == 'post') {
			$this->field->dbValue = $this->postValue;
		}
	}

	protected function setEditOnlyFieldDbValue(): void {
		if ($this->field->configurationDefault === "checked") {
			$this->field->editOnlyFieldDbValue = $this->field->attribute_value;
		}
	}

	protected function validateUserInput($inputType) {
		// value set by user
		$value = $this->$inputType;

		// Some empty values are valid but 0 is not
		if ((!isset($value)) || ($value === '')) {
			return true;
		}

		// is there a value set by user which is not allowed?
		if ($value !== $this->field->attribute_value) {
			// we have an invalid user input
			$this->field->isValid = false;
			// attach error to form
			$error = Text::sprintf('COM_VISFORMS_OPTION_HAS_INVALID_POST_VALUE', $this->field->label);
			$this->setErrorMessageInForm($error);
			// remove value from $this->$inputType
			$this->$inputType = "";
			return false;
		}

		return true;
	}

	protected function setRedirectParam(): void {
		if (isset($this->field->dataSource) && $this->field->dataSource == 'post' && (!empty($this->field->addtoredirecturl))) {
			// checkbox was not checked
			if (empty($this->postValue)) {
				return;
			}
			$this->field->redirectParam = $this->postValue;
		}
	}

	protected function removeInvalidQueryValues() {
		$app = Factory::getApplication();
		$urlparams = $app->getUserState('com_visforms.urlparams.' . $this->form->context);
		if (empty($urlparams) || !is_array($urlparams) || !isset($urlparams[$this->field->name])) {
			return;
		}
		$queryValue = $urlparams[$this->field->name];
		// empty string is a valid value (= field value is not set)
		if (($queryValue !== '')) {
			if ($queryValue !== $this->field->attribute_value) {
				//remove invalid queryValue ulrparams array and set urlparams to Null if the array is empty
				unset($urlparams[$this->field->name]);
				if (!(count($urlparams) > 0)) {
					$urlparams = null;
				}
				$app->setUserState('com_visforms.urlparams.' . $this->form->context, $urlparams);
			}
		}
	}

	protected function setConfigurationDefault(): void {
		$this->field->configurationDefault = isset($this->field->attribute_checked) ? (string) "checked" : (string) "";
		if (!$this->isEditOrSaveEditTask) {
			$urlparams = Factory::getApplication()->getUserState('com_visforms.urlparams.' . $this->form->context, null);
			if (!empty($urlparams) && (is_array($urlparams)) && (isset($urlparams[$this->field->name]))) {
				$queryValue = $urlparams[$this->field->name];
			}
			// if form was originally called with valid url params, reset to this url params
			if (isset($this->field->allowurlparam) && ($this->field->allowurlparam == true) && isset($queryValue)) {
				$this->field->configurationDefault = ($queryValue !== "") ? "checked" : "";
			}
		}
	}

	protected function setEditValue(): void {
		if ($this->isEditOrSaveEditTask) {
			$this->field->editValue = "";
			$data = $this->form->data;
			$dataFieldName = $this->getParameterFieldNameForEditValue();
			if (isset($data->$dataFieldName)) {
				$filter = InputFilter::getInstance();
				$this->field->editValue = $filter->clean($data->$dataFieldName, 'STRING');
			}
			$this->field->editValueChecked = ($this->field->editValue === $this->field->attribute_value) ? "checked" : "";
		}
	}

	// attribute_checked may contain a reference to a Joomla user field
    // check if this Joomla user field contains a value, which matches the attribute_value for this checkbox
    // if true: set attribute_checked to checked
    // if false: unset attribute_checked
    // note: the value of attribute_checked may be changed later, if other conditions require another state
	protected function setCheckedFromUserFields() {
        if ($this->isEditOrSaveEditTask) {
            return;
        }
	    if (isset($this->field->attribute_checked) && $this->field->attribute_checked == 'dynamic')  {
	        if (!empty($this->field->checkedDynamic)) {
	            $checkedDynamic = $this->field->checkedDynamic;
                if ($checkedDynamic == 'url') {
                    $value = Uri::getInstance()->toString();;
                    if ($value === (string)$this->field->attribute_value) {
                        $this->field->attribute_checked = "checked";
                        return;
                    }
                }
                $user = Factory::getApplication()->getIdentity();
                $userId = $user->id;
                if ($userId != 0) {
                    if ($checkedDynamic == "2") {
                        $value  = $user->name;
                        if ((string) $value === (string) $this->field->attribute_value) {
                            $this->field->attribute_checked = "checked";
                            return;
                        }
                    }
                    if ($checkedDynamic == "3") {
                        $value = $user->username;
                        if ((string) $value === (string) $this->field->attribute_value) {
                            $this->field->attribute_checked = "checked";
                            return;
                        }
                    }
                    if ($checkedDynamic == 'usermail') {
                        $value = $user->email;
                        if ((string) $value === (string) $this->field->attribute_value) {
                            $this->field->attribute_checked = "checked";
                            return;
                        }
                    }
                    preg_match('/^CF(\d+)$/', $checkedDynamic, $matches);
                    if ($matches && !empty($matches[1])) {
                        $customfieldid = $matches[1];
                        $values = HTMLHelper::_('visforms.getCustomUserFieldValue', $customfieldid, $user, 'loadColumn');
                        if ($values && is_array($values)) {
                            foreach ($values as $value) {
                                // empty values in CF are not stored in the database
                                // so value is never null or ''
                                $type = HTMLHelper::_('visforms.getCustomUserFieldType', $customfieldid);
                                // remove potential time, format
                                if ($type === 'calendar') {
                                    $value = HTMLHelper::_('date', $value, 'DATE_FORMAT_LC4');
                                }
                                if ((string) $value === (string) $this->field->attribute_value) {
                                    $this->field->attribute_checked = "checked" ;
                                    return;
                                }
                            }
                        }
                    }
                    $userProfile = UserHelper::getProfile($userId);
                    if ((!(empty($userProfile->profile))) && (is_array($userProfile->profile))) {
                        // if a user profile value is not set, the database value is ''
                        if (!(empty($userProfile->profile[$checkedDynamic]))) {
                            $value = $userProfile->profile[$checkedDynamic];
                            // remove potential time, format
                            if ($checkedDynamic === 'dob') {
                                $value = HTMLHelper::_('date', $value, 'DATE_FORMAT_LC4');
                            }
                            if ((string) $value === (string) $this->field->attribute_value) {
                                $this->field->attribute_checked = "checked" ;
                                return;
                            }
                        }
                    }
                }
	            // no match found. Do not check.
                unset($this->field->attribute_checked);
            }
	        // No dynamic option selected. Do not check.
            unset($this->field->attribute_checked);
        }
    }
}