<?php
/**
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2018 vi-solutions
 */

namespace Visolutions\Component\Visforms\Site\Model\Helper\Mail;

// no direct access
defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Mail\MailerFactoryInterface;
use Joomla\Database\DatabaseInterface;
use Visolutions\Component\Visforms\Site\Log\VisformsLogTrait;
use Visolutions\Component\Visforms\Site\Lib\Validation\EmailValidation;

class Verification {

    use VisformsLogTrait;

	private $code;
	private $verificationMailAddr;
	private $formId;
	private $mailFrom;
	private $mailFromName;
	private $db;

	public function __construct($mailAddr = null) {
	    $app = Factory::getApplication();
	    $input = $app->getInput();
		$verificationMail = (is_null($mailAddr)) ? $input->post->get('verificationAddr', '', 'STRING') : $mailAddr;
		$verification = new EmailValidation(array('value' => $verificationMail));
		$valid = $verification->validate();
		$this->verificationMailAddr = (empty($valid)) ? '' : $verificationMail;
		// Ajax Validation => fid, Form Submit => postid
		$this->formId = $input->post->get('fid', $input->post->get('postid', 0, 'int'), 'int');
		$config = $app->getConfig();
		$this->mailFrom = $config->get('mailfrom', '');
		$this->mailFromName = $config->get('fromname', '');
		$this->code = $this->createVerificationCode();
		$this->db = Factory::getContainer()->get(DatabaseInterface::class);
        $this->initializeLogger('visformssubmit');
	}

	public function sendVerificationMail() {
		if (empty($this->verificationMailAddr)) {
			return Text::_('COM_VISFORMS_EMAIL_VERIFICATION_EMAIL_REQUIRED');
		}
		if (empty($this->code)) {
			return Text::_('COM_VISFORMS_EMAIL_VERIFICATION_MAIL_NO_CODE');
		}
		if (empty($this->mailFrom)) {
			return Text::_('COM_VISFORMS_EMAIL_VERIFICATION_MAIL_NO_SENDER');
		}
		$codeStored = $this->storeVerificationCode();
		if (empty($codeStored)) {
			return Text::_('COM_VISFORMS_EMAIL_VERIFICATION_CANNOT_STORE_CODE');
		}
		try {
			$mail = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
			$mail->CharSet = "utf-8";
			$mail->addRecipient(explode(",", $this->verificationMailAddr));
			$mail->setSender($this->mailFrom);
			$mail->setSubject(Text::_('COM_VISFORMS_VERIFICATIONCODE') . ' ' . $this->mailFromName);
			$mail->IsHTML(true);
			$mail->Encoding = 'base64';
			$mail->setBody(Text::_('COM_VISFORMS_VERIFICATIONCODE_IS') . ' ' . $this->code);
			$sent = $mail->Send();
			if (!empty($sent)) {
				$this->removeExpiredCodes();
				return Text::_('COM_VISFORMS_EMAIL_VERIFICATION_MAIL_SEND');
			}
			else {
				$this->removeExpiredCodes();
				$this->removeVerifactionCode($codeStored);
				return Text::_('COM_VISFORMS_EMAIL_VERIFICATION_MAIL_SEND_PROBLEMS');
			}
		}
		catch (\Exception $e) {
			$this->removeExpiredCodes();
			$this->removeVerifactionCode($codeStored);
            $this->addLogEntry(Text::_('COM_VISFORMS_EMAIL_VERIFICATION_MAIL_SEND_PROBLEMS') . ': ' . $e->getMessage(), Log::ERROR);
			return Text::_('COM_VISFORMS_EMAIL_VERIFICATION_MAIL_SEND_PROBLEMS');
		}
	}

	protected function createVerificationCode() {
		if (empty($this->verificationMailAddr) || strlen($this->verificationMailAddr) < 4) {
			return '';
		}
		$code = str_replace('@', '', $this->verificationMailAddr);
		$code = str_replace('.', '', $code);
		$code = str_shuffle($code);
		$code = strtoupper(substr($code, 0, min(strlen($code), 4)));
		return $code;
	}

	public function getStoredCodes() {
		$db = $this->db;
		$query = $db->createQuery();
		$query->select('*')
			->from($db->qn('#__visverificationcodes'))
			->where($db->qn('email') . ' = ' . $db->q($this->verificationMailAddr))
			->where($db->qn('fid') . ' = ' . $db->q($this->formId));
		try {
			$db->setQuery($query);
			return $db->loadObjectList();
		}
		catch (\RuntimeException $e) {
			return false;
		}
	}

	protected function storeVerificationCode() {
		$return = true;
		// check if there are already codes stored for this mail address
		$storedCodes = $this->getStoredCodes();
		if (!empty($storedCodes) && is_array($storedCodes)) {
			// get first element and update it with new code
			$storedCode = array_shift($storedCodes);
			$storedCode->code = $this->code;
			$storedCode->created = Factory::getDate()->toSql();
			try {
				$this->db->updateObject('#__visverificationcodes', $storedCode, 'id');
				$return = $storedCode->id;
			}
			catch (\RuntimeException $e) {
				$return = false;
			}
		}
		else {
			$storedCode = new \stdClass();
			$storedCode->fid = $this->formId;
			$storedCode->email = $this->verificationMailAddr;
			$storedCode->code = $this->code;
			$storedCode->created = Factory::getDate()->toSql();
			try {
				$this->db->insertObject('#__visverificationcodes', $storedCode);
				$storedCode->id = $this->db->insertid();
				$return = $storedCode->id;
			}
			catch (\RuntimeException $e) {
				$return = false;
			}
		}
		unset($storedCode);
		if (!empty($storedCodes) && is_array($storedCodes)) {
			// database is not clean: there was more than one code stored for the mail address: remove them
			foreach ($storedCodes as $storedCode) {
				$this->removeVerifactionCode($storedCode->id);
			}
		}
		return $return;
	}

	protected function removeVerifactionCode($id) {
		$db = $this->db;
		$query = $db->createQuery();
		$query->delete($db->qn('#__visverificationcodes'))
			->where($db->qn('id') . ' = ' . $id);
		try {
			$db->setQuery($query);
			$db->execute();
			return true;
		}
		catch (\RuntimeException $e) {
			return false;
		}
	}

	protected function removeExpiredCodes() {
		$db = $this->db;
		$query = $db->createQuery();
		$query->delete($db->qn('#__visverificationcodes'))
			->where($db->qn('created') . ' < NOW() - INTERVAL 1 WEEK ');
		try {
			$db->setQuery($query);
			return $db->execute();
		}
		catch (\RuntimeException $e) {
			return false;
		}
	}
}