<?php
/**
 * Visdata controller for Visforms
 *
 * @author       Aicha Vack
 * @package      Joomla.Site
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2012 vi-solutions
 * @since        Joomla 1.6 
 */

namespace Visolutions\Component\Visforms\Site\Controller;

// Check to ensure this file is included in Joomla!
defined('_JEXEC') or die();

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;
use Joomla\Input\Input;
use Joomla\Utilities\ArrayHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Visolutions\Component\Visforms\Administrator\Helper\AefHelper;
use Visolutions\Plugin\Content\Vfdataview\Helper\Vfdataviewmodel;
use Visolutions\Component\Visforms\Administrator\Helper\CsvExportHelper;
use Visolutions\Component\Visforms\Administrator\Helper\VisformsHelper;

class VisformsdataController extends DisplayController
{
	protected $fid;
	protected $canDo;
	protected $parentViewReturnUrl;
	protected $pdfData;
	protected int $tmplId;

	public function __construct($config = array(), ?MVCFactoryInterface $factory = null, ?CMSApplicationInterface $app = null, ?Input $input = null) {
		parent::__construct($config, $factory, $app, $input);
		$this->registerTask('unpublish', 'publish');
		$this->setParentViewReturnUrl();
		$this->setFid();
		$this->canDo = VisformsHelper::getActions($this->fid);
		$this->registerTask('renderPdfList', 'renderPdf');
	}

	// todo could be moved in a shared controller
	protected function setParentViewReturnUrl() {
		$return = $this->input->get('return', '');
		$this->parentViewReturnUrl = (!empty($return)) ? HTMLHelper::_('visforms.base64_url_decode', $return) : Uri::base();
	}

	public function publish() {
		// Check for request forgeries.
		Session::checkToken() or jexit(Text::_('JINVALID_TOKEN'));
        $fid = $this->fid;
        //VisformsTableVisdata expects the parameter fid
        $this->setInputFid();
        $pk = $this->input->get('cid', null, 'array');
        //this function can be called from different views, return to return url, if a return param is set in input
	    $dataViewMenuItemExists = HTMLHelper::_('visforms.checkDataViewMenuItemExists', $fid);
        $mysubmenuexists = HTMLHelper::_('visforms.checkMySubmissionsMenuItemExists');
        if (!($dataViewMenuItemExists) && !($mysubmenuexists)) {
			$this->setRedirect($this->parentViewReturnUrl, Text::_('JERROR_ALERTNOAUTHOR'), 'error');
            return false;
        }
        $success = false;
        // Make sure the item ids are integers
		ArrayHelper::toInteger($pk);
        $data = array('publish' => 1, 'unpublish' => 0);
        $task = $this->getTask();
		$value = ArrayHelper::getValue($data, $task, 0, 'int');
        //check for permission
		if ($this->canDo->get('core.edit.data.state') || $this->canDo->get('core.edit.own.data.state')) {
			if (!empty($pk)) {
				$model = $this->getModel('Visdata', 'Administrator');
				try {
					if ($model->publish($pk, $value)) {
						if ($value === 1) {
							$this->setMessage(Text::_('COM_VISFORMS_RECORDSET_PUBLISHED'));
						}
						elseif ($value === 0) {
							$this->setMessage(Text::_('COM_VISFORMS_RECORDSET_UNPUBLISHED'));
						}
						$success = true;
					}
				}
				catch (\Exception $e) {
					$this->setMessage($e->getMessage(), 'error');
				}
			}
			else {
				$success = false;
			}
		}
		else {
			$this->setMessage(Text::_('COM_VISFORMS_NO_PUBLISH_AUTHOR'), 'error');
			$success = false;
		}
		$this->setRedirect($this->parentViewReturnUrl);
		return $success;
	}

	// Download PDF documents from frontend data view
	public function renderPdf() {
		$this->checkToken();
        if (!AefHelper::checkAEF()) {
            $this->setMessage(Text::_('COM_VISFORMS_EXPORT_PDF_NO_SUBSCRIPTION'), 'error');
            $this->setRedirect($this->parentViewReturnUrl);
            return false;
        }
		$task = $this->getTask();
		$context = $this->input->get('context', '', 'cmd');
		$mid = $this->input->get('mid', 0, 'int');
		if ($task == 'renderPdfList') {
            // hdnpparams is only passed with the post, if we come from the content plugin data view
            // it is an url encoded string of all processed (cleaned) plugin params
            // pass in to model
            // necessary in order to be able to use the original fieldselect options
            $pparams = $this->input->get('hdnpparams', '', 'cmd');
            if (!empty($pparams)) {
                $pparams = $this->cleanPlgParamsFromRequest(VisformsHelper::registryArrayFromString(HTMLHelper::_('visforms.base64_url_decode', $pparams)));
            }
            if (!empty($pparams)) {
                $dataModel = $this->getModel('Visformsdata', 'Site', array('context' => $context, 'mid' => $mid, 'pparams' => $pparams));
            }
            else {
                $dataModel = $this->getModel('Visformsdata', 'Site', array('context' => $context, 'mid' => $mid));
            }
			// trigger the populateState function in model
			$dataModel->getState('list.limit');
			// override list start and list limit
            // in order to get all records
			$dataModel->setState('list.limit', 0);
			$dataModel->setState('list.start', 0);
			$this->pdfData = $dataModel->getRawItems();
		}
		else {
            $dataModel = $this->getModel('Visformsdata', 'Site', array('context' => $context, 'mid' => $mid));
		    // create pdf of a single record
			// $this->pdfData is expected to be an array of stdClass
			$data = $dataModel->getDetail();
			if (!empty($data)) {
				$this->pdfData = array($data);
			}
		}
		$this->setPdfTemplate();
		// user has no permission to create any pdf from this template
		if (!$this->canCreatePdf()) {
            $this->setMessage(Text::_('COM_VISFORMS_EXPORT_PDF_NO_PERMISSION'), 'error');
            $this->setRedirect($this->parentViewReturnUrl);
            return false;
        }
		$this->removeForbiddenData();
		return $this->renderPdfDocument();
	}

	// create a pdf document of the data of the current submission
    // we do not check user permission in this case, because we are still within the form submit process
    // working only with the temporary request data which are stored in the session and which were submitted by the current user (guest)
    // checking permission would require, that the 'can create pdf' permission is granted if a guest should be able to download the pdf in this case
    // con create own pdf does not work for guests
    // as a consequence in any data view any user could download a pdf with data of all submitted records they can see
    // the feature 'Download PDF Link in result message' would be quit useless
	public function renderPdfFromRequestData() {
		$this->checkToken('get');
        if (!AefHelper::checkAEF()) {
            $this->setMessage(Text::_('COM_VISFORMS_EXPORT_PDF_NO_SUBSCRIPTION'), 'error');
            $this->setRedirect($this->parentViewReturnUrl);
            return false;
        }
		$visformObject = Factory::getApplication()->getUserState('visforms'. $this->fid . '.pdf.requestdatas', null);
		// we hove only one data record in the data pdfData array
		$this->pdfData = array($this->createDataObjectFromRequestData($visformObject));
        $this->setPdfTemplate();
		return $this->renderPdfDocument();
	}

	public function exportCsv() {
		$this->checkToken();
		$this->setRedirect($this->parentViewReturnUrl);
		if (!$this->canDo->get('core.export.data.frontend') || empty(AefHelper::checkAEF())) {
			$this->setMessage(Text::_('COM_VISFORMS_EXPORT_NOT_PERMITTED'), 'error');
			return false;
		}
		$context = $this->input->get('context', '', 'cmd');
		// we have to get the menu id of the menu item, from which the request was send
        // in order to create the correct pagination context string in the data view model
        // the menu item is passed as 'mid' post parameter
        // it is only used in the data view model, if the value is not 0
		$mid = $this->input->get('mid', 0, 'int');
        // hdnpparams is only passed with the post, if we come from the content plugin data view
        // it is an url encoded string of all processed (cleaned) plugin params
		// pass in to model
        // necessary in order to be able to use the original fieldselect options
        $pparams = $this->input->get('hdnpparams', '', 'cmd');
        if (!empty($pparams)) {
            $pparams = $this->cleanPlgParamsFromRequest(VisformsHelper::registryArrayFromString(HTMLHelper::_('visforms.base64_url_decode', $pparams)));
        }
        if (!empty($pparams)) {
            $dataModel = $this->getModel('Visformsdata', 'Site', array('context' => $context, 'mid' => $mid, 'pparams' => $pparams));
        }
        else {
            $dataModel = $this->getModel('Visformsdata', 'Site', array('context' => $context, 'mid' => $mid));
        }
		// trigger the populateState function in model
		$dataModel->getState('list.limit');
		// override list start and list limit
		$dataModel->setState('list.limit', 0);
		$dataModel->setState('list.start', 0);
		$items = $dataModel->getRawItems();
		if (empty($items)) {
			$this->setMessage(Text::_('COM_VISFORMS_NO_ITEM_SELECTED'), 'error');
			return false;
		}
		$exportHelper = new CsvExportHelper($this->fid, null, $items);
		$buffer = $exportHelper->createExportBuffer();
		if (empty($buffer)) {
			return false;
		}
		$defaultFileName = "visforms_" . date("Ymd");
		$fileName = $exportHelper->getExportFileName($defaultFileName);

		header("Expires: Sun, 1 Jan 2000 12:00:00 GMT");
		header("Last-Modified: " . gmdate("D, d M Y H:i:s") . "GMT");
		header("Cache-Control: no-store, no-cache, must-revalidate");
		header("Cache-Control: post-check=0, pre-check=0", false);
		header("Pragma: no-cache");
		header("Content-type: application/vnd.ms-excel");
		header("Content-disposition: attachment; filename=" . $fileName);
		echo $buffer;
		Factory::getApplication()->close();
	}

	protected function setPdfTemplate() {
		$task = $this->getTask();
		$model = $this->getModel('Visform', 'Administrator');
		$form = $model->getItem();
		// get tmplId from request if given
        $tmplId = $this->input->get('tmplid', 0, 'int');
		switch ($task) {
			case 'renderPdfList' :
                $this->tmplId = (!empty($form->frontendsettings['listPdfTemplate']) && in_array($tmplId, $form->frontendsettings['listPdfTemplate'])) ? $tmplId : 0;
                break;
			case 'renderPdfFromRequestData':
                $this->tmplId = (!empty($form->subredirectsettings['pdf_download_link_template']) && in_array($tmplId, $form->subredirectsettings['pdf_download_link_template'])) ? $tmplId : 0;
                break;
			default :
                $this->tmplId = (!empty($form->frontendsettings['singleRecordPdfTemplate']) && in_array($tmplId, $form->frontendsettings['singleRecordPdfTemplate'])) ? $tmplId : 0;
		}
		if (!VisformsHelper::checkPdfTmplIdValid($this->tmplId)) {
			$this->tmplId = 0;
		}
	}

	protected function setFid() {
		$this->fid = $this->input->get('id', 0, 'int');
	}

	protected function setInputFid() {
		$this->input->set('fid', $this->fid);
	}

	protected function renderPdfDocument() {
		$fid = $this->fid;
		$datas = $this->pdfData;
		if (!empty($datas)) {
			$dataFieldsModel    = $this->getModel('Visdataspdf', 'Administrator', array('id' => $this->fid));
			$dataFields = $dataFieldsModel->getDataFields();
			if (!empty($this->tmplId)) {
                $task = $this->getTask();
				$app = Factory::getApplication();
                $app->setUserState('visforms'. $fid . '.pdf.tmplid', $this->tmplId);
				$app->setUserState('visforms'. $fid . '.pdf.datas', $datas);
				$app->setUserState('visforms'. $fid . '.pdf.datafields', $dataFields);
                $app->setUserState('visforms'. $fid . '.pdf.pdftask', $task);
				$urlRenderPdf = Route::_("index.php?option=com_visforms&view=visformrenderpdf&format=raw&fid=$this->fid", false);
				$this->setRedirect($urlRenderPdf);
				return true;
			}
			else {
				$this->setMessage(Text::_('COM_VISFORMS_NO_PDF_TEMPLATE_AVAILABLE'), 'error');
			}
		}
		else {
			$this->setMessage(Text::_('COM_VISFORMS_EXPORT_PDF_NO_DATA'), 'error');
		}

		$this->setRedirect($this->parentViewReturnUrl);
		return false;
	}

	// special implementation for Task renderPdfFromRequestData
	// this task is triggered from result message (after form submit), not from a data view
    // create a data object which can be used to replace placeholder in pdf document
    protected function createDataObjectFromRequestData($visform) {
        // no information like modified_by, modified_at available
        // we only have the data submitted with the form (from the form view)
		$pdfRawData = new \stdClass();
		if (isset($visform->dataRecordId)) {
            $pdfRawData->id = $visform->dataRecordId;
       }
		$pdfRawData->created_by = Factory::getApplication()->getIdentity()->id;
		// todo null or now or pass now with request?
		$pdfRawData->created = null;
		foreach ($visform->fields as $field) {
			$fieldName = 'F' . $field->id;
			if (!empty($field->isDisabled)) {
				$pdfRawData->$fieldName = '';
			}
			else {
				$pdfRawData->$fieldName = $field->dbValue;
			}
		}
		return $pdfRawData;
	}

	protected function removeForbiddenData() {
		if (empty($this->pdfData)) {
		    return;
        }
        $datas = $this->pdfData;
        $user = Factory::getApplication()->getIdentity();
		// check for permission
		foreach ($datas as $i => $data) {
            if (($user->authorise('core.create.pdf', 'com_visforms.visform.' . (int) $this->fid . '.vispdf.' . (int) $this->tmplId)) ||
                (($user->authorise('core.create.own.pdf', 'com_visforms.visform.' . (int)$this->fid . '.vispdf.' . (int) $this->tmplId)) && ($data->created_by === $user->id))) {
                continue;
            }
			// remove recordset, which the user cannot export to pdf
			unset($datas[$i]);
		}
		$this->pdfData = array_values($datas);
	}

	protected function canCreatePdf(): bool {
	    // return true, if a user can create pdf of their own data or can create pdf of all data
        $user = Factory::getApplication()->getIdentity();
        return (($user->authorise('core.create.pdf', 'com_visforms.visform.' . (int) $this->fid . '.vispdf.' . (int) $this->tmplId)) ||
            (($user->authorise('core.create.own.pdf', 'com_visforms.visform.' . (int) $this->fid . '.vispdf.' . (int) $this->tmplId))));
    }

    // process (clean) parameter extracted from post
    // use data view plugin 'model' Vfdataviewmodel in order to process the plugin params extracted from post
    // with the same functions used to process the params extracted from plugin string
    protected function cleanPlgParamsFromRequest($pparams) {
	    if (empty($pparams || !is_array($pparams))) {
	        return array();
        }
        $plgModel = new Vfdataviewmodel($pparams, '', 0, $this->factory);
        return $plgModel->getCleanedParams();
    }
}