<?php
/**
 * visfield model for Visforms
 *
 * @author       Aicha Vack
 * @package      Joomla.Administrator
 * @subpackage   com_visforms
 * @link         https://www.vi-solutions.de
 * @license      GNU General Public License version 2 or later; see license.txt
 * @copyright    2012 vi-solutions
 * @since        Joomla 1.6 
 */

namespace Visolutions\Component\Visforms\Administrator\Model;

defined('_JEXEC') or die( 'Restricted access' );

use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Database\ParameterType;
use Joomla\Utilities\ArrayHelper;
use Joomla\String\StringHelper;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisfieldSaveJFormExtraDataEvent;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisfieldAfterBatchCopyFormEvent;
use Visolutions\Component\Visforms\Administrator\Event\Visforms\VisfieldPrepareJFormEvent;
use Visolutions\Component\Visforms\Administrator\Helper\AefHelper;
use Visolutions\Component\Visforms\Administrator\Helper\FormImportHelper;
use Visolutions\Component\Visforms\Administrator\Helper\VisformsHelper;
use Visolutions\Component\Visforms\Administrator\Helper\ConditionsHelper;
use Visolutions\Component\Visforms\Administrator\Helper\SqlHelper;
use Visolutions\Component\Visforms\Site\Lib\Validation\DigitValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\LatitudeValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\LongitudeValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\MaxNumberValidation;
use Visolutions\Component\Visforms\Site\Lib\Validation\MinNumberValidation;


class VisfieldModel extends ItemModelBase
{
	protected $hasSub;
	protected $allowedCalculationPlaceholderFieldTypes = array('number', 'select', 'calculation', 'hidden', 'checkbox', 'text', 'date', 'radio', 'selectsql', 'radiosql');
	public $typeAlias = "com_visforms.visfield";
	// Default value can be set with a sql statement
	protected $inputsWithSqlDefault = array('text', 'email', 'hidden', 'number', 'url', 'date', 'tel');
	// Field value can be set dynamically with a sql statement
	protected $inputsWithSqlValue = array('text', 'email', 'hidden', 'number', 'url', 'date', 'tel');

    public function __construct($config = array(), ?MVCFactoryInterface $factory = null, ?FormFactoryInterface $formFactory = null) {
        $config['events_map'] = array(
            'delete' => 'visforms',
            'save' => 'visforms',
            'change_state' => 'visforms'
            );
        $config['event_before_save'] = 'onVisformsBeforeJFormSave';
        $config['event_after_save'] = 'onVisformsAfterJFormSave';
        $config['event_before_delete'] = 'onVisformsBeforeJFormDelete';
        $config['event_after_delete'] = 'onVisformsAfterJFormDelete';
        $config['event_change_state'] = 'onVisformsJFormChangeState';
	    $this->hasSub = AefHelper::checkAEF();
        parent::__construct($config, $factory, $formFactory);
    }
	
	public function batch($commands, $pks, $contexts) {
		// sanitize field ids
		$pks = array_unique($pks);
		ArrayHelper::toInteger($pks);

		// remove any values of zero
		if (array_search(0, $pks, true)) {
			unset($pks[array_search(0, $pks, true)]);
		}

		if (empty($pks)) {
			$this->setError(Text::_('JGLOBAL_NO_ITEM_SELECTED'));
			return false;
		}

		$result = $this->batchCopyField($commands, $pks, $contexts);
		if ( !is_array($result)) {
			return false;
		}

		// clear the cache
		$this->cleanCache();

		return true;
	}

	protected function batchCopyField($commands, $oldFields, $contexts) {
		$isFormCopy = ArrayHelper::getValue($commands,'isFormCopy', false);
		$newFid = ArrayHelper::getValue($commands,'form_id', 0);
		$propertiesToSet = array(
		    'frontaccess' => ArrayHelper::getValue($commands,'assetgroup_id', 0, 'int'),
            'editonlyfield' => ArrayHelper::getValue($commands, 'editonlyfield', '', 'string'),
            'frontdisplay' => ArrayHelper::getValue($commands,'frontenddisplay', '', 'string'),
            'required' => ArrayHelper::getValue($commands, 'required', '', 'string'),
            'readonly' => ArrayHelper::getValue($commands, 'readonly', '', 'string'),
            'showlabel' => ArrayHelper::getValue($commands, 'showlabel', '', 'string'),
        );
		// array is used to mend restricts and restrictions in copied fields
		$copyFormOldNewFieldsIdMap = array();
		$isFieldInFormCopy = false;
		$table = $this->getTable();
        $extension = Factory::getApplication()->getInput()->get('option', '');
        $newIds = array();
		
		if (empty($newFid)) {
		    // only the case, if batch copy is triggerd from batch copy in fields view
		    if ($isFormCopy) {
                // batch copy field is triggered from batch copy form
                // if no newFid exists, something has gone wrong
                $this->setError((Text::_('COM_VISFORMS_ERROR_BATCH_NO_FORM_SELECTED')));
                return false;
            }
		    else {
		        // native field batch process without field copy
                // we only want ot set field options

                while (!empty($oldFields)) {
                    // pop the first ID off the stack
                    $pk = array_shift($oldFields);
                    // check that the user has edit permission for this field
                    $table->reset();
                    if (!$table->load($pk)) {
                        if ($error = $table->getError()) {
                            // fatal error
                            $this->setError($error);
                            return false;
                        }
                        else {
                            // not fatal error
                            $this->setError(Text::sprintf('JLIB_APPLICATION_ERROR_BATCH_MOVE_ROW_NOT_FOUND', $pk));
                            continue;
                        }
                    }
                    if ((!$this->user->authorise('core.edit', $extension . '.visform.' . $table->fid . 'visfield.'. $pk))
                        && (!$this->user->authorise('core.edit.own', $extension . '.visform.' . $table->fid . 'visfield.'. $pk))
                    ) {
                        // check if user has created the field and has edit.own permissions on the field. Else: no permission
                        if ($this->user->authorise('core.edit.own', $extension . '.visform.' . $table->fid . 'visfield.'. $pk) && ($table->created_by !== $this->user->id)) {
                            $this->setError(Text::sprintf('COM_VISFORMS_FIELD_ERROR_BATCH_CANNOT_EDIT', $pk));
                            return false;
                        }
                    }
                    foreach ($propertiesToSet as $property => $value) {
                        $table->setValue($property, $value);
                    }

                    // check the row
                    if (!$table->check()) {
                        $this->setError($table->getError());
                        return false;
                    }

                    // store the row
                    if (!$table->store()) {
                        $this->setError($table->getError());
                        return false;
                    }
                }
            }
		}
		// batch copy is triggered from batch copy form
        // fields are new
        // setting field options is not supported in these batch copy dialogs
		else {
            $i = 0;

            // check that the user has create permission for this form
            if (!$this->user->authorise('core.create', $extension . '.visform.' . $newFid)) {
                $this->setError(Text::_('COM_VISFORMS_FIELD_ERROR_BATCH_CANNOT_CREATE'));
                return false;
            }

            // parent exists, so we proceed
            while (!empty($oldFields)) {
                // pop the first ID off the stack
                $pk = array_shift($oldFields);
                $table->reset();

                // check that the row actually exists
                if (!$table->load($pk)) {
                    if ($error = $table->getError()) {
                        // fatal error
                        $this->setError($error);
                        return false;
                    }
                    else {
                        // not fatal error
                        $this->setError(Text::sprintf('JLIB_APPLICATION_ERROR_BATCH_MOVE_ROW_NOT_FOUND', $pk));
                        continue;
                    }
                }

                if ($newFid == $table->fid) {
                    // we copy a field into the same form so alter the label & alias
                    $data = $this->generateNewTitle('', $table->name, $table->label);
                    $table->label = $data['0'];
                    $table->name = $data['1'];
                    $isFieldInFormCopy = true;
                    //Remove values in database field restrictions
                    $table->restrictions = "";
                }
                else {
                    $formIdMapper = array($table->fid => $newFid);
                    //we either copy an form with it's field or we copy a single field into another form
                    // alter form id
                    $table->fid = $newFid;
                    if ($isFormCopy !== true) {
                        //we copy a field into another form, where the restrictor fields do not exist
                        // reset values in options that reference other fields like _validate_equalTo
                        $table->defaultvalue = ConditionsHelper::removeRestrictsValues($table->defaultvalue, $table->name, true);
                    }
                    // remove values in database field restrictions, we set them anew if necessary
                    $table->restrictions = "";
                }

                // reset the ID, created and created_by because we are making a copy
                $table->id = 0;
                $table->created_by = 0;
                $table->created = 0;
                $unpublish = ArrayHelper::getValue($commands, 'unpublish', true);
                // set to unpublished
                $table->published = ($unpublish) ? 0 : $table->published;
                // delete ordering to get the next ordering number
                $table->ordering = '';
                // check for additional batch options and set the values
                foreach ($propertiesToSet as $property => $value) {
                    $table->setValue($property, $value);
                }

                // check the row
                if (!$table->check()) {
                    $this->setError($table->getError());
                    return false;
                }

                // store the row
                if (!$table->store()) {
                    $this->setError($table->getError());
                    return false;
                }

                // are data saved for the table the copied fields belong to?
                // then we have to create a data table field
                $db = $this->getDatabase();
                $query = $db->createQuery();
                // select Ids of Fields of copied form in Table visfields
                $query
                    ->select($db->qn('saveresult'))
                    ->from($db->qn('#__visforms'))
                    ->where($db->qn('id') . ' = ' . $table->fid);
                try {
                    $db->setQuery($query);
                    $saveResult = $db->loadResult();
                    $this->createDataTableFields($table->fid, $table->id, $saveResult);
                }
                catch (\RuntimeException $e) {}

                // get the new item ID
                $newId = $table->get('id');
                // we copy a complete form and must adapt conditional fields to new form
                if ($isFormCopy === true) {
                    // create an item in the field map array
                    $copyFormOldNewFieldsIdMap[$pk] = $newId;
                }
                if ($isFieldInFormCopy === true) {
                    $oldRestricts = ConditionsHelper::setRestrictsFromDb($newId, $table->fid);
                    if (!empty($oldRestricts)) {
                        // only have to add new restrictions to existing fields
                        ConditionsHelper::setRestriction($oldRestricts);
                    }
                }
                // Batch Copy Extra Data
                PluginHelper::importPlugin('visforms');
                $afterBatchCopyFormEvent = new VisfieldAfterBatchCopyFormEvent('onVisfieldAfterBatchCopyForm', [
                    'pk' => $pk,
                    'newId' => $newId
                ]);
                $this->getDispatcher()->dispatch('onVisfieldAfterBatchCopyForm', $afterBatchCopyFormEvent);
                // add the new ID to the array
                $newIds[$i] = $newId;
                $i++;
            }
        }

		if (!empty($copyFormOldNewFieldsIdMap)) {
		    ConditionsHelper::setConditionsInCopiedFields($copyFormOldNewFieldsIdMap, $newFid);
			// fix references to data table of the old form and data table fields of the old form in radiosql, selectsql, multicheckboxsql field if we batch copy form
			if ($isFormCopy === true) {
				$helper = new FormImportHelper($copyFormOldNewFieldsIdMap, $formIdMapper);
				$helper->adaptFieldsSqlStatementToNewForm();
			}
		}

		// clean the cache
		$this->cleanCache();

		return $newIds;
	}

	public function importFieldData($oldFields, $newFid) {
		$copyFormOldNewFieldsIdMap = array();
		$table = $this->getTable();
		$i = 0;

		// check that the user has create permission for this form
		$extension = Factory::getApplication()->getInput()->get('option', '');
		if (!$this->user->authorise('core.create', $extension . '.visform.' . $newFid)) {
			$this->setError(Text::_('COM_VISFORMS_FIELD_ERROR_BATCH_CANNOT_CREATE'));
			return false;
		}

		// parent exists so we let's proceed
		while (!empty($oldFields)) {
			$pk = array_shift($oldFields);
			$oldFieldId = $pk['id'];
			$pk['id'] = 0;
			$pk['fid'] = $newFid;
			$table->reset();
			$table->bind($pk);
			// check the row
			if (!$table->check()) {
				$this->setError($table->getError());
				return false;
			}

			// store the row
			if (!$table->store()) {
				$this->setError($table->getError());
				return false;
			}
			// get the new item ID
			$newId = $table->get('id');
			$copyFormOldNewFieldsIdMap[$oldFieldId] = $newId;
		}
		if (!empty($copyFormOldNewFieldsIdMap)) {
			ConditionsHelper::setConditionsInCopiedFields($copyFormOldNewFieldsIdMap, $newFid);
		}

		// clean the cache
		$this->cleanCache();
		return $copyFormOldNewFieldsIdMap;
	}

	// Removing field from form definition does not affect the save process
    // model->save uses the table object, which load every database field
    // Ensure valid values for all configuration options in the database
    // Invalid stored values must be actively amended
	public function save($data) {
    	// ToDo if selected field type comes from subscription feature which is not available, set an error message and return false
		// Pagebreak, Cal, Map, radiosql, selectsql, multicheckboxsql, signature
        $app = Factory::getApplication();
		$task = $app->getInput()->get('task');
		$numberpattern = '/^\-?\d+\.?\d*$/';
		if ($task != 'save2copy') {
			if (empty($this->hasSub)) {
				$data['editonlyfield'] = 0;
				$data['customlabelforsummarypage'] = '';
				$data['customlabelformail'] = '';
				$data['customlabelforcsv']  = '';
				$data['fileexportformat']   = 0;
				$data['useassearchfieldonly'] = 0;
                $data['excludefieldfromsearch'] = 0;
                $data['isfilterfield'] = 0;
                $data['ismultiselectsearchfilter'] = 0;
                $data['multiselectsearchfiltercondition'] = 0;
				$data['allowferadiussearch'] = 0;
				$data['distanceunit'] = 'km';
				$data['fileattachmentname'] = '';
				$data['is_searchable'] = 0;
                $data['link_text_type'] = 0;
                $data['link_text'] = '';
                $data['url_link_text'] = '';
                $data['link_pre_text'] = '';
                $data['link_post_text'] = '';
			}
			if (empty($this->hasSub) || $data['typefield'] !== 'location' || empty($data['frontdisplay'])) {
				$data['displayAsMapInList']   = 0;
				$data['displayAsMapInDetail'] = 0;
				$data['listMapHeight'] = 100;
				$data['listMapZoom'] = 8;
				$data['detailMapHeight'] = 400;
				$data['detailMapZoom'] = 13;
			}
			if (!empty($data['excludefieldfromsearch'])) {
                $data['useassearchfieldonly'] = 0;
                $data['isfilterfield'] = 0;
            }
			if (empty($data['isfilterfield'])) {
                $data['ismultiselectsearchfilter'] = 0;
            }
            if (empty($data['ismultiselectsearchfilter'])) {
                $data['multiselectsearchfiltercondition'] = 0;
            }
			if ($data['typefield'] === 'location') {
				if($data['frontdisplay'] == "2") {
					$data['displayAsMapInDetail'] = 0;
				}
				if($data['frontdisplay'] == "3") {
					$data['displayAsMapInList'] = 0;
				}
				if (empty($data['displayAsMapInDetail'])) {
					$data['listMapHeight'] = 100;
					$data['listMapZoom'] = 8;
				}
				if (empty($data['displayAsMapInList'])) {
					$data['detailMapHeight'] = 400;
					$data['detailMapZoom'] = 13;
				}
			}
            // if file link text is not empty, we trigger a replacement of placeholders on the text
            // make sure, that file link text is only set, if file link text type is 2, in order to prevent unnecessary replace attempts
            if ($data['typefield'] === 'file') {
                if (!empty($data['link_text_type']) && ((int) $data['link_text_type'] !== 2)) {
                    $data['link_text'] = '';
                }
            }
		}
		if (isset($data['defaultvalue']) && is_array($data['defaultvalue'])) {
            // sql value statements in fields of type text, email, hidden....
            if (in_array($data['typefield'], $this->inputsWithSqlValue)) {
                if (empty($this->hasSub)) {
                    $data['defaultvalue']['f_'.$data['typefield'].'_toggle_value_from_sql'] = "";
                    $data['defaultvalue']['f_'.$data['typefield'].'_reload'] = "";
                    $data['defaultvalue']['f_'.$data['typefield'].'_sql'] = "";
                    $data['defaultvalue']['f_'.$data['typefield'].'_sql_process_in_edit_view'] = "1";
                }
            }
			// check for bad sql statements in radiosql, selectsql or multicheckboxsql
			if ((in_array($data['typefield'], array('selectsql', 'radiosql', 'multicheckboxsql'))) || (in_array($data['typefield'], $this->inputsWithSqlValue))) {
				$sql = $data['defaultvalue']['f_'.$data['typefield'].'_sql'];
				$sqlHelper = new SqlHelper($sql);
				if (!$sqlHelper->checkIsSelectSqlOnly()) {
					$app->enqueueMessage(Text::_("COM_VISFORMS_CREATE_SQL_BAD_SQL"), 'ERROR');
					return false;
				}
			}

			// sql defaultvalue statements in fields of type text, email, hidden....
            if (in_array($data['typefield'], $this->inputsWithSqlDefault)) {
                // clean data
                if (empty($this->hasSub)) {
                    if ($data['defaultvalue']['f_'.$data['typefield'].'_fillwith'] == 'sql') {
                        $data['defaultvalue']['f_'.$data['typefield'].'_fillwith'] = "0";
                    }
                }
                if ($data['defaultvalue']['f_'.$data['typefield'].'_fillwith'] !== 'sql') {
                    $data['defaultvalue']['f_'.$data['typefield'].'_defaultvalue_sql'] = "";
                }
                $sql = $data['defaultvalue']['f_'.$data['typefield'].'_defaultvalue_sql'];
                $sqlHelper = new SqlHelper($sql);
                if (!$sqlHelper->checkIsSelectSqlOnly()) {
                    $app->enqueueMessage(Text::_("COM_VISFORMS_CREATE_SQL_BAD_SQL"), 'ERROR');
                    return false;
                }
            }

            if ($task != 'save2copy') {
	            if (empty($this->hasSub)) {
		            unset($data['defaultvalue']['f_radio_show_label']);
		            unset($data['defaultvalue']['f_select_show_label']);
		            unset($data['defaultvalue']['f_multicheckbox_show_label']);
                    unset($data['defaultvalue']['f_radiosql_show_label']);
                    unset($data['defaultvalue']['f_selectsql_show_label']);
                    unset($data['defaultvalue']['f_multicheckboxsql_show_label']);
		            unset($data['defaultvalue']['f_file_show_label']);
		            unset($data['defaultvalue']['f_location_show_label']);
		            unset($data['defaultvalue']['f_calculation_show_label']);
		            unset($data['defaultvalue']['f_signature_show_label']);
                    unset($data['defaultvalue']['f_date_mindate']);
                    unset($data['defaultvalue']['f_date_maxdate']);
                    unset($data['defaultvalue']['f_date_daydate_shift']);
                    unset($data['defaultvalue']['f_date_dynamic_min_shift']);
                    unset($data['defaultvalue']['f_date_dynamic_max_shift']);
                    unset($data['defaultvalue']['f_date_minage']);
                    unset($data['defaultvalue']['f_email_validate_mailExists']);
                    unset($data['defaultvalue']['f_select_customselectvaluetext']);
	            }
                // check that fields of type calculation use only fields of "number"-types as placeholder
				$validCalculation = $this->checkCalculationString($data);
				if ($validCalculation === false) {
                    return false;
				}
				foreach ($this->allowedCalculationPlaceholderFieldTypes as $uncheckedValue) {
					if (empty($this->hasSub)) {
						unset($data['defaultvalue']["f_{$uncheckedValue}_unchecked_value"]);
					}
					else {
						if (isset($data['defaultvalue']["f_{$uncheckedValue}_unchecked_value"])) {
							$ucValue = trim(str_replace(",", ".", $data['defaultvalue']["f_{$uncheckedValue}_unchecked_value"]));
							if (!(preg_match($numberpattern, $ucValue) == true)) {
								$app->enqueueMessage(Text::sprintf('COM_VISFORMS_INVALID_UNCHECKED_VALUE'), 'error');
								return false;
							}
							$data['defaultvalue']["f_{$uncheckedValue}_unchecked_value"] = $ucValue;
						}
					}
					unset($uncheckedValue);
				}
                if (!empty($this->hasSub)) {
                    // only allow to set aef field editonlyfield to 1 if field is not used as restrictor otherwise reset it to default
                    if (!$this->canSaveEditOnlyField($data)) {
                        $data['editonlyfield'] = 0;
                    }
                    // if aef editonlyfield is set to 1
                    if (!empty($data['editonlyfield'])) {
                        // remove restricts from submitted data
                        // restrictions in restrictor fields will be removed automatically by the code below and no new restrictions will be set if restricts are empty
                        $data['defaultvalue'] = ConditionsHelper::removeRestrictsValues($data['defaultvalue'], $data['name'], false, false, array('_reload'));
                    }
                }
                // only allow to use either _showWhen or _reload
                // selctsql field
                if ($data['typefield'] == 'selectsql') {
                	// f_selectsql_render_as_datalist
	                if (!empty($data['defaultvalue']['f_selectsql_render_as_datalist'])) {
						unset($data['defaultvalue']['f_selectsql_attribute_required']);
                        unset($data['defaultvalue']['f_selectsql_attribute_readonly']);
		                unset($data['defaultvalue']['f_selectsql_attribute_multiselect']);
                        unset($data['defaultvalue']['f_selectsql_is_searchable']);
		                $data['defaultvalue']['f_selectsql_attribute_custominfo'] = "";
		                $data['defaultvalue']['f_selectsql_attribute_customerror'] = "";
		                $data['defaultvalue']['f_selectsql_attribute_size'] = "";
		                $data['defaultvalue']['f_selectsql_preSelectSolitaryOption'] = "";
	                }
	                if (!empty($data['defaultvalue']['f_selectsql_toggle_reload'])) {
		                // remove restricts from submitted data
		                // restrictions in restrictor fields will be removed automatically by the code below and no new restrictions will be set if restricts are empty
		                $data['defaultvalue']['f_selectsql_showWhen'] = "";
		                if (empty($data['defaultvalue']['f_selectsql_preSelectSolitaryOption'])) {
			                $data['defaultvalue']['f_selectsql_hideOnPreSelectedSolitaryOption']  = "";
		                }
	                }
	                else {
		                $data['defaultvalue']['f_selectsql_reload'] = "";
		                $data['defaultvalue']['f_selectsql_hideOnEmptyOptionList']  = "";
		                $data['defaultvalue']['f_selectsql_preSelectSolitaryOption']  = "";
		                $data['defaultvalue']['f_selectsql_hideOnPreSelectedSolitaryOption']  = "";
                        $data['defaultvalue']['f_selectsql_sql_process_in_edit_view']  = "1";
	                }
                }
                // text inputs with sql statement for value
                if (in_array($data['typefield'], $this->inputsWithSqlValue)) {
                    if (!empty($data['defaultvalue']['f_'.$data['typefield'].'_toggle_value_from_sql'])) {
                        // remove restricts from submitted data
                        // restrictions in restrictor fields will be removed automatically by the code below and no new restrictions will be set if restricts are empty
                        $data['defaultvalue']['f_'.$data['typefield'].'_showWhen'] = "";
                    }
                    else {
                        $data['defaultvalue']['f_'.$data['typefield'].'_toggle_value_from_sql'] = "";
                        $data['defaultvalue']['f_'.$data['typefield'].'_reload'] = "";
                        $data['defaultvalue']['f_'.$data['typefield'].'_sql_process_in_edit_view'] = "1";
                    }
                }
                if (!empty($this->hasSub) && $data['typefield'] === 'location') {
					$zoom = (int) $data['defaultvalue']['f_location_zoom'];
					if (!$this->visformsValidate('min', array('count' => $zoom, 'mincount' => 1))
					|| !$this->visformsValidate('max', array('count' => $zoom, 'maxcount' => 20))
					|| !$this->visformsValidate('digits', array('value' => $zoom))) {
						$data['defaultvalue']['f_location_zoom'] = 13;
					} else {
						$data['defaultvalue']['f_location_zoom'] = $zoom;
					}
					$latCenter = $data['defaultvalue']['f_location_defaultMapCenter_lat'];
	                $lngCenter = $data['defaultvalue']['f_location_defaultMapCenter_lng'];
	                $latDefault = $data['defaultvalue']['f_location_attribute_value_lat'];
	                $lngDefault = $data['defaultvalue']['f_location_attribute_value_lng'];
	                $validLatDefault = $this->visformsValidate('latitude', array('value' => $latDefault)) || (empty($latDefault) && empty($lngDefault));
	                $validLngDefault = $this->visformsValidate('longitude', array('value' => $lngDefault)) || (empty($latDefault) && empty($lngDefault));
	                if ($latCenter === "" || $lngCenter === "" || !$this->visformsValidate('latitude', array('value' => $latCenter)) || !$this->visformsValidate('longitude',  array('value' => $lngCenter))) {
		                $app->enqueueMessage(Text::sprintf('COM_VISFORMS_LOCATION_DEFAULT_CENTER_VALUES_REQUIRED', ''), 'error');
		                return false;
	                }
	                if (!$validLatDefault || !$validLngDefault ) {
		                $app->enqueueMessage(Text::sprintf('COM_VISFORMS_LOCATION_DEFAULT_POSITION_VALUES_INVALID_FORMAT', ''), 'error');
		                return false;
	                }
                }

                if ($data['typefield'] === "date" && $data['defaultvalue']['f_date_attribute_value'] === $this->getDatabase()->getNullDate()) {
	                $data['defaultvalue']['f_date_attribute_value'] = "";
	            }
                if ($data['typefield'] === "tel" && empty($data['defaultvalue']['f_tel_phonevalidation'])) {
                    unset($data['defaultvalue']['f_tel_phonevalidation_include']);
                }
                
                // if we deal with a select, radio or multicheckbox and one of its options,
                // that is used as a restriction in another field, is going to be removed, we have to do something
                // we will remove restricts in conditional field (defaultvalue) and give a message
	            if (!empty($data['id'])) {
		            if (in_array($data['typefield'], array('select', 'radio', 'multicheckbox'))) {
			            // get list of restritions and respective restricted field id's from database
			            $oldRestrictions = ConditionsHelper::getRestrictions($data['id']);
			            if (isset($oldRestrictions['usedAsShowWhen']) && (count($oldRestrictions['usedAsShowWhen']) > 0)) {
				            $deletedOptionsIds = ConditionsHelper::getRemovedOptionIds($data);
				            if (!empty($deletedOptionsIds)) {
					            // loop through restrictions
					            foreach ($oldRestrictions['usedAsShowWhen'] as $oRKey => $oRId) {
						            ConditionsHelper::removeDeletedOptionsDependencies($oRKey, $oRId, $deletedOptionsIds, $data);
					            }
				            }
			            }
		            }
		            // remove old restrictions
		            ConditionsHelper::removeRestriction(ConditionsHelper::setRestrictsFromDb($data['id'], $data['fid']));

		            // fix unauthorized sql statements
		            if (!$this->user->authorise('core.create.sql.statement', 'com_visforms') || empty($this->hasSub)) {
			            // if a user has no ACL to create sql statements, we just want to keep the old value!
                        if ((in_array($data['typefield'], array('selectsql', 'radiosql', 'multicheckboxsql'))) || (in_array($data['typefield'], $this->inputsWithSqlValue))) {
				            $oldDefaultvalues = VisformsHelper::registryArrayFromString(ConditionsHelper::getDefaultValueFromDb($data['id']));
				            $data['defaultvalue']['f_'.$data['typefield'].'_sql'] = (isset($oldDefaultvalues['f_'.$data['typefield'].'_sql'])) ? $oldDefaultvalues['f_'.$data['typefield'].'_sql'] : "";
			            }
		            }
	            }

				// store a copy of defaultValues array in a variable, needed to save new restrictions after recordset is saved
                $restrictorDefaultValues = $data['defaultvalue'];
            }

            $data['defaultvalue'] = VisformsHelper::registryStringFromArray($data['defaultvalue']);
		}
		// ToDo and uikit3?
		if (isset($data['gridSizes']) && is_array($data['gridSizes'])) {
			if ($task != 'save2copy') {
				// default field and label widths for some field types
				if (in_array($data['typefield'], array('submit', 'reset', 'image', 'pagebreak', 'hidden'))) {
					$data['gridSizes']['fieldsPerRow'] = 0;
					$data['gridSizes']['fieldsPerRowSm'] = 0;
					$data['gridSizes']['fieldsPerRowMd'] = 0;
					$data['gridSizes']['fieldsPerRowLg'] = 0;
					$data['gridSizes']['fieldsPerRowXl'] = 0;
                    $data['gridSizes']['fieldsPerRowXxl'] = 0;
					$data['gridSizes']['labelBootstrapWidth'] = 12;
					$data['gridSizes']['labelBootstrapWidthSm'] = 12;
					$data['gridSizes']['labelBootstrapWidthMd'] = 12;
					$data['gridSizes']['labelBootstrapWidthLg'] = 12;
					$data['gridSizes']['labelBootstrapWidthXl'] = 12;
                    $data['gridSizes']['labelBootstrapWidthXxl'] = 12;
				}
				$data['gridSizes'] = VisformsHelper::registryStringFromArray($data['gridSizes']);
			}
		}
		// alter the title for save as copy
        // remove restrictions for save as copy
        // Reset created and created_by for save as copy
		if ($task == 'save2copy') {
            list($label, $name) = $this->generateNewTitle('', $data['name'], $data['label']);
			$data['label']	= $label;
			$data['name']	= $name;
            $data['restrictions'] = "";
            $data['created']	= 0;
            $data['created_by']	= 0;
		}

		if (!empty($data['id']) ) {
			// get restrictions and stored field name, if a field is used in calculation
			$restrictions = ConditionsHelper::getRestrictions($data['id']);
			if (!empty($restrictions) && ! empty($restrictions['usedInCal'])) {
				$oldFieldName = ConditionsHelper::getOldFieldNameFromDb($data['id']);
			}
		}

		if (parent::save($data)) {
			$isNew = $this->getState($this->getName() . '.new');
			$newId = $this->getState($this->getName() . '.id');
			if (!$isNew) {
                if (!empty($data['fid']) && !empty($data['id']) && !empty($data['fordering'])) {
                    $this->reOrderFields($data);
                }
                if (!empty($data['fid']) && !empty($data['id']) && !empty($data['fdataordering'])) {
                    $this->reOrderFields($data, 'data');
                }
            }
			$restrictorId = (!empty($newId)) ? $newId : $data['id'];
			if ((!empty($isNew)) && (!empty($restrictorId)) && ($app->getInput()->get('task') == 'save2copy')) {
                $oldRestricts = ConditionsHelper::setRestrictsFromDb($restrictorId, $data['fid']);
				// only have to add new restrictions to existing fields
				if (!empty($oldRestricts)) {
					ConditionsHelper::setRestriction($oldRestricts);
				}
			}
			else {
				// save restrictions
				ConditionsHelper::setRestriction(ConditionsHelper::setRestricts($restrictorId, $restrictorDefaultValues, $data['name'], $data['fid']));
				// fix modified field name of fields used in calculation
				if (!empty($oldFieldName) && $oldFieldName != $data['name']) {
					foreach ($restrictions['usedInCal'] as $calFieldId) {
						ConditionsHelper::fixModifiedFieldNameInCalculation($oldFieldName, $data['name'], $calFieldId);
					}
				}
			}
            // trigger saveExtraData Event
            PluginHelper::importPlugin('visforms');
            $saveJFormExtraDataEvent = new VisfieldSaveJFormExtraDataEvent('onVisfieldSaveJFormExtraData', [
                'subject' => $data,
                'isNew' => $isNew,
                'id' => $newId,
            ]);
            $this->getDispatcher()->dispatch('onVisfieldSaveJFormExtraData', $saveJFormExtraDataEvent);
			return true;
		}
		else {
			// error case, save was not successful
			if (($app->getInput()->get('task') != 'save2copy') && !empty($data['id'])) {
				// restrictions were deleted previously, the must be reset
				ConditionsHelper::setRestriction(ConditionsHelper::setRestrictsFromDb($data['id'], $data['fid']));
			}
			return false;
		}
	}
	
	// Create a field in datatable
	public function createDataTableFields($fid, $id, $saveresult) {
        if (!$this->createDataTableField($fid, $id, $saveresult)) {
            // throw error
        }
        if (!$this->createDataTableField($fid, $id, $saveresult, true)) {
             // throw error
        }
	    return true;
	}

	public function getForm($data = array(), $loadData = true) {
		// get the form
		$form = $this->loadForm('com_visforms.visfield', 'visfield', array('control' => 'jform', 'load_data' => $loadData));
		if (empty($form)) {
			return false;
		}
		$app = Factory::getApplication();
		$fid = $app->getInput()->getInt('fid', isset($data['fid']) ? $data['fid'] : 0);
		$id = $app->getInput()->getInt('id', 0);
		if ($fid != 0) {
			// visforms form configuration dependent settings
			$model = new VisformModel(array('ignore_request' => true));
			$visform = $model->getItem($fid);
			// form layout specific adaptations
            // fix invalid layout selection in stored forms
            if (in_array($visform->layoutsettings['formlayout'], array('btdefault', 'bt3default', 'bt4mcindividual'))) {
                $visform->layoutsettings['formlayout'] = 'bt5';
            }
            if ($visform->layoutsettings['formlayout'] === 'uikit2') {
                $visform->layoutsettings['formlayout'] = 'uikit3';
            }
			// unsupported feature in default layout
			if (empty($this->hasSub) || (!empty($visform) && (!in_array($visform->layoutsettings['formlayout'], array('bt5', 'uikit3'))))) {
				$form->removeField('f_radio_show_label', 'defaultvalue');
				$form->removeField('f_select_show_label', 'defaultvalue');
				$form->removeField('f_multicheckbox_show_label', 'defaultvalue');
                $form->removeField('f_radiosql_show_label', 'defaultvalue');
                $form->removeField('f_selectsql_show_label', 'defaultvalue');
                $form->removeField('f_multicheckboxsql_show_label', 'defaultvalue');
				$form->removeField('f_file_show_label', 'defaultvalue');
				$form->removeField('f_calculation_show_label', 'defaultvalue');
				$form->removeField('f_location_show_label', 'defaultvalue');
				$form->removeField('f_signature_show_label', 'defaultvalue');
				$form->removeGroup('gridSizes');
			}
			// unsupported feature in default layout
            // unsupported feature if sub-layout is not individual
			if (empty($this->hasSub) || ((!empty($visform) && (!isset($visform->layoutsettings['displaysublayout']) || ($visform->layoutsettings['displaysublayout'] != 'individual'))))) {
				$form->removeGroup('gridSizes');
			}
			// only 5 breakpoints in uikit 3 layout
            if (!empty($visform)  && in_array($visform->layoutsettings['formlayout'], array('uikit3')) && isset($visform->layoutsettings['displaysublayout']) && $visform->layoutsettings['displaysublayout'] == 'individual') {
                $form->removeField('fieldsPerRowXxl', 'gridSizes');
                $form->removeField('labelBootstrapWidthXxl', 'gridSizes');
            }
		}
		// visforms form configuration independent settings
		// modify the form based on edit state access controls
		if ($id != 0 && (!$this->user->authorise('core.edit.state', 'com_visforms.visform.'. $fid . '.visfield.'.(int) $id))
		    || ($id == 0 && !$this->user->authorise('core.edit.state', 'com_visforms.visform.'. $fid))) {
			// disable fields for display
			$form->setFieldAttribute('published', 'disabled', 'true');
		}
        $form->setFieldAttribute('ordering', 'disabled', 'true');
        // remove aef options
        if (empty($this->hasSub)) {
            foreach ($this->inputsWithSqlValue as $fieldName) {
                $form->removeField('f_'.$fieldName.'_reload', 'defaultvalue');
                $form->removeField('f_'.$fieldName.'_toggle_value_from_sql', 'defaultvalue');
                $form->removeField('f_'.$fieldName.'_sql', 'defaultvalue');
                $form->removeField('f_'.$fieldName.'_sql_process_in_edit_view', 'defaultvalue');
                $form->removeField('f_'.$fieldName.'_spacerhrvaluesql', 'defaultvalue');
            }
	        foreach ($this->allowedCalculationPlaceholderFieldTypes as $uncheckedValue) {
		        $form->removeField('f_'.$uncheckedValue.'_unchecked_value', 'defaultvalue');
	        }
            $form->setFieldAttribute('f_date_minvalidation_type', 'label', Text::_('COM_VISFORMS_MIN_DATE_VALIDATION_TYPE'). Text::_('COM_VISFORMS_SUBSCRIPTION_ONLY'), 'defaultvalue');
            $form->setFieldAttribute('f_date_maxvalidation_type', 'label', Text::_('COM_VISFORMS_MAX_DATE_VALIDATION_TYPE'). Text::_('COM_VISFORMS_SUBSCRIPTION_ONLY'), 'defaultvalue');
            $form->setFieldAttribute('f_checkbox_checkedDynamic', 'label', Text::_('COM_VISFORMS_SELECT_DYNAMIC_DATA'). Text::_('COM_VISFORMS_SUBSCRIPTION_ONLY'), 'defaultvalue');
            $fieldSets = $form->getFieldsets();
            foreach ($fieldSets  as $name => $fieldSet) {
                foreach ($form->getFieldset($name) as $field) {
                    $dataAttributes = $field->getDataAttributes();
                    if (!empty($dataAttributes) && is_array($dataAttributes) && isset($dataAttributes['data-visibility']) && ((string) $dataAttributes['data-visibility'] === 'subscription')) {
                        $form->removeField($field->fieldname, $field->group);
                    }
                }
            }
		}

		if (!$this->user->authorise('core.create.sql.statement', 'com_visforms')) {
			$form->setFieldAttribute('f_selectsql_sql', 'disabled', 'true', 'defaultvalue');
			$form->setFieldAttribute('f_radiosql_sql', 'disabled', 'true', 'defaultvalue');
			$form->setFieldAttribute('f_multicheckboxsql_sql', 'disabled', 'true', 'defaultvalue');
            $form->setFieldAttribute('f_text_defaultvalue_sql', 'disabled', 'true', 'defaultvalue');
            $form->setFieldAttribute('f_email_defaultvalue_sql', 'disabled', 'true', 'defaultvalue');
            $form->setFieldAttribute('f_hidden_defaultvalue_sql', 'disabled', 'true', 'defaultvalue');
            $form->setFieldAttribute('f_number_defaultvalue_sql', 'disabled', 'true', 'defaultvalue');
            $form->setFieldAttribute('f_url_defaultvalue_sql', 'disabled', 'true', 'defaultvalue');
            $form->setFieldAttribute('f_tel_defaultvalue_sql', 'disabled', 'true', 'defaultvalue');
		}
        // use to modify form (i.e. add plugin specific form fields)
        PluginHelper::importPlugin('visforms');
        $prepareJFormEvent = new VisfieldPrepareJFormEvent('onVisfieldPrepareJForm', [
            'subject' => $form,
        ]);
        $this->getDispatcher()->dispatch('onVisfieldPrepareJForm', $prepareJFormEvent);
		return $form;
	}

    protected function loadFormData() {
		// check the session for previously entered form data
		$data = Factory::getApplication()->getUserState('com_visforms.edit.visfield.data', array());
		if (empty($data)) {
            $data = $this->getItem();
            if (!empty($data->id) && !empty($this->hasSub)) {
                $data->fordering = $data->id;
            }
            if (!empty($data->id) && !empty($this->hasSub)) {
                $data->fdataordering = $data->id;
            }
		}
		return $data;
	}

    protected function loadFormFieldsParameters() {
        $item = $this->item;
        $item->defaultvalue = VisformsHelper::registryArrayFromString($item->defaultvalue);
        $item->restrictions = VisformsHelper::registryArrayFromString($item->restrictions);
		$item->gridSizes = VisformsHelper::registryArrayFromString($item->gridSizes);
    }

    protected function getReorderConditions($table) {
	    return 'fid = '.Factory::getApplication()->getInput()->get('fid', 0);
	}
	
	protected function canEditState($record) {
		// check for existing field
		if (!empty($record->id)  && !empty($record->fid)) {
            return $this->user->authorise('core.edit.state', 'com_visforms.visform.' . (int) $record->fid . '.visfield.' .(int) $record->id);
		}
        else {
            // default to component settings
            return parent::canEditState($record);
		}
	}
	
    protected function canDelete($record) {
		if (!empty($record->id)  && !empty($record->fid)) {
			$canDelete = ConditionsHelper::canDelete($record->id, $record->name);
			if (empty($canDelete)) {
				return false;
			}
			return $this->user->authorise('core.delete', 'com_visforms.visform.' . (int) $record->fid . '.visfield.' .(int) $record->id);
		}
		else {
			// use component settings
			return parent::canDelete($record);
		}
	}
    
    protected function canSaveEditOnlyField($data) {
        if (!empty($data['id'])) {
            if (empty($this->hasSub)) {
                return true;            
            }
            if (empty($data['editonlyfield'])) {
                return true;            
            }
	        //db field restrictions is not part of the data array, therefore we pass id and name
	        $canSaveEditOnlyField = ConditionsHelper::canSaveEditOnlyField($data['id'], $data['name']);
	        if (empty($canSaveEditOnlyField)) {
		        return false;
	        }
		}
        return true;
    }
	
	protected function generateNewTitle( $catid, $name, $label) {
		// alter the label & name
		$table = $this->getTable();
		while ($table->load(array('name' => $name))) {
			$label = StringHelper::increment($label);
			$name = StringHelper::increment($name, 'dash');
		}
		return array($label, $name);
	}

	/**
	 * Method to assure creation of data table fileds for already loaded field model
	 * used by controller::postSaveHook() and others
	 * @params no
	 * @return void
	 * @since Joomla 1.6
	 */
	public function assureCreateDataTableFields() {
		$item = $this->getItem();
		$id = $item->id;
		$fid = $item->fid;
		if ($fid && $id) {
			$db = $this->getDatabase();
			$query = $db->createQuery();
			$query->select('*')
				->from($db->qn('#__visforms'))
				->where($db->qn('id') . ' = ' . $fid);
			// Add field to data tables
            try {
                $db->setQuery($query);
                $forms = $db->loadObjectList();
                if (is_array($forms) && count($forms) > 0) {
                    $this->createDataTableFields($fid, $id, $forms[0]->saveresult);
                }
            }
            catch (\RuntimeException $e) {}
		}
	}

    /**
     * Method to create a new field in the data table (used for storing submitted user inputs)
     * @param int $fid form id
     * @param int $id field id
     * @param boolean $save set to true if field is to be created in the data save table
     * @return boolean
     */
    public function createDataTableField($fid, $id, $saveresult, $save = false) {
	    $db	= $this->getDatabase();
    	$tn = "#__visforms_".$fid;
	    $tnFull = strtolower($db->getPrefix(). 'visforms_'.$fid);
        if ($save === true) {
            $tn .= "_save";
	        $tnFull .= "_save";
        }

	    $tablesAllowed = $db->getTableList();
	    if (!empty($tablesAllowed)) {
		    $tablesAllowed = array_map('strtolower', $tablesAllowed);
	    }

	    if (!in_array($tnFull, $tablesAllowed) && !$saveresult) {
		    return true;
	    }
		$tableFields = $db->getTableColumns($tn,false);
		$fieldName = "F" . $id;
		if (!isset( $tableFields[$fieldName] )) {
            $query = "ALTER TABLE ".$db->qn($tn)." ADD ".$db->qn($fieldName)." TEXT NULL";
            try {
                $db->setQuery($query);
                $db->execute();
            }
            catch (\RuntimeException $e) {
                Factory::getApplication()->enqueueMessage( Text::_( 'COM_VISFORMS_PROBLEM_WITH' )." (".$query.")", 'warning');
                return false;
            }
		    return true;
		}
	    return true;
	}

    public function publish(&$pks, $value = 1) {
		$pks = (array) $pks;
		// look for restrictions
		foreach ($pks as $i => $pk) {
            $restrictions = ConditionsHelper::getRestrictions($pk);
            if ((is_array($restrictions)) && (count($restrictions) > 0) && isset($this->state->task) && $this->state->task == 'unpublish') {
                // give an error message
                Factory::getApplication()->enqueueMessage(Text::sprintf("COM_VISFORMS_FIELD_HAS_RESTICTIONS", $pk), 'warning');
                // unset the pk
                unset($pks[$i]); 
            }
        }
        return parent::publish($pks, $value);
    }

	protected function checkCalculationString($data) {
		if ($data['typefield'] != 'calculation') {
			return true;
		}
		$equation = $data['defaultvalue']['f_calculation_equation'];
		if (empty($equation)) {
            Factory::getApplication()->enqueueMessage(Text::_('COM_VISFORMS_EQUATION_REQUIRED'), 'error');
			return false;
		}
		$pattern = '/\[[A-Z0-9]{1}[A-Z0-9\-]*]/';
		if (preg_match_all($pattern, $equation, $matches)) {
			$fid = $data['fid'];
			$allowedFieldTypes = implode('","', $this->allowedCalculationPlaceholderFieldTypes);
			$db = $this->getDatabase();
			$query = $db->createQuery();
			$query->select($db->qn('name'))
				->from($db->qn('#__visfields'))
				->where($db->qn('typefield') . ' in ("' . $allowedFieldTypes .'")')
				->where($db->qn('fid') . " = " . $fid)
				->where($db->qn('published') . " = " . 1);
			try {
				$db->setQuery($query);
                $fields = $db->loadColumn();
			}
			catch (\RuntimeException $e) { }
            if (empty($fields)) {
                // if we have no fields here then we do not have a select sql field either,
                // so we can return here
                return true;
			}
			// found matches are store in the $matches[0] array
			foreach ($matches[0] as $match) {
                $str = trim($match, '\[]');
				$fieldName = StringHelper::strtolower($str);
				if (!(in_array($fieldName, $fields))) {
                    Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_INVALID_PLACEHOLDER_TYPE_IN_CALCULATION', $match), 'error');
					return false;
				}
			}
			// Check Select SQL is not _as_data_list
            $db = $this->getDatabase();
            $query = $db->createQuery();
            $query->select(array($db->qn('name'), $db->qn('defaultvalue')))
                ->from($db->qn('#__visfields'))
                ->where($db->qn('typefield') . ' = ' . $db->q('selectsql'))
                ->where($db->qn('fid') . " = " . $fid)
                ->where($db->qn('published') . " = " . 1);
            try {
                $db->setQuery($query);
                $fields = $db->loadObjectList();
            }
            catch (\RuntimeException $e) { }
            if (!empty($fields)) {
                // found matches are store in the $matches[0] array
                foreach ($matches[0] as $match) {
                    $str = trim($match, '\[]');
                    $fieldName = StringHelper::strtolower($str);
                    foreach ($fields as $field) {
                        if ($field->name != $fieldName) {
                            continue;
                        }
                        if (empty($field->defaultvalue)) {
                            continue;
                        }
                        $defaultvalues = VisformsHelper::registryArrayFromString($field->defaultvalue);
                        // $restrictions {} is not empty but results in emypt array
                        if (!empty($defaultvalues) && !empty($defaultvalues['f_selectsql_render_as_datalist'])) {
                            Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_SELECT_SQL_IS_DATALIST_CALCULATION', $fieldName), 'error');
                            return false;
                        }
                    }
                        if (!(in_array($fieldName, $fields))) {

                        }
                }
            }

			// Prevent use of an select sql field which uses this calcualtion field as reload trigger
            $query = $db->createQuery();
            $query->select($db->qn('restrictions'))
                ->from($db->qn('#__visfields'))
                ->where($db->qn('id') .  ' = ' . $data['id']);
            try {
                $db->setQuery($query);
                $restrictions = $db->loadResult();
            }
            catch (\RuntimeException $e) { }
            if (empty($restrictions)) {
                return true;
            }
            $restrictions = VisformsHelper::registryArrayFromString($restrictions);
            // $restrictions {} is not empty but results in empty array
            if (empty($restrictions)) {
                return true;
            }
            foreach ($matches[0] as $match) {
                $str = trim($match, '\[]');
                $fieldName = StringHelper::strtolower($str);
                if (isset($restrictions['usedAsReloadTrigger'])) {
                    if (is_array($restrictions['usedAsReloadTrigger'])) {
                        foreach ($restrictions['usedAsReloadTrigger'] as $name => $value) {
                            if ($name == $fieldName) {
                                Factory::getApplication()->enqueueMessage(Text::sprintf('COM_VISFORMS_CALCULATION_FIELD_USED_AS_RELOAD_TRIGGER', $fieldName), 'error');
                                return false;
                            }
                        }
                    }
                }
            }
		}
		return true;
	}

	protected function visformsValidate($type, $arg) {
        switch ($type) {
            case 'min':
                $validation =  new MinNumberValidation($arg);
                break;
            case 'max':
                $validation =  new MaxNumberValidation($arg);
                break;
            case 'digits':
                $validation =  new DigitValidation($arg);
                break;
            case 'latitude':
                $validation =  new LatitudeValidation($arg);
                break;
            case 'longitude':
                $validation =  new LongitudeValidation($arg);
                break;
            default :
                return true;
        }
        return $validation->validate();
	}

	public function saveorderDataDetail($pks, $order) {
    	// initBatch sets this->table
		$this->initBatch();
		$this->table->setColumnAlias('ordering', 'dataordering');
		$return = parent::saveorder($pks, $order);
		$this->table->setColumnAlias('ordering', 'ordering');
		return $return;
	}

	// change field sort order/ field sort order in data view from field configuration
    // get a list of all field id /field order pairs
    // $type '' = field, 'data' = data
	protected function getReorderList($fid, $start, $end, $type = '') {
        $fieldName = $type . 'ordering';
        $db = $this->getDatabase();
        $query = $db->createQuery();
        $query->select(
            [
                $db->quoteName('id'),
                $db->quoteName($fieldName),
            ]
        )
            ->from($db->quoteName('#__visfields'))
            ->where($db->quoteName($fieldName) . ' >= ' . $start)
            ->where($db->quoteName($fieldName) . ' <= ' . $end)
            ->where($db->quoteName('fid') . ' = :fid')
            ->bind(':fid', $fid, ParameterType::INTEGER);

        $query->order($db->quoteName('ordering') . ' ASC');
        try {
            $db->setQuery($query);
            return $db->loadAssocList();
        }
        catch (\RuntimeException $e) {
            return false;
        }
    }

    protected function getMaxOrdering($fid, $type = '') {
        $fieldName = $type . 'ordering';
        $db = $this->getDatabase();
        $query = $db->createQuery();
        $query->select('max('.$fieldName.')')
            ->from($db->quoteName('#__visfields'))
            ->where($db->quoteName('fid') . ' = :fid')
            ->bind(':fid', $fid, ParameterType::INTEGER);
        try {
            $db->setQuery($query);
            return $db->loadResult();
        }
        catch (\RuntimeException $e) {
            return false;
        }
    }

    protected function reOrderFields($data, $type = '') {
        // xml form contains order fields which have no matching database fields (fordering, fdataordering)
        // option values are the field id of the field behind which we want to reorder
        // additional values (-1 = first, -2 = last)
        // database has fields to store the ordering (ordering, dataordering)
        // values in these database fields are not touched, when field configuration is saved
        // if position of field is change in the field configuration correct the values in the db order fields 'manually'
        $xmlOrderFieldName = 'f' . $type . 'ordering';
        $dbOrderFieldName = $type . 'ordering';
        // check if we have to fix sort order
        $data[$xmlOrderFieldName] = (int) $data[$xmlOrderFieldName];
        // order was changed
        if ($data['id'] !== $data[$xmlOrderFieldName]) {
            $table = $this->getTable();
            // get current ordering position of the field
            $table->load($data['id']);
            $curPos = (int) $table->$dbOrderFieldName;
            // get new ordering position of the field
            $table->load($data[$xmlOrderFieldName]);
            $newPos = (int) $table->$dbOrderFieldName;
            // move to last position
            // get the actual highest ordering value
            if ($data[$xmlOrderFieldName] === -2) {
                $newPos = $this->getMaxOrdering($data['fid'], $type);
            }
            // Reordering has to be done for every field, between the old and the new position
            // minimum value
            // newPos is the field after which we insert
            // we do not need to change the field with that ordering (newPos + 1) if we go up

            $start = ($curPos < $newPos) ? $curPos : $newPos + 1;
            // maximum value
            $end = ($curPos < $newPos) ? $newPos : $curPos;
            // direction
            $reOrderDir = ($curPos > $newPos) ? 'up' : 'down';
            // get list of all effected fields (id and order value)
            $order = $this->getReorderList($data['fid'], $start, $end, $type);
            foreach ($order as $field) {
                $table->load((int) $field['id']);
                // field which was reordered in the first place
                // order value = new position
                if ($table->id === (int) $data['id']) {
                    if ($reOrderDir === 'up') {
                        $table->$dbOrderFieldName = $newPos + 1;
                    }
                    else {
                        $table->$dbOrderFieldName = $newPos;
                    }
                }
                else {
                    // every other filed 1 up or 1 down
                    if ($reOrderDir === 'up') {
                        $table->$dbOrderFieldName = ++$table->$dbOrderFieldName;
                    }
                    else {
                        $table->$dbOrderFieldName = --$table->$dbOrderFieldName;
                    }
                }
                $table->store();
            }
        }
    }
}