<?php
/**
 * @copyright	Copyright (C) 2010 vi-solutions. All rights reserved.
 * @license		GNU General Public License version 2 or later; see LICENSE.txt
 * @author		MysteryFCM, modified by Robert Kuster and Aicha Vack
 * @link         https://www.vi-solutions.de
 */

namespace Visolutions\Plugin\Visforms\Spambotcheck\Helper;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\Database\DatabaseInterface;

class SpambotCheckHelper {

    public static function cleanEMailWhitelist($list): string {
        if ($list != '') {
            // delete blanks
            $list = str_replace(' ', '', $list);
            // delete ',' at string end
            while ($list[strlen($list) - 1] == ',') {
                $list = substr($list, 0, strlen($list) - 1);
            }
        }
        return $list;
    }

    public static function cleanEMailBlacklist($list): string {
        if ($list != '') {
            // delete blanks
            $list = str_replace(' ', '', $list);
            // delete ',' at string end
            while ($list[strlen($list) - 1] == ',') {
                $list = substr($list, 0, strlen($list) - 1);
            }
        }
        return $list;
    }

    public static function cleanUsername($name): string {
        if ($name != '') {
            $name = addslashes(htmlentities($name));
            $name = urlencode($name);
            $name = str_replace(" ", "%20", $name); // no spaces
        }
        return $name;
    }

    public static function isCUrlAvailable(): bool {
        $extension = 'curl';
        if (extension_loaded($extension)) {
            return true;
        }
        return false;
    }

    public static function isURLOnline($URL): bool {
        // check, if curl is available
        if (self::isCUrlAvailable()) {
            // check if url is online
            $curl = @curl_init($URL);
            curl_setopt($curl, CURLOPT_TIMEOUT, 10);
            curl_setopt($curl, CURLOPT_FAILONERROR, 1);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
            @curl_exec($curl);
            if (curl_errno($curl) != 0) {
                curl_close($curl);
                return false;
            }
            curl_close($curl);
            return true;
        }

        // curl is not loaded, this won't work
        return false;
    }

    public static function getURL($URL): string {
        if (self::isURLOnline($URL) == false) {
            $returnURL = 'Unable to connect to server';
            return $returnURL;
        }
        if (function_exists('file_get_contents') && ini_get('allow_url_fopen') == true) {
            // use file_get_contents
            $returnURL = @file_get_contents($URL);
        }
        else {
            // use cURL (if available)
            if (self::isCUrlAvailable()) {
                $curl = @curl_init();
                curl_setopt($curl, CURLOPT_URL, $URL);
                curl_setopt($curl, CURLOPT_VERBOSE, 1);
                curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($curl, CURLOPT_HEADER, 0);
                $returnURL = @curl_exec($curl);
                curl_close($curl);
            }
            else {
                return 'Unable to connect to server';
            }
        }

        return $returnURL;
    }

    public static function isValidIP($IP): string {
        if ($IP != '') {
            $regex = "'\b(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b'";
            return preg_match($regex, $IP) ? $IP : '';
        }

        return '';
    }

    public static function isValidEmail($email): string {
        if ($email != '') {
            $regex = '/^([a-zA-Z0-9_\.\-\+%])+\@(([a-zA-Z0-9\-])+\.)+([a-zA-Z0-9]{2,4})+$/';
            return preg_match($regex, $email) ? $email : '';
        }

        return '';
    }

    // Usage example:
    // ---------------
    // logSpammerToDB('test@test.com', '12.12.12.12', 'username', 'ProjectHoneyPot', '127.41.11.5', 'ThreatScore=11, DaysSinceLastActivity=41', $plgParams)
    //
	public static function logSpammerToDB($email, $IP, $username, $engine, $request, $rawReturn, $parsedReturn, &$params): bool {
        // check plugin settings: save spambots to DB
        if (!$params->get('spbot_log_to_db', 0)) {
            return false;
        }

        // change empty vars to "NULL"
        if ($email == '') {
            $email = 'NULL';
        }
        if ($IP == '') {
            $IP = 'NULL';
        }
        if ($username == '') {
            $username = 'NULL';
        }
        // trim anything that could screw up SQL
        $email = str_replace(array("0x", ",", "%", "'", "\r\n", "\r", "\n"), "", $email);
        $IP = str_replace(array("0x", ",", "%", "'", "\r\n", "\r", "\n"), "", $IP);
        // add DB record
        $db      = Factory::getContainer()->get(DatabaseInterface::class);
        $date = gmdate("Y-m-d H:i:s", time());
        $action  = $params->get('current_action', '-');
        $query   = $db->createQuery();
        $columns = array('email', 'ip', 'engine', 'request', 'raw_return', 'parsed_return', 'attempt_date');
        $values  = array($db->quote($email), $db->quote($IP), $db->quote($engine), $db->quote($request), $db->quote($rawReturn), $db->quote($parsedReturn), $db->quote($date));
        /** @noinspection SqlResolve */
        $query
            ->insert($db->quoteName('#__visforms_spambot_attempts'))
            ->columns($db->quoteName($columns))
            ->values(implode(',', $values));
        try {
            $db->setQuery($query);
            $db->execute();
        }
        catch (\Exception $e) {
            return false;
        }
        return true;
    }
}